// CubeCom
// 

#define COM_FRAME_MAXLENGTH 128
#define COM_POLL_TIMEOUT 1000
#define COM_ERR_TIMEOUT_READ 1

#ifdef COM_ONCUBE
#include <cubeos.h>
#include <ttyio.h>
#include <signal.h>
#include <kerror.h>
#include <malloc.h>
#else
#include <sys/poll.h>
#ifndef sun
#include <error.h>
#endif
#endif

#include <stdio.h>


#ifdef COM_ONCUBE
void COM_Timer();
#endif

#include "cubecom.h"

int COM_Timer_Flag;
int COM_Timer_Thread;
int COM_receive_status;
int COM_transmit_status;
#ifndef COM_ONCUBE
int COM_fd;
#endif


COM_rxstat_T COM_rxstat;
COM_txstat_T COM_txstat;

COM_receiver_T COM_receivers[256];

int COM_errorcode;

int COM_process_echo(COM_framehead_T * frame)
{
  if (frame) 
    {
      frame->dsap=frame->ssap;
      frame->ssap=0;
      if (frame->data) {
	COM_send_frame(frame);
	free (frame->data);
      } else {
	frame->data=NULL;
	frame->length=0;
	COM_send_frame(frame);
      }
      free(frame);
    }
}

int COM_process_console(COM_framehead_T * frame)
{
  if (frame) 
    {
      frame->dsap=frame->ssap;
      frame->ssap=0;
      if (frame->data) {
	printf("CONSOLE: %s\n",frame->data);
	free (frame->data);
      } else {
	frame->data=NULL;
	frame->length=0;
	printf("CONSOLE: no data\n");
      }
      free(frame);
    }
}

int COM_register(unsigned char dsap,void * function)
{

  if (function){
    COM_receivers[dsap].function = function;
    COM_receivers[dsap].counter = 0;
    return 0;
  } else {
    return -1;
      }

}

int COM_unregister(unsigned char dsap)
{
    COM_receivers[dsap].function = NULL;
    COM_receivers[dsap].counter = 0;
    return 0;
}



int COM_init(int fd)
{
#ifdef COM_ONCUBE
  COM_Timer_Thread = KERN_create(COM_Timer);
#endif
 COM_rxstat.goodframe=0;
 COM_rxstat.badheader=0;
 COM_rxstat.baddata=0;
 COM_rxstat.overflow=0;
 COM_rxstat.timeout=0;
 COM_rxstat.framecount=0;
 COM_txstat.framecount=0;

 COM_register(0,(void *) COM_process_echo);
#ifndef COM_ONCUBE
 COM_fd=fd;
 COM_register(1,(void *) COM_process_console);
#endif

return (0);
}

int COM_deinit()
{
#ifdef COM_ONCUBE
  kill(COM_Timer_Thread,SIGKILL);
  KERN_wakeup(COM_Timer_Thread); // this makes it disappear
#endif
  COM_unregister(0);
  return(0);
}

int COM_printstats()
{
  printf("COM reception statistics:\n");
  printf("good frames: %d\n",COM_rxstat.goodframe);
  printf("bad headers: %d\n",COM_rxstat.badheader);
  printf("bad data: %d\n",COM_rxstat.baddata);
  printf("buffer overflow: %d\n",COM_rxstat.overflow);
  printf("timeout: %d\n",COM_rxstat.timeout);
  printf("total frames: %d\n",COM_rxstat.framecount);
  printf("COM reception statistics:\n");
  printf("total frames: %d\n",COM_txstat.framecount);
  return(0);
}


#ifdef COM_ONCUBE
void COM_Timer()
{

  while(1) {
    KERN_suspend(-1);
    KERN_psleep(COM_TIMEOUT);
    COM_Timer_Flag=1;
  }
}
#endif

void COM_outchar(char c)
{
  int i;
#ifdef COM_ONCUBE
      TTY_outchar(c);
#else
      i=write(COM_fd,&c,1);
      if (i!=1) {
	perror("CUBECOM: couldn't write");
      }      
      //      printf("TX: >%c< %hhx \t",c,c);
      for(i=1;i<100000;i++); // this is the magic delay for the Cube SCI
                            // tested for old 1 Ghz PIII: 25000
                            // for new 1Ghz PIII: 50000
#endif
}

char COM_inchar()
{
#ifndef COM_ONCUBE
struct pollfd nfds[1];
#endif
  char buffer[1];
  char c;
  int i;
#ifdef COM_ONCUBE
      c=  TTY_inchar();
      //      printf("RX: %c %x\n",c,c);
      return(c);
#else

      nfds[0].fd=COM_fd;
      nfds[0].events=POLLIN;
      if (poll(nfds,1,COM_POLL_TIMEOUT)) {
	i=read(COM_fd,buffer,1);
	if (i!=1) {
	  perror("CUBECOM: couldn't read");
	  return(0);
	}
      } else {
	COM_errorcode|=COM_ERR_TIMEOUT_READ;
	COM_rxstat.timeout++;
	return(0);
      }
      //      printf("RX: %c %x\t",buffer[0],buffer[0]);
      return buffer[0];
#endif
}




char COM_receive_byte(int raw)
{

  char c;

  COM_receive_status=0;
  if (raw)
    {
      // anything goes through unfiltered
      return (COM_inchar());
    } else {
      // this removes byte stuffing
      c = COM_inchar();
      switch (c) {
      case COM_ESC:
	// handle escaped characters
	// COM_outchar('E');
	c = COM_inchar();
	c ^= COM_SCRAMBLE; 
	break;
      case 0x03:
	// this is the ctrl-c character, it should reset
	// the cube anyway. Maybe just to let the user know...
	printf("CUBE RESET CHAR RECEIVED\n");fflush(stdout);
	COM_receive_status = COM_STATE_RST;
	break;
      case COM_SOF:
	COM_receive_status = COM_STATE_SOF;
	break;
      case COM_EOF:
	COM_receive_status = COM_STATE_EOF;
	break;
      default:
	break;
      }
      return c;
    }
}

int COM_send_byte(char c,int raw)
{
  COM_transmit_status=0;
  //  printf("TX: >%c< %hhx \n",c,c);

  if (raw)
    {
      COM_outchar(c);
      // anything goes through unfiltered
    } else {
      // this does byte stuffing
      switch (c) {
      case COM_ESC:
      case 0x03:
      case COM_SOF:
      case COM_EOF:
	// handle escaped characters
	COM_outchar(COM_ESC);
	COM_outchar(c ^ COM_SCRAMBLE); 
	//	printf(" E: \t");
	break;
      default:
	//	printf(" U: %d ",c);
	COM_outchar(c);
	break;
    }
    }

}

int COM_checkhead(COM_framehead_T * theframe)
{
 unsigned  short checksum;

 checksum = theframe->length+theframe->ssap+theframe->dsap+theframe->fcs;

 if (checksum != theframe->hcheck) return -1;

 if (theframe->length > COM_FRAME_MAXLENGTH) return -1;

 return 0;
}

void COM_fixhead(COM_framehead_T * theframe)
{

 theframe->hcheck = theframe->length+theframe->ssap+theframe->dsap+theframe->fcs;

}


unsigned short COM_fcs(char * data, unsigned short length)
{
  unsigned short sum=0;
  int i;
  for (i=0;i<length;i++) sum+=data[i];
  return (sum);
}


int COM_send_frame(COM_framehead_T * theframe)
{
  int i;

  theframe->fcs = COM_fcs(theframe->data, theframe->length);
  COM_fixhead(theframe);

  COM_send_byte(COM_SOF,1);

  COM_send_byte((unsigned char)(theframe->length >>8),0);
  COM_send_byte((unsigned char)(theframe->length & 0xff),0);

  COM_send_byte((unsigned char)(theframe->flags >>8),0);
  COM_send_byte((unsigned char)(theframe->flags & 0xff),0);

  COM_send_byte(theframe->ssap,0);
  COM_send_byte(theframe->dsap,0);

  COM_send_byte((unsigned char)(theframe->fcs >>8),0);
  COM_send_byte((unsigned char)(theframe->fcs & 0xff),0);

  COM_send_byte((unsigned char)(theframe->hcheck >>8),0);
  COM_send_byte((unsigned char)(theframe->hcheck & 0xff),0);

  for (i=0; i<theframe->length;i++) COM_send_byte(theframe->data[i],0);

  COM_txstat.framecount++;

  COM_send_byte(COM_EOF,1);
  COM_send_byte(COM_EOF,1);
  COM_send_byte(COM_EOF,1);
  COM_send_byte(COM_EOF,1);
  COM_send_byte(COM_EOF,1);


}



int COM_send(char * data, unsigned short length, unsigned char ssap, unsigned char dsap, unsigned char flags)
{

  COM_framehead_T * theframe;

  theframe = (COM_framehead_T *) malloc(sizeof(COM_framehead_T));

  if (!theframe) {
#ifdef COM_ONCUBE
    KERN_complain(ERR_EMERG,"COM running out of memory");
#else
    fprintf(stderr,"COM running out of memory\n");
#endif
    return(-1);
  }

  theframe->length = length;
  theframe->ssap = ssap;
  theframe->dsap = dsap;
  theframe->flags = flags;
  theframe->data = data;

  return(COM_send_frame(theframe));
} 


void COM_printframe(COM_framehead_T * theframe)
{
  printf("\nFRAME:");
  printf("Length: %d SSAP: %d DSAP:%d FLAGS: %hx hck: %hx fcs: %hx \n",
	 theframe->length,
	 theframe->ssap,
	 theframe->dsap,
	 theframe->flags,
	 theframe->hcheck,
	 theframe->fcs
 ); 

}
  





COM_framehead_T * COM_receive_frame()
{
  int i;
  COM_framehead_T * theframe;
  unsigned char l1;
  unsigned char l2;

  theframe = malloc(sizeof(COM_framehead_T));

  if (!theframe) {
#ifdef COM_ONCUBE
    KERN_complain(ERR_EMERG,"COM running out of memory");

#else
    fprintf(stderr,"COM running out of memory\n");
#endif
    return(NULL);
  }
  COM_errorcode=0;
  while ((COM_receive_byte(1)!=COM_SOF) && !(COM_errorcode & COM_ERR_TIMEOUT_READ) );// COM_outchar('-'); // read until SOF
 

  if(COM_errorcode & COM_ERR_TIMEOUT_READ)
    {
      free(theframe);
      return(NULL);
    }

  COM_rxstat.framecount++;

  //  COM_outchar('A');
  l1=COM_receive_byte(0);

  if(COM_errorcode & COM_ERR_TIMEOUT_READ)
    {
      free(theframe);
      return(NULL);
    }

  l2=COM_receive_byte(0);

  theframe->length = (l1<<8)+l2;

  l1=COM_receive_byte(0);

  if(COM_errorcode & COM_ERR_TIMEOUT_READ)
    {
      free(theframe);
      return(NULL);
    }

  l2=COM_receive_byte(0);

  if(COM_errorcode & COM_ERR_TIMEOUT_READ)
    {
      free(theframe);
      return(NULL);
    }


  theframe->flags = (l1<<8)+l2;

  theframe->ssap = COM_receive_byte(0);

  if(COM_errorcode & COM_ERR_TIMEOUT_READ)
    {
      free(theframe);
      return(NULL);
    }

  theframe->dsap = COM_receive_byte(0);

  if(COM_errorcode & COM_ERR_TIMEOUT_READ)
    {
      free(theframe);
      return(NULL);
    }

  l1=COM_receive_byte(0);

  if(COM_errorcode & COM_ERR_TIMEOUT_READ)
    {
      free(theframe);
      return(NULL);
    }

  l2=COM_receive_byte(0);

  if(COM_errorcode & COM_ERR_TIMEOUT_READ)
    {
      free(theframe);
      return(NULL);
    }


  theframe->fcs = (l1<<8)+l2;

  l1=COM_receive_byte(0);

  if(COM_errorcode & COM_ERR_TIMEOUT_READ)
    {
      free(theframe);
      return(NULL);
    }

  l2=COM_receive_byte(0);

  if(COM_errorcode & COM_ERR_TIMEOUT_READ)
    {
      free(theframe);
      return(NULL);
    }


  theframe->hcheck = (l1<<8)+l2;

  if (COM_checkhead(theframe))
    {
      COM_rxstat.badheader++;
      free(theframe);
      //      COM_outchar('X');
      return(NULL);
    }
  //  COM_outchar('B');
  theframe->data = malloc(theframe->length);
  if (!theframe->data) {
    // no malloc
    COM_rxstat.overflow++;
    for (i=0; i<theframe->length;i++) COM_receive_byte(0);
    free(theframe);
    return(NULL);
  } 


  //  printf("length = %d\n",theframe->length);

  //  COM_outchar('C');
  for (i=0; i<theframe->length;i++){
    theframe->data[i]=COM_receive_byte(0);

  if(COM_errorcode & COM_ERR_TIMEOUT_READ)
    {
      free(theframe->data);
      free(theframe);
      return(NULL);
    }

  }
  //  COM_outchar('D');
  if (COM_fcs(theframe->data,theframe->length)!=theframe->fcs)
    {
      COM_rxstat.baddata++;
      free(theframe->data);
      free(theframe);
      //      COM_outchar('Y');
      return(NULL);
    }
  //  COM_outchar('E');
  COM_rxstat.goodframe++;
  return (theframe);
}



int COM_send_console(char * string)
{
  COM_send(string,strlen(string)+1,1,1,0);
}



COM_process_one()
{
  COM_framehead_T * frame;

  frame = COM_receive_frame();
  if (frame) 
    {
      if (COM_receivers[frame->dsap].function)
	{
	  COM_receivers[frame->dsap].counter++;
	  COM_receivers[frame->dsap].function(frame);
	} else {
	  COM_send_console("dsap unknown");
	  if (frame->data)
	    free(frame->data);
	  free(frame);
	}
    } else
      {
	  COM_send_console("bad frame");
      }
}
