/*
 * Unix RoboCube Interface 
 * H. Kenn V0.1
 * based on PB911 laser scanner IF by S. Carpin
 */


#include <stdio.h>   /* Standard input/output definitions */
#include <string.h>  /* String function definitions */
#include <unistd.h>  /* UNIX standard function definitions */
#include <fcntl.h>   /* File control definitions */
#include <errno.h>   /* Error number definitions */
#include <termios.h> /* POSIX terminal control definitions */

#include "unixif.h"


/******************** 
  LOW LEVEL ROUTINES
  Not to be used by client code.
  All subject to static linkage to prevent pollution of names and clashes
********************/


/*
 * Opens the serial port.
 * dev: device to open
 * Returns the file handle on success or -1 on error.
 */
static int _CUBE_open_port(const char *dev)
{

  int fd; 

  fd = open(dev, O_RDWR | O_NOCTTY | O_NDELAY);
  if (fd != -1) 
    //    fcntl(fd, F_SETFL, FNDELAY);  //non blocking read */
    fcntl(fd, F_SETFL, 0);   /* blocking read */
  
  return (fd);
} 

/* 
 * Configure the given port according to the cube parameters
 * int fd: handle of the device to initialize
 * Returns the return code of tcsetattr, thus success it at least one
 * of the requests has been honored
 * Call print_conf to print out the status of the settings and see if
 * this succeeded
 */
static int _CUBE_configure_port(int fd) {
  struct termios options;
  tcgetattr(fd, &options);
  /* cube works at 57.6 */
  cfsetispeed(&options, B57600);
  cfsetospeed(&options, B57600);

 
  /* No flow control */
  options.c_cflag &= ~CRTSCTS;  


  /* No parity, 1 stop bit, 8 data bits */
  options.c_cflag &= ~PARENB;
  options.c_cflag &= ~CSTOPB;  
  options.c_cflag &= ~CSIZE;
  options.c_cflag |= CS8;

  /* select raw input */
  options.c_lflag &= ~(ICANON | ECHO | ECHOE | ISIG | IEXTEN);
  options.c_iflag &= ~(BRKINT | ICRNL | INPCK | ISTRIP |IXON );
  options.c_oflag &= ~(OPOST);


  /* Allow reading and does not became owner */
  options.c_cflag |= (CLOCAL | CREAD);

  cfmakeraw(&options);

  return tcsetattr(fd, TCSANOW, &options);
}

/* 
 * Configure the given port according to the cube parameters
 * int fd: handle of the device to initialize
 * Returns the return code of tcsetattr, thus success it at least one
 * of the requests has been honored
 * Call print_conf to print out the status of the settings and see if
 * this succeeded
 */
int _CUBE_configure_port_slow(int fd) {
  struct termios options;
  tcgetattr(fd, &options);
  /* slow down to 19200 */
  cfsetispeed(&options, B19200);
  cfsetospeed(&options, B19200);

 
  /* No flow control */
  options.c_cflag &= ~CRTSCTS;  


  /* No parity, 1 stop bit, 8 data bits */
  options.c_cflag &= ~PARENB;
  options.c_cflag &= ~CSTOPB;  
  options.c_cflag &= ~CSIZE;
  options.c_cflag |= CS8;

  /* select raw input */
  options.c_lflag &= ~(ICANON | ECHO | ECHOE | ISIG | IEXTEN);
  options.c_iflag &= ~(BRKINT | ICRNL | INPCK | ISTRIP |IXON );
  options.c_oflag &= ~(OPOST);


  /* Allow reading and does not became owner */
  options.c_cflag |= (CLOCAL | CREAD);

  cfmakeraw(&options);

  return tcsetattr(fd, TCSANOW, &options);
}




/********************
  HIGH LEVEL ROUTINES
  These are the routines to be used by the client code
*********************/

/*!
 * Prints out the configuration of the port.
 * Should print 38400 Input and output speeds, no parity, 1 stop bit, 8 data bits,
 * flow control disabled, read enabled. If not, the port is not properly configured.
 * \param fd: handle of the port
 * \return nothing
 */
void CUBE_printconfig(int fd) {
  struct termios options;
  tcgetattr(fd, &options);
  printf("Detected port status:\n");

  if ( cfgetispeed(&options) == B57600 )
    printf("\tInput speed 57600\n");
  if ( cfgetospeed(&options) == B57600 )
    printf("\tOutput speed 57600\n");

  if ( options.c_cflag & PARENB )
    printf("\t Parity enabled\n");
  else printf("\tParity disabled\n");

  if ( options.c_cflag & CSTOPB )
    printf("\t2 stop bits\n");
  else printf("\t1 stop bit\n");

  if ( options.c_cflag & CS8 )
    printf("\t8 data bits\n");

  if ( options.c_cflag & CREAD )
    printf("\tAllowed to read\n");

  if ( options.c_cflag & CRTSCTS ) 
    printf("\tFlow control enabled\n");
  else
    printf("\tFlow control disabled\n");
    

}

/*!
 * Checks if the configuration of the port satisfies
 * the requirements of the Cube
 * \param fd: handle of the device
 * \return 1 if OK, 0 otherwise
 */
int CUBE_checkconfig(int fd) {

 struct termios options;
 tcgetattr(fd, &options);

 if ( ( ! ( cfgetispeed(&options) == B57600 ) ) ||   /* Input speed should must  be 38400 */
      ( ! ( cfgetospeed(&options) == B57600 ) ) ||   /* Output speed should must  be 38400 */
      ( options.c_cflag & PARENB )  ||               /* Parity bit must be disabled */
      ( options.c_cflag & CSTOPB )  ||               /* Two stops bits must be disabled */
      ( ! ( options.c_cflag & CS8 ) ) ||             /* Data lenght must be 8 bits */
      ( ! ( options.c_cflag & CREAD ) ) ||           /* Reading must be enables */
      ( options.c_cflag & CRTSCTS ) )                /* flow control must be disabled */
   return 0;
 else
   return 1;
}


/*! 
 * Starts communication with the cube device
 * \param dev: name of the device ("/dev/ttyS0" and so on)
 * \return the handle to the device or -1 in case of error
 */
int CUBE_open(const char* dev) {

  int handle;
  handle = _CUBE_open_port(dev);

  if ( handle == -1 )
    { perror("unixif:");
      return -1;
    }

  _CUBE_configure_port(handle);  /* the error code is not so significative */

//  if (CUBE_checkconfig(handle) ) 
//{ printf("checkconfig failed\n");
//    CUBE_printconfig(handle);
//    return -1;
//}
  CUBE_printconfig(handle);

  if (CUBE_reset(handle) ) 
{
  printf("reset failed\n");
    return -1;
}

printf("handle is %d\n",handle);  
  return handle;
}

int CUBE_reset(int fd) {
  const  char reset[2] = { 0x03, 0x0d};
  int i;

  i =  write(fd,reset,2);
  printf("\n CubeReset written %d \n",i);
  if (i !=2)
	{
		printf("Can't reset...\n");
		return(-1);
	}
  sleep(2); // give the cube watchdog some time

  return(0);
}


/*!
 * Closes the communication with the sensor. Indeed not really necessary.
 * \param fd: file handle 
 * \return 0 if success, -1 if some error occurred
 */
int CUBE_close(int fd) {
  if ( CUBE_reset(fd) )
    return -1;

    return close(fd);
}





