/***********************************************************/
/* (c) Peter Stuer 1996                                    */
/*                                                         */
/* AI-LAB Vrije Universiteit Brussel                       */
/*                                                         */
/***********************************************************/
/*                                                         */
/* Smb2pdl1.c vers 1.01                                    */
/*                                                         */
/***********************************************************/

/****************************************************************************/
/*                                                                          */
/* 1.01                                                                     */
/*   fixed bug causing pdl to exit afther limited run without giving chance */
/*   to save datalog.                                                       */
/*                                                                          */
/****************************************************************************/

#include <stdlib.h> /* malloc */
#include <string.h> /* strcpy, strlen */
#include <stdio.h> /* printf (in debug)*/
#include "smb2kern.h" /* Byte */
#include "smb2io.h"
#include "smb2pdl1.h"
#include <math.h> /* modf*/
#include "smb2dlog.h"




#define set_value(q,v) (((q) -> value) = (v))


struct process_struct;
typedef struct process_struct *pdl_process;
struct process_struct{
   char *name;
   void (*the_process)();
	 pdl_process next_process;
};

static quantity first_quantity = (quantity) NULL;
static pdl_process first_process = (pdl_process) NULL;
static quantity sensor_quantity[LASTSENSOR + 1];
static quantity actuator_quantity[LASTACTUATOR + 1];
static quantity dummy_quantity = (quantity) NULL;


int PauseButton = 0;

int storedumped;


byte ByteWise(float theIntegerAsFloat)
{
	if( theIntegerAsFloat > 255.0F)
	{
		return (byte) 255;
	}
	else
	{
		if (theIntegerAsFloat < 0.0F)
		{
			return (byte) 0;
		}
		else
		{
			return (byte)((int)(theIntegerAsFloat));
		}
	}
}

byte PDLDefaultFloatToByte(float theFloat)
{
	float fraction, integer;
	fraction = modff(theFloat, &integer);

 
	if(fraction < 0.5F)
	{
		return ByteWise(integer);
	}
	else
	{
		return ByteWise(integer + 1.0F);
	}
}

float PDLDefaultByteToFloat(byte theByte)
{
	return (float)theByte;
}

quantity make_quantity(const char *name)
{
	quantity q;

	q = (quantity) malloc (sizeof (struct quantity_struct));
	if(q == NULL)
	{
		printf("Could not allocate new quantity");
		return q;
	}
  q -> name = (char *)malloc (sizeof(char) * (strlen(name)+1));
  if((q -> name) == NULL)
	{
		printf("Could not allocate name of new quantity");
		return q;
	}

	strcpy(q -> name,name);
	q -> value     = 0.0;
	q -> max_value = 0.0;
	q -> min_value = 0.0;
	q -> new_value = 0.0;
	q -> FloatToByte = PDLDefaultFloatToByte;
	q -> ByteToFloat = PDLDefaultByteToFloat;
	q -> next_quantity = (quantity) NULL;
  return q;
}

quantity add_quantity(const char *name, float max, float min, float init_value)
{
  quantity q;

  q = make_quantity(name);
  q -> value = init_value;
  q -> max_value = max;
  q -> min_value = min;
  q -> new_value = 0;

  q -> next_quantity = first_quantity;
  first_quantity = q;
  return q;
}

void add_process(const char *name, void(*my_process)())
{
  pdl_process p;

  p = (pdl_process) malloc (sizeof (struct process_struct));
  if(p == NULL)
  {
    printf("Could not allocate new process_struct\n\r");
  }
  p -> name = (char *)malloc (sizeof (char) * (strlen(name) + 1));
  if((p -> name) == NULL)
  {
    printf("Could not allocate name of new process_struct\n\r");
  }
  strcpy (p -> name,name);
  p -> the_process = my_process;
  p -> next_process = first_process;
  first_process = p;
}




void connect_sensor(byte selector, quantity q)
{
	sensor_quantity[selector] = q;
	SMB2MarkSensorInUse(selector);
}

void connect_actuator(byte selector, quantity q)
{
	actuator_quantity[selector] = q;
	SMB2MarkActuatorInUse(selector);
}

void update_sensor_quantities(void)
{
	byte selector;

	/* Binary Device 1 */

	for(selector = SID_BIN0; selector <= SID_BIN7; selector++)
	{
		if(!(sensor[AP_BIN0] & (1U << (selector - SID_BIN0))))
		{
			set_value(sensor_quantity[selector],1);
		}
		else
		{
			set_value(sensor_quantity[selector],0);
		}
	}

	/* Binary Device 2 */
	
	for(selector = SID_BIN8; selector <= SID_BIN15; selector++)
	{
		if(sensor[AP_BIN8] & (1U << (selector - SID_BIN8)))
		{
			set_value(sensor_quantity[selector],((sensor_quantity[selector]-> ByteToFloat)(1)));
		}
		else
		{
			set_value(sensor_quantity[selector],((sensor_quantity[selector]-> ByteToFloat)(0)));
		}
	}

	/* All other devices (1 byte per device) */
		
	for(selector = SID_IR0; selector <= SID_PAC3_AC; selector++)
  {
		set_value(sensor_quantity[selector],((sensor_quantity[selector]-> ByteToFloat)(sensor[selector-14])));
	}

	/* correct the sign shift for the PAC accellerations */


	set_value(sensor_quantity[SID_PAC1_AC], value(sensor_quantity[SID_PAC1_AC]) - 128.0F);
	set_value(sensor_quantity[SID_PAC2_AC], value(sensor_quantity[SID_PAC2_AC]) - 128.0F);
	set_value(sensor_quantity[SID_PAC3_AC], value(sensor_quantity[SID_PAC3_AC]) - 128.0F);
	
}

void run_pdl_processes(void)
{
	pdl_process p;


	for (p =  first_process;
			 p != (pdl_process) NULL;
			 p =  p -> next_process)
		(*p -> the_process)();

}

void update_quantities(void)
{
  quantity q;
	float the_sum;

  for(q = first_quantity; q!=(quantity)NULL; q = q -> next_quantity)
  {
    the_sum = (q -> value) + (q -> new_value);
    if(the_sum > q -> max_value)
    {
      q -> value = q ->max_value;
    }
    else
    {
      if(the_sum < q -> min_value)
      {
				q -> value = q -> min_value;
      }
      else
      {
        q -> value = the_sum;
      }
    }
		q -> new_value = 0.0F;
  }
}

void update_action_quantities (void)
{
	/* Motors */

	if(ActuatorInUse[AID_MOTOR1])
	{ 

		 UpdateMotor1(((actuator_quantity[AID_MOTOR1]->FloatToByte))((value(actuator_quantity[AID_MOTOR1])+100)));
	}
	
	if(ActuatorInUse[AID_MOTOR2])
	{ 

		 UpdateMotor2(((actuator_quantity[AID_MOTOR2]->FloatToByte))((value(actuator_quantity[AID_MOTOR2])+100)));
	}

	if(ActuatorInUse[AID_MOTOR3])
	{ 
		 UpdateMotor3(((actuator_quantity[AID_MOTOR3]->FloatToByte))((value(actuator_quantity[AID_MOTOR3])+100)));
 	} 

	
	/* IR Emitters */

	UpdateIREmitters(((actuator_quantity[AID_IREM0]->FloatToByte)(((float)ActuatorInUse[AID_IREM0]) * value(actuator_quantity[AID_IREM0]))),
									 ((actuator_quantity[AID_IREM1]->FloatToByte)(((float)ActuatorInUse[AID_IREM1]) * value(actuator_quantity[AID_IREM1]))),
									 ((actuator_quantity[AID_IREM2]->FloatToByte)(((float)ActuatorInUse[AID_IREM2]) * value(actuator_quantity[AID_IREM2]))),
									 ((actuator_quantity[AID_IREM3]->FloatToByte)(((float)ActuatorInUse[AID_IREM3]) * value(actuator_quantity[AID_IREM3]))),
									 ((actuator_quantity[AID_IREM4]->FloatToByte)(((float)ActuatorInUse[AID_IREM4]) * value(actuator_quantity[AID_IREM4]))),
									 ((actuator_quantity[AID_IREM5]->FloatToByte)(((float)ActuatorInUse[AID_IREM5]) * value(actuator_quantity[AID_IREM5]))),
									 ((actuator_quantity[AID_IREM6]->FloatToByte)(((float)ActuatorInUse[AID_IREM6]) * value(actuator_quantity[AID_IREM6]))),
									 ((actuator_quantity[AID_IREM7]->FloatToByte)(((float)ActuatorInUse[AID_IREM7]) * value(actuator_quantity[AID_IREM7]))));

	/* display */

	UpdateDisplay(((actuator_quantity[AID_DISP1]->FloatToByte)(((float)ActuatorInUse[AID_DISP1]) * value(actuator_quantity[AID_DISP1]))),
								((actuator_quantity[AID_DISP2]->FloatToByte)(((float)ActuatorInUse[AID_DISP2]) * value(actuator_quantity[AID_DISP2]))),
								((actuator_quantity[AID_DISP3]->FloatToByte)(((float)ActuatorInUse[AID_DISP3]) * value(actuator_quantity[AID_DISP3]))),
								((actuator_quantity[AID_DISP4]->FloatToByte)(((float)ActuatorInUse[AID_DISP4]) * value(actuator_quantity[AID_DISP4]))));
								
}


void end_pdl(void)
{
	UpdateMotor1(100);
	UpdateMotor2(100);
	UpdateMotor3(100);
	UpdateDisplay(0x0F,0x0F,0x0F,0x0F);
}





void init_pdl(void)
{
  int selector;

	dummy_quantity = make_quantity("dummy");
  for (selector = FIRSTSENSOR; selector <= LASTSENSOR; selector++)
  {
    sensor_quantity[selector] = dummy_quantity;
  }
  for (selector = FIRSTACTUATOR; selector <= LASTACTUATOR; selector++)
  {
		actuator_quantity[selector] = dummy_quantity;
	}

	SMB2IOInit();
	
	SMB2MarkSensorInUse(SID_BIN15);
	
	UpdateDisplay(0x01,0x01,0x01,0x01);
	init_datalog();
}

void pause_robot(void)
{
		UpdateMotor1(100);
		UpdateMotor2(100);
		UpdateMotor3(100);
		printf("did pause_robot, stopped all motors\n\r");
}



void pause_pdl(void)
{
	printf("PDL paused\n\r");

	pause_robot();
	storedumped = 0;
	while (!Pause_Button_Down())
	{
		if(Log_Button_Down())
		{
			DumpStore();
			storedumped = 1;
		}
	}
	while (Pause_Button_Down())
	{
		if(storedumped)
		{
			ClearStore();
		}
	}
	printf("PDL reactivated\n\r");
}

int Pause_Button_Down()
{
	static int DownTime = 0;


	if(!(sensor[AP_BIN15]  & 0x80)) /*Sensor = 1, wich means switch is not puched (sic. dany)*/
	{
		if( (DownTime < 3))
		{
			DownTime ++;
		}

	}
	else /*sensor = 0*, according to dany's illogical mind this means the switch is pushed */
	{
		DownTime = 0;
	}

	if (DownTime == 3)
	{
		return 1;
	}
	else
	{
		return 0;
	}
}

	

	


void run_pdl(long cyclecounter)
{
	int safetymargin;
	
	SMB2IOStart();
	start_datalog();
	pause_pdl();
  if(cyclecounter < 0)
  {
    for(;;)
		{
			safetymargin = wait_for_tick();
			if(Pause_Button_Down())
      {
				pause_robot();
				while(Pause_Button_Down());
				pause_pdl();
      }
      update_sensor_quantities();
      run_pdl_processes();
      update_quantities();
			update_action_quantities();
			update_datalog();
			
    }
  }
  else
  {
    while(cyclecounter > 0)
    {
			cyclecounter--;
			safetymargin = wait_for_tick();
			if(Pause_Button_Down())
      {
        pause_pdl();
      }
      update_sensor_quantities();
      run_pdl_processes();
      update_quantities();
			update_action_quantities();
			update_datalog();
    }
		end_pdl();
		pause_pdl();
		printf("pdl system finished.\n");
  }
}

