#include <stdio.h>
#include <malloc.h>
#include <pdl.h>
#include <cubeos.h>
#include <tpu.h>
#include <tpud.h>
#include <i2cd.h>
#include <analog.h>
#include <digital.h>
#include <motor.h>
#include <ptimer.h>
#include <ttyio.h>
#include <softreset.h>
#include <schedule.h>

extern unsigned long _time_seconds;
extern unsigned short _time_ticks;

quantity qlisthead ;
actuator alisthead ;
sensor slisthead ;
pdl_process plisthead;

sensor binin[2];
sensor adin[24];
actuator daout[6];
actuator binout[2];
actuator motor[3];
sensor qd[2];

int dtvalue;

// quantitiy actuator sensor handling 

#ifdef IQ
quantity add_quantity(const char *name, short int max, short int min, short int init_value)
#endif
#ifdef FQ
quantity add_quantity(const char *name, float max, float min, float init_value)
#endif
{
        quantity q;

        q = (quantity) malloc (sizeof (struct quantity_struct));
        if (q == NULL)
        {
                printf("Could not allocate new quantity");
                return q;
	}
	q -> name = (char *)malloc (sizeof(char) * (strlen(name)+1));
	if((q -> name) == NULL)
        {
                printf("Could not allocate name of new quantity");
                return q;
	}

        strcpy(q -> name,name);
	q -> value = init_value;
	q -> max_value = max;
	q -> min_value = min;
	q -> new_value = init_value;
	q -> a_sum     = 0;
	q -> a_count   = 0;
	q -> act=NULL;
	q -> sen=NULL;
	q -> next = qlisthead;
	qlisthead = q;

  return q;
}

actuator new_actuator(const char *name,void (*act_set) (actuator a, quantity q),  short int (*act_update) (actuator a), short int act_update_arg)
{
        actuator q;

        q = (actuator) malloc (sizeof (struct actuator_struct));
        if (q == NULL)
        {
                printf("Could not allocate new actuator");
                return q;
	}
	q -> name = (char *)malloc (sizeof(char) * (strlen(name)+1));
	if((q -> name) == NULL)
        {
                printf("Could not allocate name of new actuator");
                return q;
	}

        strcpy(q -> name,name);

	q -> set= act_set;
        q -> update = act_update;
	q -> update_arg = act_update_arg;
	q -> inuse=0;
        q -> next = (actuator) NULL;
  return q;
}

void add_actuator(actuator q)
{
	q -> next = alisthead;
	alisthead = q;
}

sensor new_sensor(const char *name, void (*sen_get) (sensor s, quantity q),  short int (*sen_update) (sensor s), short int sen_update_arg)
{
        sensor q;

        q = (sensor) malloc (sizeof (struct sensor_struct));
        if (q == NULL)
        {
                printf("Could not allocate new sensor");
                return q;
	}
	q -> name = (char *)malloc (sizeof(char) * (strlen(name)+1));
	if((q -> name) == NULL)
        {
                printf("Could not allocate name of new sensor");
                return q;
	}

        strcpy(q -> name,name);
	q -> get = sen_get;
	q -> update = sen_update;
	q -> update_arg = sen_update_arg;
	q -> inuse =0;
	q -> next=(sensor) NULL;
  return q;
}

void add_sensor(sensor q)
{

	q -> next = slisthead;
	slisthead = q;
}

// process handler !!! This is not using cubeos threads !!!

void add_process(const char *name, void(*my_process)())
{
  pdl_process p;

  p = (pdl_process) malloc (sizeof (struct process_struct));
  if(p == NULL)
  {
    printf("Could not allocate new process_struct\n\r");
  }
  p -> name = (char *)malloc (sizeof (char) * (strlen(name) + 1));
  if((p -> name) == NULL)
  {
    printf("Could not allocate name of new process_struct\n\r");
  }
  strcpy (p -> name,name);
  p -> the_process = my_process;
  p -> next = plisthead;
  plisthead = p;
}


// sensor actuator class handler 

// get - set 
// get : from sensor to quantity
// set : from quantity to actuator
// the quantity contains values for which the rang has been set by the user
// the sensor/actuator contains hardware-dependent values. 
// get-set has to do the conversion

void binin_get(sensor s,quantity q)
{
q->value= (((s->value)&(1<<q->sen_arg))>>q->sen_arg); /* sets q->value the value of the a'th sensor bit */
}

void binout_set(actuator a,quantity q){
a->value=((a->value&(0xff- (1<<q->act_arg)))| (q->value!=0?(1<<q->act_arg):0)); /* Ouch ! Should set the sen_arg'th bit in s->value to the value of q */
}

void ad_get(sensor s,quantity q)
{
q->value= s->value;
}

void da_set(actuator a,quantity q){
a->value=q->value;
}

void qd_get(sensor s,quantity q)
{// how to scale this in an useful way ?
q->value= s->value;
}

// update :
// for sensors: read hardware, write sensor->value
// for actuators: read actuator->value, write hardware

short int binin_update(sensor s)
{
//printf("binin_update called for sensor %s, arg %d\n",s->name,s->update_arg);
s->value = ReadBinIn(s->update_arg);
//printf("Answer was %d\n",s->value);
return(0);
}

short int binout_update(actuator a)
{
//printf("binout_update called for actuator %s, arg %d",a->name,a->update_arg);
//printf(" Value is %d\n",a->value);
WriteBinOut(a->update_arg,a->value);
return(0);
}

short int ad_update(sensor s)
{
//printf("ad_update called for sensor %s, arg %d\n",s->name,s->update_arg);
s->value = ReadAnalogIn (s->update_arg/4, s->update_arg%4);

return(0);
}

short int qd_update(sensor s)
{
//printf("qd_update called for sensor %s, arg %d\n",s->name,s->update_arg);
switch(s->update_arg) {
	case 0:
		s->value = motor_leftencoder();
		break;
	case 1:
		s->value = motor_rightencoder();
		break;
	default:
	}
return(0);
}

short int da_update(actuator a)
{
//printf("da_update called for actuator %s, arg %d",a->name,a->update_arg);
//printf(" Value is %d\n",a->value);
WriteAnalogOut(a->update_arg,a->value);
return(0);
}

void motor_set(actuator a,quantity q)
{
int scale;
// now the trick happens here. We use (max_value-min_value) to scale.
scale = ((int)q->max_value)-((int)q->min_value);
a->value = (short)(((int)q->value*20000)/scale);
//printf("q->value %d a->value %d\n",q->value,a->value);

}

short int motor_update(actuator a)
{
//printf("motor_update called for actuator %s, arg %d",a->name,a->update_arg);
//printf(" Value is %d\n",a->value);
switch (a->update_arg) {
	case 0: // this is the left motor
//		printf("setting left motor to %d\n",a->value);
		motor_setPWleft(a->value);
		break;
	case 1: // this is the right motor
//		printf("setting right motor to %d\n",a->value);
		motor_setPWright(a->value);
		break;
	default: // maybe we could add an error message here
	}

return(0);
}

void connect_sensor(sensor s,short int arg, quantity q)
{
q->sen = s;
q->sen_arg=arg;
if (s->inuse==0) { add_sensor(s); }
s->inuse++;
}

void connect_actuator(actuator a,short int arg, quantity q)
{
q->act = a;
q->act_arg=arg;

if (a->inuse==0) { add_actuator(a); }

a->inuse++;
}

void init_pdl()
{
short int i;
printf("This is init_pdl !");


qlisthead = (quantity) NULL;
alisthead = (actuator) NULL;
slisthead = (sensor) NULL;
plisthead = (pdl_process) NULL;


i2c_init (I2CA, I2CA_BASE);
i2c_init (I2CB, I2CB_BASE);
init_analog ();
init_digital ();


binin[0] = new_sensor("binin",&binin_get,&binin_update,0);
binin[1] = new_sensor("binin",&binin_get,&binin_update,1);
for (i=0;i<16;i++) {adin[i] = new_sensor("ad",&ad_get,&ad_update,i); }
for (i=0;i<6;i++) {daout[i] = new_actuator("da",&da_set,&da_update,i); }
binout[0] = new_actuator("binout",&binout_set,&binout_update,0);
binout[1] = new_actuator("binout",&binout_set,&binout_update,1);


init_fastbin();
TPU_init();
motor_set_directionbits();
motor_set_tpchannels(100);
motor[0] = new_actuator("motor",&motor_set,&motor_update,0);
motor[1] = new_actuator("motor",&motor_set,&motor_update,1);

qd[0] = new_sensor("qd",&qd_get,&qd_update,0);
qd[1] = new_sensor("qd",&qd_get,&qd_update,1);


#ifdef USE_SMBIIBASE

TPU_makepwm(0);
TPU_makepwm(1);
TPU_setpwmperiod(0,26);
TPU_setpwmdc(0,50);
TPU_setpwmperiod(1,6666);
TPU_setpwmhigh(1,5000);

#endif

}

void show_quantities()
{
quantity q;
q=qlisthead;
while (q!=NULL) {
#ifdef IQ
	printf ("Quanitiy %s: Value %d\n",q->name,q->value);
#endif
#ifdef FQ
	printf ("Quanitiy %s: Value %f\n",q->name,q->value);
#endif
	if (q->sen) printf("connected to sensor %s.%d.%d\n",q->sen->name,q->sen->update_arg,q->sen_arg); 
	if (q->act) printf("connected to actuator %s.%d.%d\n",q->act->name,q->act->update_arg,q->act_arg); 
	q=q->next;
}
}
void show_sensors()
{
sensor s;
s=slisthead;
while (s!=NULL) {
	printf ("Sensor %s.%d: Value %d\n",s->name,s->update_arg,s->value);
	printf (" inuse: %d\n",s->inuse);
	s=s->next;
}
}
void show_actuators()
{
actuator a;
a=alisthead;
while (a!=NULL) {
	printf ("Actuator %s.%d: Value %d\n",a->name,a->update_arg,a->value);
	printf (" inuse: %d\n",a->inuse);
	a=a->next;
}
}

void update_sensors()
{
sensor s;
s=slisthead;
while (s!=NULL) {
	if (s->update(s) ) { printf(" Update of Sensor %s.%d failed\n",s->name,s->update_arg); }
	s=s->next;
}
}


void update_actuators()
{
actuator a;
a=alisthead;
while (a!=NULL) {
	if (a->update(a) ) {printf(" Update of Actuator %s.%d failed\n",a->name,a->update_arg); }
	a=a->next;
}
}

void update_quantities()
{
quantity q;
q=qlisthead;
while (q!=NULL) {
        if (q->a_count)
	{
	  q->new_value+=(q->a_sum/q->a_count);
	  q->a_count=0;q->a_sum=0;
	}
	if (q->new_value<q->min_value) { q->value = q->min_value;}
	else if (q->new_value>q->max_value) {q->value = q->max_value;}
	else {	q->value=q->new_value;}

	q->new_value = q->value;

	if (q->sen) q->sen->get(q->sen,q);
	if (q->act) q->act->set(q->act,q);
	q=q->next;
}

}

unsigned int  pdl_cycle()
{
unsigned long s1,s2;
unsigned short t1,t2;

unsigned int deltat;

pdl_process p;

disable();
s1 = _time_seconds;
t1 = _time_ticks;
enable();

update_sensors();

p=plisthead;
while (p!=NULL) {
	p->the_process();
	p=p->next;
}
update_quantities();
update_actuators();

disable();
s2 = _time_seconds;
t2 = _time_ticks;
enable();

deltat = ((s2-s1)*TICKS_PER_SECOND)+(t2-t1);

return deltat;
}

void pdl_mon()
{
char ch;

while (1) {
ch = inchar();
switch (ch) {
	case 's': 
		show_sensors();
		break;
	case 'a':
		show_actuators();
		break;
	case 'q':
		show_quantities();
		break;
	case 'r':
		softreset();
		break;
}
}
}


int dt()
{
return dtvalue;
}

void run_pdl(short int n)
{
int i;
unsigned int time_s;
unsigned int time_t;
unsigned int ticks;
//int pid;

//pid = create(pdl_mon);
dtvalue=0;

if (n==0) {
	printf (" running pdl forever\n");
	while (1) dtvalue=pdl_cycle();
	}
else {
	time_s=0;time_t=0;
	printf (" running pdl %d times\n",n);
	for(i=0;i<n;i++) {
		ticks=pdl_cycle();
		dtvalue=ticks;
//		printf("ticks = %d\n",ticks);
		time_t+=ticks;
		time_s+=(time_t/TICKS_PER_SECOND);
		time_t%=TICKS_PER_SECOND;
	}
	printf(" %d cycles %ds %dt  %f ticks/cycle\n",n,time_s,time_t,time_s+1.0*time_t/TICKS_PER_SECOND);
}
}

