/*  src_experimental/drivers/fastbin/fastbin.c
   CubeOS Version 0.4.90 experimental
   Copyright (C) 1999,2000 Holger Kenn

   CubeOS is free software; you can redistribute it and/or
   modify it under the terms of the GNU Library General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or any later version.

   CubeOS is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Library General Public License for more details.

 */
#include <cubeos.h>
#include <kerror.h>
#include <fastbin.h>
#include <mc68681.h>
#include <mc68332.h>
#include <tpud.h>
#include <io_duart.h>		/* for OPCR */
#include <hw_drivers.h>

#define MAXFB 16
/*! \file fastbin.c
\ingroup FBIN
*/

/*!
\brief internal fastbin config structure
\ingroup FBIN
*/
struct fb_info {
	char type;
	char id;
};

/*!
\brief internal fastbin config structure
\ingroup FBIN
*/
struct fb_info fbs[MAXFB];

unsigned int _FBIN_ttlfastbin_state;
unsigned int FBIN_rcjfastbin_state;

/*!
\brief prepares the fastbin system for use
\ingroup FBIN

prepares the fastbin system for use. Initializes the fastbin type
code for all fastbins to FBTYPE_EMPTY.
*/
int FBIN_init ()
{
	int i;
//	printf("Fastbin init\n");

	for (i = 1; i < MAXFB; i++) {
		fbs[i].type = FBTYPE_EMPTY;
		fbs[i].id = 0;
	}
	RCJ_init ();
	RG_init ();

	return 0;
}

/*!
\brief configures one fastbin for usage
\param nr is the fastbin number
\param type is the fastbin type
\param id is the fastbin id
\ingroup FBIN

This function configures one of the 16 hardware-independent fast
binary outputs (FBINs) for use. type specifies the hardware
connection of the fastbin. Available types are:

FBIN_DUART for MC68681 DUART binary outputs as they are used for 
motor control in the SMBII and Cube 1 hardware.

FBIN_TPU for using TPU channels as binary outputs via the DIO function

FBIN_TTL for the dedicated fastbin register in Cube3.

FBIN_CPU for using MC68332 cpu output signal pins for motor control.
Currently, only the PCS1 pin is supported.

FBIN_RCJ for using a decoded external 8bit register as used on RCJ boards.

All types require an additional id parameter to specify the pin to use.
For the register, id denotes the bit from LSB to MSB, for the TPU, id is
the channel to use, for DUART and CPU, the pins are implementation-dependent. 

*/
int FBIN_make (int nr, char type, char id)
{
//      printf ("make_fastbin %d: \n", nr);

	if (nr >= MAXFB)
		return (-1);
	fbs[nr].type = type;
	fbs[nr].id = id;

	switch (type) {
	case FBTYPE_DUART:
//              printf ("type DUART, id %d \n", id);
		/* writebyte(DUART_OPCR,0); */
#ifdef DUART_BASE
		switch (id) {
		case 2:
			DUART_clearOPCR (0);
			DUART_clearOPCR (1);
			break;
		case 3:
			DUART_clearOPCR (2);
			DUART_clearOPCR (3);
			break;
		case 4:
		case 5:
		case 6:
		case 7:
			DUART_clearOPCR (id);
			break;
		default:
			fbs[nr].type = FBTYPE_EMPTY;
			return (-1);
		}
#else
		return -1;
#endif
		break;
	case FBTYPE_TPU:
//              printf ("type TPU, id %d \n", id);
		TPU_makedio (id);
		TPU_setdio (id, 0);
		break;
	case FBTYPE_TTL:
//              printf ("type TTL, id %d ", id);
#ifdef FASTBIN_BASE
		if (FASTBIN_BASE == 0) {
			fbs[nr].type = FBTYPE_EMPTY;
			KERN_complain (ERR_WARN, "Trying to use fastbin register without any register around");
			return (-1);
		} else {
			/* clear all fastbins */
			writeshort (FASTBIN_BASE, 0);
			/* and save that fact */
			_FBIN_ttlfastbin_state = 0;
		}
#else
		fbs[nr].type = FBTYPE_EMPTY;
		KERN_complain (ERR_WARN, "Trying to use fastbin register without any register around");
		return (-1);
#endif
		break;
	case FBTYPE_CPU:
//              printf ("type CPU, id %d \n", id);
		switch (id) {
		case FBCPU_PCS1:
			writebyte (SPI_QPDR, readbyte (SPI_QPDR) & 0xEF);
			writebyte (SPI_QPAR, readbyte (SPI_QPAR) & 0xEF);
			writebyte (SPI_QDDR, readbyte (SPI_QDDR) | 0x10);
			break;
		default:
			fbs[nr].type = FBTYPE_EMPTY;
			return (-1);
		}
		break;
	case FBTYPE_RCJ:
//              printf ("type RCJ, id %d \n", id);
		if (id > 7) {
			fbs[nr].type = FBTYPE_EMPTY;
			return (-1);
		}
#ifdef RCJFB_BASE
		if ((RCJFB_BASE) == 0) {
			fbs[nr].type = FBTYPE_EMPTY;
			KERN_complain (ERR_WARN, "Trying to use rcj fastbin register on non-rcj hardware");
			return (-1);
		} else {
			/* clear all fastbins */
//			printf("doing write\n");
			writebyte (RCJFB_BASE, 0);
			/* and save that fact */
			FBIN_rcjfastbin_state = 0;
		}
#else
		fbs[nr].type = FBTYPE_EMPTY;
		KERN_complain (ERR_WARN, "Trying to use rcj fastbin register on non-rcj hardware");
#endif
		break;
	default:
		return (-1);
	}
//      printf (" OK\n");
	return (0);


}

/*!
\brief sets a preconfigured fast binary output
\param i is the fastbin number
\param v is the level (1 or 0)
\ingroup FBIN

*/
int FBIN_set (int i, char v)
{
	char type, id;

	type = fbs[i].type;
	id = fbs[i].id;
	switch (type) {
	case FBTYPE_DUART:
#ifdef DUART_BASE
		switch (id) {
		case 2:
			if (v) {
				setDUARTOPR (2);
			} else {
				clearDUARTOPR (2);
			}
			break;
		case 3:
			if (v) {
				setDUARTOPR (3);
			} else {
				clearDUARTOPR (3);
			}
			break;
		case 4:
			if (v) {
				setDUARTOPR (4);
			} else {
				clearDUARTOPR (4);
			}
			break;
		case 5:
			if (v) {
				setDUARTOPR (5);
			} else {
				clearDUARTOPR (5);
			}
			break;
		case 6:
			if (v) {
				setDUARTOPR (6);
			} else {
				clearDUARTOPR (6);
			}
			break;
		case 7:
			if (v) {
				setDUARTOPR (7);
			} else {
				clearDUARTOPR (7);
			}
			break;
		default:
			return -1;

		}
#else
		return -1;
#endif
		break;
	case FBTYPE_TPU:
		TPU_setdio (id, v);
		break;
	case FBTYPE_TTL:
		if (v) {
			_FBIN_ttlfastbin_state |= (1 << id);
		} else {
			_FBIN_ttlfastbin_state &= (~(1 << id));
		}
#ifdef FASTBIN_BASE
		writeshort (FASTBIN_BASE, _FBIN_ttlfastbin_state);
#endif
		break;
	case FBTYPE_CPU:
		switch (id) {
		case FBCPU_PCS1:
			if (v) {
				writebyte (SPI_QPDR, readbyte (SPI_QPDR) | 0x10);
			} else {
				writebyte (SPI_QPDR, readbyte (SPI_QPDR) & 0xEF);
			}
			break;
		default:
			return (-1);
		}
		break;
	case FBTYPE_RCJ:
//              printf("register is %x\n",FBIN_rcjfastbin_state);
//              printf(" id is %d v is %d\n",id,v);
		if (v) {
			FBIN_rcjfastbin_state |= ((1) << id);
		} else {
			FBIN_rcjfastbin_state &= (~(1 << id));
		}
//              printf("setting register to %x\n",FBIN_rcjfastbin_state);
#ifdef RCJFB_BASE
		writebyte (RCJFB_BASE, FBIN_rcjfastbin_state);
#endif
		break;
	default:
		return (-1);
	}
	return 0;
}
