/*  src_experimental/kernel/cubeos.h
    CubeOS Version 0.4.90 experimental
    Copyright (C) 1999,2000 Holger Kenn

    CubeOS is free software; you can redistribute it and/or
    modify it under the terms of the GNU Library General Public
    License as published by the Free Software Foundation; either
    version 2 of the License, or any later version.

    CubeOS is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Library General Public License for more details.

*/  
#define NULL 0
#ifndef CUBEOS_H
#define CUBEOS_H
#include <config.h>
#include <mc68332.h>
#include <sys_var.h>
#include <schedule.h>
#include <taskconst.h>
/*! \file cubeos.h
\ingroup KERN
*/

/*! \fn static inline void disable()
\brief disables interrupt processing
\ingroup KERN

disable() disabled interrupt processing by setting 
the interrupt mask in the CPU status register.
Note: This functions should be used with extreme care since it also disables IO
processing and preemptive scheduling.
*/

/*! \fn static inline void enable()
\brief enable interrupt processing
\ingroup KERN

enable() enables interrupt processing by clearing 
the interrupt mask in the CPU status register.
Note: This functions should be used with extreme care since it also re-enables IO
processing and preemptive scheduling.
*/

#ifdef NEW_INTSEM
static inline void disable()
{
if ((_KERN_disablecount++) == 0)
	{
	/* we're the first to disable interrupts */
	asm("move.w %%sr,%0" : "=m" (KERN_srsave) ); 
	asm("ori.w #0x0700,%sr");  /* Disable Interrupts */
	}
}

static inline void enable()
{
if ((--KERN_disablecount)<=0){
     if ((KERN_srsave | 0x0700) != 0x0700)
	asm("move.w %0,%%sr" : :"m" (KERN_srsave)); 
     else /* something went wrong, fall back to enabled interrupts */
	asm("move.w #0x2100,%sr"); 
     _KERN_disablecount=0;
     }
}
#else
static inline void disable()
{
	asm("ori.w #0x0700,%sr");  /* Disable Interrupts */
}
static inline void enable()
{
	asm("move.w #0x2000,%sr"); 
}
#endif

#ifdef MULTI_SEM

#include <seminl.h>

#else

/*!
 \defgroup ssem Simple CubeOS/POSIX Semaphores with spinlock

 These semaphores are SAFE as long as there is only one producer and
 one consumer, like in i2cd
 However, they are not Mutex-Safe !!! 
*/
typedef int sem_t; //!< \ingroup ssem \var sem_t for simple semaphores 
#define sem_post(x) x++; //!< \ingroup ssem increment semaphore counter
#define sem_signal(x) sem_post(x) //!< \ingroup ssem alias for post
#define sem_wait(x) x--;while(x<0) //!< \ingroup ssem waits for a semaphore to become available. Does not return a propper value ?!?


#define sem_trywait(x) (x<0) //!< \ingroup ssem tests, if a semaphore might block when wait()ed upon.
#define sem_getvalue(x,i) (*i=x) //!< \ingroup ssem gets the current semaphore counter value


#define sem_init(x,i,v) (x.c=v) //!< \ingroup ssem  posix conventions. Quite useless here, but anyway 
#define sem_destroy(x) (0) //!< \ingroup ssem  posix conventions. Quite useless here, but anyway 

#endif



/*  mutex */
/* Should be posix too. */
//#define mutex sem_t; 
/* A mutex is a semaphore that starts out eqal 1 */
//#define mutex_init(i) sem_init(i,1,1)
//#define mutex_enter(i) sem_wait(i)
//#define mutex_leave(i) sem_signal(i)
/* This is unsafe !!! */



#define readbyte(x)    (*(volatile unsigned char *)x)	//!< reads one byte from memory location x
#define writebyte(x,y) *(unsigned char *)(x) = y		//!< writes one byte y to memory location x
#define readshort(x)    (*(unsigned short *)(void *)(x)) //!< reads one short from memory location x
#define writeshort(x,y) *(unsigned short *)(x) = y	//!< writes one short y to memory location x
#define readint(x)    (*(unsigned int *)(void *)(x))	//!< reads one int from memory location x
#define writeint(x,y) *(unsigned int *)(x) = y		//!<  writes one int y to memory location x

/*! \def writeshortpos(x,y,mask,pos)
\brief writes some bits of y to memory location x.

\hideinitializer
writeshortpos(x,y,mask,pos) masks y with mask, shifts it pos bits left and masks the remaining bits into
the memory location x.
example: writeshortpos(TPU_CFSR3,0x9,0xf,0x8) writes function code 0x9 (4 bits wide, so mask 0xf) into TPU 
channel function select register for TPU Channel 2 (bits 8-11)
*/
#define writeshortpos(x,y,mask,pos) writeshort(x,(readshort(x)\
				      &(~((mask)<<(pos)))) \
				      |(((mask)&y)<<(pos)))

/*! \def readshortpos(x,mask,pos)
\brief reads some bits of y to memory location x.


\hideinitializer
readshortpos(x,mask,pos) reads memoy location x, shifts it pos bits right and masks the remaining bits with mask.
example: readshortshortpos(TPU_CFSR3,0xf,0x8) reads the function code (4 bits wide, so mask 0xf) out of the TPU 
channel function select register of TPU Channel 2 (bits 8-11)
*/
#define readshortpos(x,mask,pos) ((readshort(x)>>(pos))&(mask))

/*! 
\brief init control of the CUBE-II onboard status LED.

This reconfigures the Port E of the mc68332 so that one bit is used for output.
*/
static inline void init_LED()
{
writebyte(SIM_PORTE,0x0);
writebyte(SIM_DDRE,0x8);
writebyte(SIM_PEPAR,0xf7);
}
//! switches the CUBE-II onboard LED on
static inline void LED_ON()
{
writebyte(SIM_PORTE,0x8);
}

//! switches the CUBE-II onboard LED off
static inline void LED_OFF()
{
writebyte(SIM_PORTE,0x0);
}

#endif
