/*  src_experimental/kernel/schedule.c
   CubeOS Version 0.4.90 experimental
   Copyright (C) 1999,2000 Holger Kenn

   CubeOS is free software; you can redistribute it and/or
   modify it under the terms of the GNU Library General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or any later version.

   CubeOS is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Library General Public License for more details.

 */

/*! \file schedule.c
\ingroup KERN
*/

#undef SCHED_DEBUG
/* This is my first scheduler */
#include <stddef.h>
#include <stdio.h>
#include <signal.h>
#include <cubeos.h>
#include <sys_var.h>
#include <context.h>
#include <schedule.h>
#include <kerror.h>
#include <malloc.h>
#include <cubereent.h>
#include <list.h>


#define MAGIC 0xbabababa;

int _LIBC_init_reent();

struct process _KERN_ptable[MAX_PROCESSNUM];	/* The process table */
//int _KERN_deltahead; /* this is the head of the delta list */

list _KERN_delta;
list _KERN_prio[MAX_PRIONUM+1];

/*!
\brief this ends a thread
\ingroup KERN
*/
void _KERN_taskend ()
{
	_KERN_ptable[__MYPID].signal |= SIGKILL;
	KERN_schedule ();
}

/*!
\brief initalizes a context
\ingroup KERN
*/
regptr
_KERN_initcontext (regptr context, short sr, void *sp, void *function)
{
	/* initializes context so that it looks like function is just called */

	int i;

	void *stack;

	stack = sp;

	for (i = 0; i < PNREGS; i++)
		(*context).regs[i] = 0;		/* Clear everything, just in case... */

	(*context).regs[POS_SR] = sr;	/* set up SR */


	(*context).regs[POS_PC] = (unsigned long) function;	/* set up PC */

/*      The context switch is called in KERN_schedule(). The initial stack setup has to
   simulate this origin. The Stack that we're pointing to must look like this:

   *KERN_contextsw()
   *KERN_schedule()
   *function()
 */

	/* Stack Bottom */
	writeint (stack, (unsigned int) MAGIC);
	stack -= 4;
	writeint (stack, (unsigned int) _KERN_taskend);


	(*context).regs[POS_SSP] = (unsigned long) stack;	/* set up SSP */


	return context;
}


/*!
\brief create a new threas
\param function is a pointer to the function to be executed by the thread
\param prio is the priority of the thread to be created
\ingroup KERN
*/
int KERN_create_prio (void *function,int prio)
{				/* no arguments yet */
	int i, j;
	short sr;
	void *stack;
	if ((prio<0)||(prio>MAX_PRIONUM))
		return(-1);

/* MUTEX! */
	disable ();

	/* find empty process slot */
	i = 1;
	while ((i < MAX_PROCESSNUM) && (_KERN_ptable[i].state != STATE_EMPTY))
		i++;
	if (i == MAX_PROCESSNUM) {	/* No processes left. */
		enable ();
		KERN_complain (ERR_EMERG, "process table full");
		/* errno=ERR_NO_MORE_PROCESSES; */
		return (-1);
	}
	for (j = 0; j < PNREGS; j++)
		_KERN_ptable[i].regs.regs[j] = 0;	/* Zero out Slot i */
	/* Slot i is now ready to use */

	stack = (void *) malloc (TASK_INIT_STACKSIZE);	/* Get us a stack */
	if (stack == NULL) {
		enable ();
		KERN_complain (ERR_EMERG, "no room for stack");
		return (-1);
	}
	sr = TASK_INIT_SR;	/* Supervisor state */
	_KERN_ptable[i].irq = TASK_INIT_IRQLVL;		/* standard IRQ mask */

	_KERN_ptable[i].stack = stack;	/* for freeing the stack later */

	_KERN_initcontext (&(_KERN_ptable[i].regs), sr, (stack + TASK_INIT_STACKSIZE), function);
	_LIBC_init_reent (&_KERN_ptable[i].reent);
	_KERN_ptable[i].ppid = getpid ();	/* Parent ID */

	_KERN_ptable[i].state = STATE_READY;
	_KERN_ptable[i].signal = 0;
	_KERN_ptable[i].next = NO_TASK;		/* old list handling */

	_KERN_ptable[i].time_delta = 0;
	_KERN_ptable[i].prio = prio;

//	_KERN_ptable[i].me.data = &(_KERN_ptable[i]);
//	_KERN_ptable[i].me.len = sizeof (_KERN_ptable[i]);
	LIST_insert_tail (&_KERN_prio[prio], &(_KERN_ptable[i].me));
/* MUTEX END */
	enable ();
	return i;
}

/*!
\brief starts a new thread
\param function is a pointer to the function to be executed by the thread
\ingroup KERN
*/
int KERN_create (void *function)
{			
KERN_create_prio (function,0);
}

/*!
\brief this is the scheduler. It is either called by the periodic timer interrupt or from a user thread
\ingroup KERN
*/
void KERN_schedule (void)
{
	int old, new;

/* prepare to switch context */

      asm ("move.w %%sr,%0":"=m" (_KERN_context_srsave));
	asm ("ori.w #0x0700,%sr");	/* Disable Interrupts */


	old = getpid ();
	if ((_KERN_ptable[old].state != STATE_RUNNING) && (_KERN_ptable[old].state != STATE_SUSPEND)) {
		/* We're not the running task ?!? */
	      asm ("move.w %0,%%sr": :"m" (_KERN_context_srsave));
		KERN_complain (ERR_PANIC, "scheduler not called from running task");
		return;		// we're in panic, so this will not be called 

	}
	if ((_KERN_ptable[old].signal & (1 << SIGKILL))) {
		free (_KERN_ptable[old].stack);
		LIST_delete (&_KERN_ptable[old].me);
		/* removes thread from any queue */
		_KERN_ptable[old].state = STATE_EMPTY;
	}

#ifdef PLRR_SCHEDULER
/* priority-less round robin */
	new = old;
	while ((new < MAX_PROCESSNUM) && (_KERN_ptable[new].state != STATE_READY))
		new++;
	if (new == MAX_PROCESSNUM) {	/* wrap around */
		new = 0;
		while ((new < old) && (_KERN_ptable[new].state != STATE_READY))
			new++;
	}
#else
#ifdef RR_SCHEDULER
/* round robin over priorities, */
	{
		int prio = MAX_PRIONUM;
		int quit = 0;
		entry *this;

#ifdef SCHED_DEBUG
		printf("s%d\n",old);
		printf("task = 0x%x\n",(unsigned int)&(_KERN_ptable[old].me));
#endif
		new = old;


		while ((!quit) && (prio >= 0)) {
			/* look into process class prio */
			if (LIST_entries (&_KERN_prio[prio]) > 0) {
				/* there are processes in this class */
				this = LIST_head (&_KERN_prio[prio]);
				/* this should give us the next thread to run */
				/* the rest are sanity checks */
				while (
					      (this) &&
					      (this->data) &&
					      (((struct process *) (this->data))->state != STATE_READY))
					this = this->next;
				if (this)
					quit = 1;	/* if not, we'll retry one class lower */
			}
			prio--;
		}
		if (quit == 1) {	/* we've found another thread */
#ifdef SCHED_DEBUG
			printf("this = %x\n",(unsigned int) this);
#endif
			new = ((struct process *) (this->data))->pid;
#ifdef SCHED_DEBUG
			printf("n%d\n",new);
#endif
		}

	}
#else
      asm ("move.w %0,%%sr": :"m" (_KERN_context_srsave));
	KERN_complain (ERR_PANIC, "NO Scheduler defined !");

#endif
#endif
	if ((new <0)
	    ||(new>MAX_PROCESSNUM)
	    ||(
	       (_KERN_ptable[new].state!=STATE_READY)
	       &&(new!=old))) {
			printf("new=%d\n",new);
			if ((new >0)&&(new<MAX_PROCESSNUM))
				printf("new.state = %d",_KERN_ptable[new].state);
			KERN_complain (ERR_PANIC, "Scheduler did something stupid !");
	}
	if (old == new) {	/* Nobody else ready to run */
		if (_KERN_ptable[old].state == STATE_EMPTY) {
			/* We've just killed the last task ?!? */
		      asm ("move.w %0,%%sr": :"m" (_KERN_context_srsave));
			KERN_complain (ERR_PANIC, "AIEEE, just killed my last task !");
			while (1);
			return;	// we're in panic, so this will not be called 

		}
	      asm ("move.w %0,%%sr": :"m" (_KERN_context_srsave));
		return;
	}
	if (_KERN_ptable[old].state == STATE_RUNNING) {
		_KERN_ptable[old].state = STATE_READY;
		LIST_insert_tail (&_KERN_prio[_KERN_ptable[old].prio], &_KERN_ptable[old].me);
		/* insert old thread into its process class */
	}
	__MYPID = new;

	_KERN_ptable[new].state = STATE_RUNNING;
	LIST_delete (&_KERN_ptable[new].me);
	/* remove thread from its process class */

/*      print("\n\rcontext\n\rStack:");putnum(_KERN_ptable[new].regs.regs[POS_SSP]);print("\n\r");
   print("\n\rFramep:");putnum(_KERN_ptable[new].regs.regs[POS_A6]);print("\n\r"); */

	_impure_ptr = &(_KERN_ptable[__MYPID].reent);

	KERN_contextsw (&(_KERN_ptable[old].regs), &(_KERN_ptable[new].regs));

	/* now we're the new task ! */
/*
   asm("move.l %%sp,%0":"m=" (stackp): );

   printf("Stack:");putnum(stackp);print("\n\r");
   asm("move.l %%fp,%0":"m=" (stackp): );

   printf("Framep:");putnum(stackp);print("\n\r");
 */

	return;

}

/*!
\brief initalizes the scheduler
\ingroup KERN
*/
int KERN_schedinit ()
{
	int i, j;


	__MYPID = 0;

//      _KERN_deltahead=NO_TASK; /* means deltalist is inactive */

	LIST_init (&_KERN_delta);
	_KERN_delta.type = LIST_TYPE_SYS;

	for (i = 0; i < MAX_PRIONUM; i++) {
		LIST_init (&_KERN_prio[i]);
		_KERN_prio[i].type = LIST_TYPE_PRIO;
	}

	/* empty process slots */
	for (i = 0; i < MAX_PROCESSNUM; i++) {
		_KERN_ptable[i].state = STATE_EMPTY;
		for (j = 0; j < PNREGS; j++)
			_KERN_ptable[i].regs.regs[j] = 0;
		_KERN_ptable[i].stack = (void *) 0;
		_KERN_ptable[i].signal = 0;
		_KERN_ptable[i].ppid = 0;
		_KERN_ptable[i].pid = i;
		LIST_makeentry (&_KERN_ptable[i].me);
		_KERN_ptable[i].me.data = &(_KERN_ptable[i]);
		_KERN_ptable[i].me.len = sizeof (_KERN_ptable[i]);
	}

	_KERN_ptable[0].state = STATE_RUNNING;
	_KERN_ptable[0].prio = 0;
	_LIBC_init_reent (&_KERN_ptable[0].reent);
	_impure_ptr = &(_KERN_ptable[0].reent);		/* initialise libc reentrance */
	return 0;

}

/*!
\brief suspend a thread
\param i is the process id of the thread to be suspended. -1 suspends the current thread.
\ingroup KERN
*/
int KERN_suspend (int i)
{
// suspend sets task i into suspend mode, 
	// if i=-1 it suspends the running task.
	// if the running task was suspended, the scheduler is called

// sanity checks
	if ((i < -1) || (i > MAX_PROCESSNUM)){
		printf("cannot suspend %d\n",i);
		return -1;
	}
	if ((i != -1) && (_KERN_ptable[i].state == STATE_EMPTY)){
		printf("cannot suspend %d: empty\n",i);
		return -1;
	}

	if ((i == -1) || (i == __MYPID)) {
		// we're suspending the running task...

		i = __MYPID;
		// assert( _KERN_ptable[i].state == STATE_RUNNING);
		_KERN_ptable[i].state = STATE_SUSPEND;
		KERN_schedule ();
		// here, the suspended task will continue afer wakeup...
	} else {
		// we're suspending some other task.
		// assert( _KERN_ptable[i].state == STATE_READY);
		disable ();	/* don't confuse the scheduler */
		_KERN_ptable[i].state = STATE_SUSPEND;
		if (_KERN_ptable[i].me.list->type == LIST_TYPE_PRIO)
			LIST_delete (&_KERN_ptable[i].me);
		/* if the process is in one of the priority class lists, remove it */
		/* so that the scheduler does not have to worry. */
		enable ();
	}

	return 0;
}

/*!
\brief wakes up a thread
\param i ist the process id of the thread to be woken up
\ingroup KERN
*/
int KERN_wakeup (int i)
{

// sanity checks
	if (i > MAX_PROCESSNUM)
		return -1;
	if (_KERN_ptable[i].state != STATE_SUSPEND)
		return -1;

	_KERN_ptable[i].state = STATE_READY;
	LIST_delete (&_KERN_ptable[i].me);	/* remove process from any queue */
	LIST_insert_head (&_KERN_prio[_KERN_ptable[i].prio], &_KERN_ptable[i].me);

// KERN_schedule() ???
	return 0;
}

/*!
\brief this is the kernel delta list handler. It is called by the periodic timer interrupe
\ingroup KERN
*/
int KERN_delta_handler ()
{
	int schedflag = 0;

//if (_KERN_deltahead==NO_TASK) return (0); /* Delta list is empty */

	if (LIST_entries (&_KERN_delta) == 0)
		return (0);

	((struct process *) (LIST_head (&_KERN_delta)->data))->time_delta--;

	while ((LIST_entries (&_KERN_delta) > 0) &&
	       (((struct process *) (LIST_head (&_KERN_delta)->data))->time_delta == 0)) {
		KERN_wakeup (((struct process *) (LIST_head (&_KERN_delta))->data)->pid);
		schedflag = 1;	// we want to reschedule

	}

/* old code 
   _KERN_ptable[_KERN_deltahead].time_delta--;

   while ((_KERN_ptable[_KERN_deltahead].time_delta==0) //
   && (_KERN_deltahead!=NO_TASK)) //
   {
   if (KERN_wakeup(_KERN_deltahead))
   {
   //           iTTY_outchar('X');
   } else {
   //           iTTY_outchar('G');
   }
   schedflag=1; // we want to reschedule 
   _KERN_deltahead=_KERN_ptable[_KERN_deltahead].next;
   }
 */


	if (schedflag)
		return 1;

	return 0;		/* nothing new */
}
