/*  src_experimental/kernel/seminl.h
    CubeOS Version 0.4.90 experimental
    Copyright (C) 1999,2000 Holger Kenn

    CubeOS is free software; you can redistribute it and/or
    modify it under the terms of the GNU Library General Public
    License as published by the Free Software Foundation; either
    version 2 of the License, or any later version.

    CubeOS is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Library General Public License for more details.

*/  
#ifndef _SEMINL_H
#define _SEMINL_H

/*! \file seminl.h
\ingroup KERN
CubeOS/POSIX Semaphores: complex semaphores that are thread-aware
*/

#ifdef MULTI_SEM

/*!
\brief contains all data about a semaphore
\ingroup KERN
*/
typedef struct sem_s {
	int init; //!< flag that tells wheter the semaphore has been initialized
	int c;    //!< the semaphore counter
	int task; //!< process ID of a thread blocking on the semaphore
	} sem_t;



/*!
\brief post operation on a semaphore
\param x is a semaphore
\ingroup KERN
\todo Should this be changed into a pointer ?
\todo instead of the simple list, this should also use the LIST component
*/
static inline void sem_post(sem_t x)
{
	if (!x.init) return;
	disable();
	x.c++;
	if (x.task!=NO_TASK)
	{
		wakeup(x.task); // this resets the task state, 
						// but we're still in disable()d state!
 		x.task=_KERN_ptable[x.task].next;
	}
	enable();
}


#define sem_signal(x) sem_post(x) //!< alias to sem_post(), for thomas

/*!
\brief wait operation on a semaphore
\param x is a semaphore
\ingroup KERN
\todo Should this be changed into a pointer ?
\todo instead of the simple list, this should also use the LIST component
*/
static inline void sem_wait(sem_t x)
{
	if (!x.init) return;
	disable();	
	x.c--;
	if (x.c<0){
		// put ourself to sleep
		if (x.task == NO_TASK)
		{
			_KERN_ptable[getpid()].next=NO_TASK;
		} 
		else
		{
			_KERN_ptable[getpid()].next=x.task;
		}
		x.task=getpid();
		enable();
		KERN_suspend(getpid());
	} 
	else 
	{
		enable(); // the semaphore was free
	}
}

#define sem_trywait(x) (x.c<0) //!< checks, if a sem_wait() operation might block.
#define sem_getvalue(x,i) (*i=x.c) //!< returns the semaphore counter

/*!
\brief prepares a semaphore
\param x is a semaphore
\ingroup KERN
*/
static inline int sem_init(sem_t x,int i,int v)
{
	if (x.init) return (-1);
	disable();
	x.c=v;
	x.task=NO_TASK;
	x.init=1;
	enable();
	return (0);
}

/*!
\brief destroys a semaphore and releases all threads waiting on it.
\param x is a semaphore
\ingroup KERN
*/
static inline int sem_destroy(sem_t x)
{
	if (!x.init) return (-1);
	disable();
	while(x.task!=NO_TASK)
		{
			wakeup(x.task); // this resets the task state, 
							// but we're still in disable()d state!
 			x.task=_KERN_ptable[x.task].next;
		}
	enable();
	return(0);
}
#endif // of ifdef MULTI_SEM
#endif // of ifndef _SEMINL_H
