/*
 * Copyright (c) 1982, 1986, 1988, 1990 Regents of the University of California.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *      This product includes software developed by the University of
 *      California, Berkeley and its contributors.
 * 4. Neither the name of the University nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE REGENTS AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 *      @(#)ip_output.c 7.23 (Berkeley) 11/12/90
 */

#include "param.h"
#include "malloc.h"
#include "mbuf.h"
#include "errno.h"
#include "protosw.h"
#include "socket.h"
#include "socketvar.h"

#include "../net/if.h"
#include "../net/route.h"

#include "in.h"
#include "in_systm.h"
#include "ip.h"
#include "in_pcb.h"
#include "in_var.h"
#include "ip_var.h"

#ifdef vax
#include "machine/mtpr.h"
#endif

struct mbuf *ip_insertoptions ();

/*
 * IP output.  The packet in mbuf chain m contains a skeletal IP
 * header (with len, off, ttl, proto, tos, src, dst).
 * The mbuf chain containing the packet will be freed.
 * The mbuf opt, if present, will not be freed.
 */
ip_output (m0, opt, ro, flags)
     struct mbuf *m0;
     struct mbuf *opt;
     struct route *ro;
     int flags;
{
	register struct ip *ip, *mhip;
	register struct ifnet *ifp;
	register struct mbuf *m = m0;
	register int hlen = sizeof (struct ip);
	int len, off, error = 0;
	struct route iproute;
	struct sockaddr_in *dst;
	struct in_ifaddr *ia;

#ifdef	DIAGNOSTIC
	if ((m->m_flags & M_PKTHDR) == 0)
		panic ("ip_output no HDR");
#endif
	if (opt) {
		m = ip_insertoptions (m, opt, &len);
		hlen = len;
	}
	ip = mtod (m, struct ip *);
	/*
	 * Fill in IP header.
	 */
	if ((flags & IP_FORWARDING) == 0) {
		ip->ip_v = IPVERSION;
		ip->ip_off &= IP_DF;
		ip->ip_id = htons (ip_id++);
		ip->ip_hl = hlen >> 2;
	} else {
		hlen = ip->ip_hl << 2;
		ipstat.ips_localout++;
	}
	/*
	 * Route packet.
	 */
	if (ro == 0) {
		ro = &iproute;
		bzero ((caddr_t) ro, sizeof (*ro));
	}
	dst = (struct sockaddr_in *) &ro->ro_dst;
	/*
	 * If there is a cached route,
	 * check that it is to the same destination
	 * and is still up.  If not, free it and try again.
	 */
	if (ro->ro_rt && ((ro->ro_rt->rt_flags & RTF_UP) == 0 ||
			  dst->sin_addr.s_addr != ip->ip_dst.s_addr)) {
		RTFREE (ro->ro_rt);
		ro->ro_rt = (struct rtentry *) 0;
	}
	if (ro->ro_rt == 0) {
		dst->sin_family = AF_INET;
		dst->sin_len = sizeof (*dst);
		dst->sin_addr = ip->ip_dst;
	}
	/*
	 * If routing to interface only,
	 * short circuit routing lookup.
	 */
	if (flags & IP_ROUTETOIF) {

		ia = (struct in_ifaddr *) ifa_ifwithdstaddr ((struct sockaddr *) dst);
		if (ia == 0)
			ia = in_iaonnetof (in_netof (ip->ip_dst));
		if (ia == 0) {
			error = ENETUNREACH;
			goto bad;
		}
		ifp = ia->ia_ifp;
	} else {
		if (ro->ro_rt == 0)
			rtalloc (ro);
		if (ro->ro_rt == 0) {
			error = EHOSTUNREACH;
			goto bad;
		}
		ia = (struct in_ifaddr *) ro->ro_rt->rt_ifa;
		ifp = ro->ro_rt->rt_ifp;
		ro->ro_rt->rt_use++;
		if (ro->ro_rt->rt_flags & RTF_GATEWAY)
			dst = (struct sockaddr_in *) ro->ro_rt->rt_gateway;
	}
#ifndef notdef
	/*
	 * If source address not specified yet, use address
	 * of outgoing interface.
	 */
	if (ip->ip_src.s_addr == INADDR_ANY)
		ip->ip_src = IA_SIN (ia)->sin_addr;
#endif
	/*
	 * Look for broadcast address and
	 * and verify user is allowed to send
	 * such a packet.
	 */
	if (in_broadcast (dst->sin_addr)) {
		if ((ifp->if_flags & IFF_BROADCAST) == 0) {
			error = EADDRNOTAVAIL;
			goto bad;
		}
		if ((flags & IP_ALLOWBROADCAST) == 0) {
			error = EACCES;
			goto bad;
		}
		/* don't allow broadcast messages to be fragmented */
		if ((u_short) ip->ip_len > ifp->if_mtu) {
			error = EMSGSIZE;
			goto bad;
		}
		m->m_flags |= M_BCAST;
	}
	/*
	 * If small enough for interface, can just send directly.
	 */
	if ((u_short) ip->ip_len <= ifp->if_mtu) {
		ip->ip_len = htons ((u_short) ip->ip_len);
		ip->ip_off = htons ((u_short) ip->ip_off);
		ip->ip_sum = 0;
		ip->ip_sum = in_cksum (m, hlen);
		error = (*ifp->if_output) (ifp, m,
					(struct sockaddr *) dst, ro->ro_rt);
		goto done;
	}
	ipstat.ips_fragmented++;
	/*
	 * Too large for interface; fragment if possible.
	 * Must be able to put at least 8 bytes per fragment.
	 */
	if (ip->ip_off & IP_DF) {
		error = EMSGSIZE;
		goto bad;
	}
	len = (ifp->if_mtu - hlen) & ~7;
	if (len < 8) {
		error = EMSGSIZE;
		goto bad;
	} {
		int mhlen, firstlen = len;
		struct mbuf **mnext = &m->m_nextpkt;

		/*
		 * Loop through length of segment after first fragment,
		 * make new header and copy data of each part and link onto chain.
		 */
		m0 = m;
		mhlen = sizeof (struct ip);
		for (off = hlen + len; off < (u_short) ip->ip_len; off += len) {
			MGETHDR (m, M_DONTWAIT, MT_HEADER);
			if (m == 0) {
				error = ENOBUFS;
				goto sendorfree;
			}
			m->m_data += max_linkhdr;
			mhip = mtod (m, struct ip *);
			*mhip = *ip;
			if (hlen > sizeof (struct ip)) {
				mhlen = ip_optcopy (ip, mhip) + sizeof (struct ip);
				mhip->ip_hl = mhlen >> 2;
			}
			m->m_len = mhlen;
			mhip->ip_off = ((off - hlen) >> 3) + (ip->ip_off & ~IP_MF);
			if (ip->ip_off & IP_MF)
				mhip->ip_off |= IP_MF;
			if (off + len >= (u_short) ip->ip_len)
				len = (u_short) ip->ip_len - off;
			else
				mhip->ip_off |= IP_MF;
			mhip->ip_len = htons ((u_short) (len + mhlen));
			m->m_next = m_copy (m0, off, len);
			if (m->m_next == 0) {
				error = ENOBUFS;	/* ??? */
				goto sendorfree;
			}
			m->m_pkthdr.len = mhlen + len;
			m->m_pkthdr.rcvif = (struct ifnet *) 0;
			mhip->ip_off = htons ((u_short) mhip->ip_off);
			mhip->ip_sum = 0;
			mhip->ip_sum = in_cksum (m, mhlen);
			*mnext = m;
			mnext = &m->m_nextpkt;
			ipstat.ips_ofragments++;
		}
		/*
		 * Update first fragment by trimming what's been copied out
		 * and updating header, then send each fragment (in order).
		 */
		m = m0;
		m_adj (m, hlen + firstlen - (u_short) ip->ip_len);
		m->m_pkthdr.len = hlen + firstlen;
		ip->ip_len = htons ((u_short) m->m_pkthdr.len);
		ip->ip_off = htons ((u_short) (ip->ip_off | IP_MF));
		ip->ip_sum = 0;
		ip->ip_sum = in_cksum (m, hlen);
	      sendorfree:
		for (m = m0; m; m = m0) {
			m0 = m->m_nextpkt;
			m->m_nextpkt = 0;
			if (error == 0)
				error = (*ifp->if_output) (ifp, m,
					(struct sockaddr *) dst, ro->ro_rt);
			else
				m_freem (m);
		}
	}
      done:
	if (ro == &iproute && (flags & IP_ROUTETOIF) == 0 && ro->ro_rt)
		RTFREE (ro->ro_rt);
	return (error);
      bad:
	m_freem (m0);
	goto done;
}

/*
 * Insert IP options into preformed packet.
 * Adjust IP destination as required for IP source routing,
 * as indicated by a non-zero in_addr at the start of the options.
 */
struct mbuf *
  ip_insertoptions (m, opt, phlen)
     register struct mbuf *m;
     struct mbuf *opt;
     int *phlen;
{
	register struct ipoption *p = mtod (opt, struct ipoption *);
	struct mbuf *n;
	register struct ip *ip = mtod (m, struct ip *);
	unsigned optlen;

	optlen = opt->m_len - sizeof (p->ipopt_dst);
	if (optlen + (u_short) ip->ip_len > IP_MAXPACKET)
		return (m);	/* XXX should fail */
	if (p->ipopt_dst.s_addr)
		ip->ip_dst = p->ipopt_dst;
	if (m->m_flags & M_EXT || m->m_data - optlen < m->m_pktdat) {
		MGETHDR (n, M_DONTWAIT, MT_HEADER);
		if (n == 0)
			return (m);
		n->m_pkthdr.len = m->m_pkthdr.len + optlen;
		m->m_len -= sizeof (struct ip);
		m->m_data += sizeof (struct ip);
		n->m_next = m;
		m = n;
		m->m_len = optlen + sizeof (struct ip);
		m->m_data += max_linkhdr;
		bcopy ((caddr_t) ip, mtod (m, caddr_t), sizeof (struct ip));
	} else {
		m->m_data -= optlen;
		m->m_len += optlen;
		m->m_pkthdr.len += optlen;
		ovbcopy ((caddr_t) ip, mtod (m, caddr_t), sizeof (struct ip));
	}
	ip = mtod (m, struct ip *);
	bcopy ((caddr_t) p->ipopt_list, (caddr_t) (ip + 1), (unsigned) optlen);
	*phlen = sizeof (struct ip) + optlen;
	ip->ip_len += optlen;
	return (m);
}

/*
 * Copy options from ip to jp,
 * omitting those not copied during fragmentation.
 */
ip_optcopy (ip, jp)
     struct ip *ip, *jp;
{
	register u_char *cp, *dp;
	int opt, optlen, cnt;

	cp = (u_char *) (ip + 1);
	dp = (u_char *) (jp + 1);
	cnt = (ip->ip_hl << 2) - sizeof (struct ip);
	for (; cnt > 0; cnt -= optlen, cp += optlen) {
		opt = cp[0];
		if (opt == IPOPT_EOL)
			break;
		if (opt == IPOPT_NOP)
			optlen = 1;
		else
			optlen = cp[IPOPT_OLEN];
		/* bogus lengths should have been caught by ip_dooptions */
		if (optlen > cnt)
			optlen = cnt;
		if (IPOPT_COPIED (opt)) {
			bcopy ((caddr_t) cp, (caddr_t) dp, (unsigned) optlen);
			dp += optlen;
		}
	}
	for (optlen = dp - (u_char *) (jp + 1); optlen & 0x3; optlen++)
		*dp++ = IPOPT_EOL;
	return (optlen);
}

/*
 * IP socket option processing.
 */
ip_ctloutput (op, so, level, optname, mp)
     int op;
     struct socket *so;
     int level, optname;
     struct mbuf **mp;
{
	register struct inpcb *inp = sotoinpcb (so);
	register struct mbuf *m = *mp;
	register int optval;
	int error = 0;

	if (level != IPPROTO_IP)
		error = EINVAL;
	else
		switch (op) {

		case PRCO_SETOPT:
			switch (optname) {
			case IP_OPTIONS:
#ifdef notyet
			case IP_RETOPTS:
				return (ip_pcbopts (optname, &inp->inp_options, m));
#else
				return (ip_pcbopts (&inp->inp_options, m));
#endif

			case IP_TOS:
			case IP_TTL:
			case IP_RECVOPTS:
			case IP_RECVRETOPTS:
			case IP_RECVDSTADDR:
				if (m->m_len != sizeof (int))
					  error = EINVAL;
				else {
					optval = *mtod (m, int *);
					switch (optname) {

					case IP_TOS:
						inp->inp_ip.ip_tos = optval;
						break;

					case IP_TTL:
						inp->inp_ip.ip_ttl = optval;
						break;
#define	OPTSET(bit) \
	if (optval) \
		inp->inp_flags |= bit; \
	else \
		inp->inp_flags &= ~bit;

					case IP_RECVOPTS:
						OPTSET (INP_RECVOPTS);
						break;

					case IP_RECVRETOPTS:
						OPTSET (INP_RECVRETOPTS);
						break;

					case IP_RECVDSTADDR:
						OPTSET (INP_RECVDSTADDR);
						break;
					}
				}
				break;
#undef OPTSET

			default:
				error = EINVAL;
				break;
			}
			if (m)
				(void) m_free (m);
			break;

		case PRCO_GETOPT:
			switch (optname) {
			case IP_OPTIONS:
			case IP_RETOPTS:
				*mp = m = m_get (M_WAIT, MT_SOOPTS);
				if (inp->inp_options) {
					m->m_len = inp->inp_options->m_len;
					bcopy (mtod (inp->inp_options, caddr_t),
					       mtod (m, caddr_t), (unsigned) m->m_len);
				} else
					m->m_len = 0;
				break;

			case IP_TOS:
			case IP_TTL:
			case IP_RECVOPTS:
			case IP_RECVRETOPTS:
			case IP_RECVDSTADDR:
				*mp = m = m_get (M_WAIT, MT_SOOPTS);
				m->m_len = sizeof (int);
				switch (optname) {

				case IP_TOS:
					optval = inp->inp_ip.ip_tos;
					break;

				case IP_TTL:
					optval = inp->inp_ip.ip_ttl;
					break;

#define	OPTBIT(bit)	(inp->inp_flags & bit ? 1 : 0)

				case IP_RECVOPTS:
					optval = OPTBIT (INP_RECVOPTS);
					break;

				case IP_RECVRETOPTS:
					optval = OPTBIT (INP_RECVRETOPTS);
					break;

				case IP_RECVDSTADDR:
					optval = OPTBIT (INP_RECVDSTADDR);
					break;
				}
				*mtod (m, int *) = optval;
				break;

			default:
				error = EINVAL;
				break;
			}
			break;
		}
	return (error);
}

/*
 * Set up IP options in pcb for insertion in output packets.
 * Store in mbuf with pointer in pcbopt, adding pseudo-option
 * with destination address if source routed.
 */
#ifdef notyet
ip_pcbopts (optname, pcbopt, m)
     int optname;
#else
ip_pcbopts (pcbopt, m)
#endif
     struct mbuf **pcbopt;
     register struct mbuf *m;
{
	register cnt, optlen;
	register u_char *cp;
	u_char opt;

	/* turn off any old options */
	if (*pcbopt)
		(void) m_free (*pcbopt);
	*pcbopt = 0;
	if (m == (struct mbuf *) 0 || m->m_len == 0) {
		/*
		 * Only turning off any previous options.
		 */
		if (m)
			(void) m_free (m);
		return (0);
	}
#ifndef	vax
	if (m->m_len % sizeof (long))
		  goto bad;
#endif
	/*
	 * IP first-hop destination address will be stored before
	 * actual options; move other options back
	 * and clear it when none present.
	 */
	if (m->m_data + m->m_len + sizeof (struct in_addr) >= &m->m_dat[MLEN])
		  goto bad;
	cnt = m->m_len;
	m->m_len += sizeof (struct in_addr);
	cp = mtod (m, u_char *) + sizeof (struct in_addr);
	ovbcopy (mtod (m, caddr_t), (caddr_t) cp, (unsigned) cnt);
	bzero (mtod (m, caddr_t), sizeof (struct in_addr));

	for (; cnt > 0; cnt -= optlen, cp += optlen) {
		opt = cp[IPOPT_OPTVAL];
		if (opt == IPOPT_EOL)
			break;
		if (opt == IPOPT_NOP)
			optlen = 1;
		else {
			optlen = cp[IPOPT_OLEN];
			if (optlen <= IPOPT_OLEN || optlen > cnt)
				goto bad;
		}
		switch (opt) {

		default:
			break;

		case IPOPT_LSRR:
		case IPOPT_SSRR:
			/*
			 * user process specifies route as:
			 *      ->A->B->C->D
			 * D must be our final destination (but we can't
			 * check that since we may not have connected yet).
			 * A is first hop destination, which doesn't appear in
			 * actual IP option, but is stored before the options.
			 */
			if (optlen < IPOPT_MINOFF - 1 + sizeof (struct in_addr))
				  goto bad;
			m->m_len -= sizeof (struct in_addr);
			cnt -= sizeof (struct in_addr);
			optlen -= sizeof (struct in_addr);
			cp[IPOPT_OLEN] = optlen;
			/*
			 * Move first hop before start of options.
			 */
			bcopy ((caddr_t) & cp[IPOPT_OFFSET + 1], mtod (m, caddr_t),
			       sizeof (struct in_addr));
			/*
			 * Then copy rest of options back
			 * to close up the deleted entry.
			 */
			ovbcopy ((caddr_t) (&cp[IPOPT_OFFSET + 1] +
					    sizeof (struct in_addr)),
				   (caddr_t) & cp[IPOPT_OFFSET + 1],
				   (unsigned) cnt + sizeof (struct in_addr));
			break;
		}
	}
	if (m->m_len > MAX_IPOPTLEN + sizeof (struct in_addr))
		  goto bad;
	*pcbopt = m;
	return (0);

      bad:
	(void) m_free (m);
	return (EINVAL);
}
