/***********************************************************/
/* (c) Peter Stuer 1996                                    */
/*                                                         */
/* AI-LAB Vrije Universiteit Brussel                       */
/*                                                         */
/* Based on work by Filip Vertommen                        */
/*                                                         */
/***********************************************************/
/*                                                         */
/* Smb2kern.c vers 1.0                                     */
/*                                                         */
/***********************************************************/


#include <stdarg.h>
#include "smb2kern.h"


/* The type that defines the SMBII commands. */
typedef enum {SMB2_CM_DEF_DEVICE=0,
							SMB2_CM_EXPORT_DATA_TO_DEV=1,
							SMB2_CM_WRITE_DATA_TO_DEV=2,
							SMB2_CM_WRITE_PART_DATA=3,
							SMB2_CM_WRITE_DATA=4,
							SMB2_CM_UNDO_EXPORT=5,
							SMB2_CM_KERNEL_INT_RATE=6,
							SMB2_CM_MOTORS_DC=7,
							SMB2_CM_MOTORS_PWM=8,
							SMB2_CM_IR_CARRIER=9,
							SMB2_CM_IR_CARRIER_DC=10,
							SMB2_CM_IR_MOD=11,
							SMB2_CM_IR_MOD_DC=12,
							SMB2_CM_SOUND=13,
							SMB2_CM_SOUND_CARRIER_DC=14,
							SMB2_CM_SOUND_MOD=15,
							SMB2_CM_SOUND_MOD_DC=16,
							SMB2_CM_FILTER=17,
							SMB2_CM_FILTER_DC=18,
              SMB2_CM_RADIOMODE=19} SMB2_Command;


/* The interface function to the SMB2-kernel.  */
void (*AddToCommandQueue) (byte *data, byte dLength);

/* Store information on the defined devices to reduce run-time errors.*/

typedef struct	{	logic installed;
									byte txNmOfParam;}	SMB2DevInfoStruct;
									
SMB2DevInfoStruct SMB2DevInfo[MAX_DEVICES];

/*-------------------------------------------------------------------------*/
/*                           THE FUNCTIONS                                 */
/*-------------------------------------------------------------------------*/

/* The internal help functions*/

void InitDevInfo (void)
{
	int i;
	
	for (i=0;i<MAX_DEVICES;i++)
	{
		SMB2DevInfo[i].installed=false;
		SMB2DevInfo[i].txNmOfParam=0;
	}
}

/* The SMB2-API */

/* Command 0 */

logic SMB2DefineDevice (byte DevID,
												SMB2_Dev_Type DevType,
												SMB2_Dev_Addr DevAddr,
												byte RefreshInterval,
												byte SyncGroup,
												byte RXBaseAddr,
												byte RXNmOfParam,
												byte RXFlags,
												byte RXOffset,
												byte RXInitByte,
												byte TXBaseAddr,
												byte TXNmOfParam,
												byte TXFlags,
												byte TXOffset,
												byte TXInitByte,
												SMB2_Channel Channel)
{
	static int amountOfDevices=0;
	byte buffer[17];
	int cellNm;
	int d;

	if (amountOfDevices == MAX_DEVICES)
	{
		printf("SMB2DefineDevice: exceeded MAX_DEVICES\n\r");
		return false;
	}
	
	buffer[0]=(byte)SMB2_CM_DEF_DEVICE;
	buffer[1]=(byte)DevID;
	buffer[2]=(byte)DevType;
	buffer[3]=(byte)DevAddr;
	buffer[4]=RefreshInterval;
	buffer[5]=SyncGroup;
	buffer[6]=RXBaseAddr;
	buffer[7]=RXNmOfParam;
	buffer[8]=RXFlags;
	buffer[9]=RXOffset;
	buffer[10]=RXInitByte;
	buffer[11]=TXBaseAddr;
	buffer[12]=TXNmOfParam;
	buffer[13]=TXFlags;
	buffer[14]=TXOffset;
	buffer[15]=TXInitByte;
	buffer[16]=(byte)Channel;

	AddToCommandQueue(buffer,17);
	SMB2DevInfo[(int)DevID].installed=true;  /*store information*/
	SMB2DevInfo[(int)DevID].txNmOfParam=TXNmOfParam;
	amountOfDevices++;

	/*printf("SMB2DefineDevice: defined %i devices\n\r",amountOfDevices);
	for(d=1;d<=16;d++)
	{
		printf("%i ",(int)buffer[d]);
	}
	printf("\n\r");*/ 
	 
	return true;
}

/* Command 1*/
logic SMB2ExportData(byte DataBaseAddr,byte ToDevID,byte Margin)
{
	byte buffer[4];

	if (!SMB2DevInfo[(int)ToDevID].installed)     /*not recognised device*/
	{
		return false;
	}

	buffer[0]=(byte)SMB2_CM_EXPORT_DATA_TO_DEV;
	buffer[1]=DataBaseAddr;
	buffer[2]=(byte)ToDevID;
	buffer[3]=Margin;

	AddToCommandQueue(buffer,4);
	/*printf("SMB2ExportData %i %i %i\n\r",(int)DataBaseAddr, (int) ToDevID, (int)Margin);*/ 
	return true;
}

/* Command 2*/
logic SMB2WriteDataToDev (byte DevID,int DataCount,...)
{
	int i;
	byte buffer[256];
	va_list ap;
	va_start(ap,DataCount);

	if ((!SMB2DevInfo[(int)DevID].installed)      ||	     /*not recognised device*/
			(SMB2DevInfo[(int)DevID].txNmOfParam <=0) ||	     /*no transmit param possible*/
			(SMB2DevInfo[(int)DevID].txNmOfParam != DataCount)) /*transmit param do not fit*/
	{
		va_end(ap);
		return false;
	}

	buffer[0]=(byte)SMB2_CM_WRITE_DATA_TO_DEV;
	buffer[1]=(byte)DevID;
	for (i=2;i<DataCount+2;i++)
	{
		buffer[i]=(byte)va_arg(ap,unsigned int);
	}

	AddToCommandQueue(buffer,DataCount+2);
	va_end(ap);
	return true;
}

/*Command 3 */
logic SMB2WritePartDataToDev (byte DevID,int DataCount,...)
{

	va_list ap;
	
	va_start(ap, DataCount);
/* he kernel still contains bugson this command so implementation is not finalised */
	va_end(ap);
	return false;
}

/* Command 4*/

logic SMB2WriteData(int NumberOfBytesAfterThis,byte StartAddress,...)

{
	byte buffer[4];
	int i;
	va_list ap;

	va_start(ap,StartAddress);

	buffer[0] = (byte)SMB2_CM_WRITE_DATA;
	buffer[1] = NumberOfBytesAfterThis;
	buffer[2] = StartAddress;
	for(i=3; i < (NumberOfBytesAfterThis + 2); i++)
	{
		buffer[i] = (byte)va_arg(ap, unsigned int); /*according to BC 4 cannot use unsigned char (byte)*/
	}

	AddToCommandQueue(buffer, NumberOfBytesAfterThis + 2);
	va_end(ap);

	return true;
}

/* Command 5*/

logic SMB2UndoExport(byte FromDataNumber,byte ToDevID)
{
	byte buffer[3];

	buffer[0]=(byte)SMB2_CM_UNDO_EXPORT;
	buffer[1]=FromDataNumber;
	buffer[2]=ToDevID;

	AddToCommandQueue(buffer,3);
	return true;
}

/* Command 6*/

logic SMB2KernelIntRate(byte Msbyte,byte Lsbyte,byte Divider)
{
	byte buffer[4];

	buffer[0]=(byte)SMB2_CM_KERNEL_INT_RATE;
	buffer[1]=Msbyte;
	buffer[2]=Lsbyte;
	buffer[3]=Divider;

	AddToCommandQueue(buffer,4);
	return true;
}

/* Command 7*/

logic SMB2MotorsDc(byte MotorMask,byte DcPercentage)
{
	byte buffer[3];

	buffer[0]=(byte)SMB2_CM_MOTORS_DC;
	buffer[1]=MotorMask;
	buffer[2]=DcPercentage;

	AddToCommandQueue(buffer,3);
	return true;
}

/* Command 8*/

logic SMB2MotorsPwm(byte Msbyte,byte Lsbyte)
{
	byte buffer[3];

	buffer[0]=(byte)SMB2_CM_MOTORS_PWM;
	buffer[1]=Msbyte;
	buffer[2]=Lsbyte;

	AddToCommandQueue(buffer,3);
	return true;
}

/* Command 9*/

logic SMB2IRCarrier(byte Msbyte,byte Lsbyte)
{
	byte buffer[3];

	buffer[0]=(byte)SMB2_CM_IR_CARRIER;
	buffer[1]=Msbyte;
	buffer[2]=Lsbyte;

	AddToCommandQueue(buffer,3);
	return true;
}

/* Command 10*/

logic SMB2IRCarrierDc(byte Percentage)
{
	byte buffer[2];

	buffer[0]=(byte)SMB2_CM_IR_CARRIER_DC;
	buffer[1]=Percentage;

	AddToCommandQueue(buffer,2);
	return true;
}

/* Command 11*/
logic SMB2IRMod(byte Msbyte,byte Lsbyte)
{
	byte buffer[3];

	buffer[0]=(byte)SMB2_CM_IR_MOD;
	buffer[1]=Msbyte;
	buffer[2]=Lsbyte;

	AddToCommandQueue(buffer,3);
	return true;
}

/* Command 12*/

logic SMB2IRModDc(byte Percentage)
{
	byte buffer[2];

	buffer[0]=(byte)SMB2_CM_IR_MOD_DC;
	buffer[1]=Percentage;

	AddToCommandQueue(buffer,2);
	return true;
}

/* Command 13*/
logic SMB2Sound(byte Msbyte,byte Lsbyte)
{
	byte buffer[3];

	buffer[0]=(byte)SMB2_CM_SOUND;
	buffer[1]=Msbyte;
	buffer[2]=Lsbyte;

	AddToCommandQueue(buffer,3);
	return true;
}

/* Command 14*/

logic SMB2SoundCarrierDc(byte Percentage)
{
	byte buffer[2];

	buffer[0]=(byte)SMB2_CM_SOUND_CARRIER_DC;
	buffer[1]=Percentage;

	AddToCommandQueue(buffer,2);
	return true;
}

/* Command 15*/

logic SMB2SoundMod(byte Msbyte,byte Lsbyte)
{
	byte buffer[3];

	buffer[0]=(byte)SMB2_CM_SOUND_MOD;
	buffer[1]=Msbyte;
	buffer[2]=Lsbyte;

	AddToCommandQueue(buffer,3);
	return true;
}

/* Command 16*/

logic SMB2SoundModDc(byte Percentage)
{
	byte buffer[2];

	buffer[0]=(byte)SMB2_CM_SOUND_MOD_DC;
	buffer[1]=Percentage;

	AddToCommandQueue(buffer,2);
	return true;
}

/* Command 17*/

logic SMB2Filter(byte Msbyte,byte Lsbyte)
{
	byte buffer[3];

	buffer[0]=(byte)SMB2_CM_FILTER;
	buffer[1]=Msbyte;
	buffer[2]=Lsbyte;

	AddToCommandQueue(buffer,3);
	return true;
}

/* Command 18*/
logic SMB2FilterDc(byte Percentage)
{
	byte buffer[2];

	buffer[0]=(byte)SMB2_CM_FILTER_DC;
	buffer[1]=Percentage;

	AddToCommandQueue(buffer,2);
	return true;
}

/* Command 19*/
logic SMB2SetRadioMode(byte Mode)
{
  byte buffer[2];

	buffer[0]=(byte)SMB2_CM_RADIOMODE;
	buffer[1]=Mode;

	AddToCommandQueue(buffer,2);
	return true;
}

logic SMB2Init (void)
{
	static logic Initialised=false;

	if (!Initialised)
	{
		Initialised = true;
	}
	else 
	{
		return false;
	}

	AddToCommandQueue=((void *)(*((unsigned long*)0x514)));
	InitDevInfo();
	return true;
}

