/*  src_experimental/drivers/i2c/analog.c
   CubeOS Version 0.4.90 experimental
   Copyright (C) 1999,2000 Holger Kenn

   CubeOS is free software; you can redistribute it and/or
   modify it under the terms of the GNU Library General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or any later version.

   CubeOS is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Library General Public License for more details.

 */
#include <stdlib.h>
#include <stdio.h>
#include <analog.h>
#include <i2cd.h>

/*! \file analog.c
\ingroup I2C
*/

/*!
\brief keeps information about the PCF8591 A/D - D/A converters in the system
\ingroup I2C
*/
struct _I2C_ad_datas _I2C_ad_data[16];
int _I2C_adnum;

char _I2C_analog_mbuf[MAXI2CMESSLENGTH];
struct i2cmess _I2C_analog_m;

/*!
\brief read one A/D input channel of a PCF8591 device
\param chip is the chip ID as set by init_analog()
\param channel is the A/D channel to read 
*/
int I2C_ReadAnalogIn (int chip, int channel)
{

	if (!_I2C_ad_data[chip].active)
		return (0);	/* inactive a/d are 0 */

/* set a/d channel */
	_I2C_analog_m.address = _I2C_ad_data[chip].address;
	_I2C_analog_m.nrBytes = 1;
	_I2C_analog_m.buf = _I2C_analog_mbuf;
	_I2C_analog_mbuf[0] = (_I2C_ad_data[chip].status & 0x70) | (channel & 0x03);	/* no autoincrement */
	I2C_process (_I2C_ad_data[chip].bus, I2C_MASTER, &_I2C_analog_m);

/* read value */
	_I2C_analog_m.address = _I2C_ad_data[chip].address | 0x1;	/* Read adress */
	_I2C_analog_m.nrBytes = 2;	/* Read one false byte, then read value */
	_I2C_analog_mbuf[0] = 0;	/* clear buffer */
	_I2C_analog_mbuf[1] = 0;
	_I2C_analog_m.buf = _I2C_analog_mbuf;


	I2C_process (_I2C_ad_data[chip].bus, I2C_MASTER, &_I2C_analog_m);

	return ((int) (((unsigned int) _I2C_analog_mbuf[1]) & 0xff));

}

/*!
\brief configure the a/d function of one PCF8591 chip
\param chip is the chip ID as set by init_analog()
\param how is the config word
\ingroup I2C

This function allows the configuration of the PCF8591 chip as
differential or single-ended A/D. For more information,
check the datasheet.
*/
int I2C_ConfigureAnalog (int chip, char how)
{

	if (!_I2C_ad_data[chip].active)
		return (-1);

	_I2C_ad_data[chip].status = how & 0x70;

/* set status */
	_I2C_analog_m.address = _I2C_ad_data[chip].address;
	_I2C_analog_m.nrBytes = 1;
	_I2C_analog_m.buf = _I2C_analog_mbuf;
	_I2C_analog_mbuf[0] = (_I2C_ad_data[chip].status);
	I2C_process (_I2C_ad_data[chip].bus, I2C_MASTER, &_I2C_analog_m);

	return (_I2C_analog_m.status == I2C_OK);
}

/*!
\brief sets the D/A output channel of a PCF8591 chip
\param chip is the chip ID as set by init_analog()
\param value is the output value
\ingroup I2C
*/
int I2C_WriteAnalogOut (int chip, int value)
{

	if (!_I2C_ad_data[chip].active)
		return (-1);


/* set status */
	_I2C_analog_m.address = _I2C_ad_data[chip].address;
	_I2C_analog_m.nrBytes = 2;
	_I2C_analog_m.buf = _I2C_analog_mbuf;
	_I2C_analog_mbuf[0] = (_I2C_ad_data[chip].status);
	_I2C_analog_mbuf[1] = value & 0xff;
	I2C_process (_I2C_ad_data[chip].bus, I2C_MASTER, &_I2C_analog_m);

	return (_I2C_analog_m.status == I2C_OK);
}


/*!
\brief configure all PCF8591 A/D-D/A devices
\ingroup I2C

This function looks for PCF8591 devices on all I2C channels, 
configures the devices as 4 times single-ended and lists
all of them in an internal table. The devices get increasing device
IDs from 0 to 15 depending on their bus address.
*/
int I2C_init_analog ()
{
	int i;
	_I2C_adnum = 0;

	for (i = 0; i < 15; i++)
		_I2C_ad_data[i].active = 0;

// later this should come from the HWinfo struct 
	for (i = 0; i < 15; i++) {

		if (i < 8) {
			_I2C_ad_data[_I2C_adnum].bus = I2CA;
		} else {
			_I2C_ad_data[_I2C_adnum].bus = I2CB;
		}

		_I2C_ad_data[_I2C_adnum].address = ((i & 0x07) << 1) | 0x90;

		_I2C_ad_data[_I2C_adnum].status = 0x00;		// 4x single ended, analog output off 

		// this is the power-on reset state

		_I2C_analog_m.address = _I2C_ad_data[_I2C_adnum].address | 0x1;		/* Read adress */
		_I2C_analog_m.nrBytes = 1;	/* Read one false byte, then read value */
		_I2C_analog_mbuf[0] = 0;	/* clear buffer */
		_I2C_analog_m.buf = _I2C_analog_mbuf;
		I2C_process (_I2C_ad_data[_I2C_adnum].bus, I2C_MASTER, &_I2C_analog_m);

		switch (_I2C_analog_m.status) {
		case I2C_OK:
			printf ("analog device %d at adress 0x%x \n", _I2C_adnum, _I2C_ad_data[_I2C_adnum].address);
			_I2C_ad_data[_I2C_adnum].active = 1;
			_I2C_adnum++;
			break;
		case I2C_NO_BUS:
		case I2C_TIME_OUT:
		case I2C_NACK_ON_ADDRESS:
			break;
		default:
			I2C_messagestatus (&_I2C_analog_m);
		}


	}

	return (_I2C_adnum > 0 ? 0 : 1);

}
