/*  src_experimental/drivers/i2c/digital.c
   CubeOS Version 0.4.90 experimental
   Copyright (C) 1999,2000 Holger Kenn

   CubeOS is free software; you can redistribute it and/or
   modify it under the terms of the GNU Library General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or any later version.

   CubeOS is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Library General Public License for more details.

 */
#include <stdlib.h>
#include <stdio.h>
#include <digital.h>
#include <i2cd.h>

/*! \file digital.c
\ingroup I2C
*/

/*!
\brief keeps the internal data for all PCF8574(A) bin i/o devices
\ingroup I2C
*/
struct _I2C_bin_datas _I2C_bin_data[32];
int _I2C_binnum;


unsigned char _I2C_digital_mbuf[MAXI2CMESSLENGTH];
struct i2cmess _I2C_digital_m;

/*!
\brief read the binary input on a PCF8574(A) device
\param chip denotes the chip ID as set by init_digital()
\ingroup I2C
*/
unsigned char I2C_ReadBinIn (int chip)
{

	if (_I2C_bin_data[chip].active == 0)
		return (0);

/* read value */
	_I2C_digital_m.address = _I2C_bin_data[chip].address | 0x1;	/* Read adress */
	_I2C_digital_m.nrBytes = 1;	/* Read one byte */
	_I2C_digital_mbuf[0] = 0;	/* clear buffer */
	_I2C_digital_m.buf = _I2C_digital_mbuf;


	I2C_process (_I2C_bin_data[chip].bus, I2C_MASTER, &_I2C_digital_m);

	return _I2C_digital_mbuf[0];

}

int I2C_WriteBinOut (int chip, unsigned char value)
{

	if (_I2C_bin_data[chip].active == 0)
		return (-1);

/* read value */
	_I2C_digital_m.address = _I2C_bin_data[chip].address;	/* Write adress */
	_I2C_digital_m.nrBytes = 1;	/* Write one byte */
	_I2C_digital_mbuf[0] = value;
	_I2C_digital_m.buf = _I2C_digital_mbuf;

	I2C_process (_I2C_bin_data[chip].bus, I2C_MASTER, &_I2C_digital_m);

	return (_I2C_digital_m.status == I2C_OK);

}

/*!
\brief initializes all PCF8574 devices in the system
\ingroup I2C

This function scanns all I2C busses for PCF8574 and PCF8574A devices.
It first looks for PCF8574 (device address 0x4x), then for PCF8574A
(device address 0x7x) on the first, then on the second I2C bus.
Devices found get assigned an ID from 0 to 31.
*/
int I2C_init_digital ()
{
	int i;

	_I2C_binnum = 0;
	for (i = 0; i < 31; i++)
		_I2C_bin_data[i].active = 0;

// later this should come from the HWinfo struct 
	for (i = 0; i < 31; i++) {

		if (i < 16) {
			_I2C_bin_data[_I2C_binnum].bus = I2CA;
		} else {
			_I2C_bin_data[_I2C_binnum].bus = I2CB;
		}

		if ((i & 0xf) < 8) {
			_I2C_bin_data[_I2C_binnum].address = ((i & 0x07) << 1) | 0x40;	// PCF8574

		} else {
			_I2C_bin_data[_I2C_binnum].address = ((i & 0x07) << 1) | 0x70;	// PCF8574A

		}
		_I2C_bin_data[_I2C_binnum].status = 0x00;	// maybe later...



		_I2C_digital_m.address = _I2C_bin_data[_I2C_binnum].address | 0x1;	/* Read adress */
		_I2C_digital_m.nrBytes = 1;	/* Read one false byte, then read value */
		_I2C_digital_mbuf[0] = 0;	/* clear buffer */
		_I2C_digital_m.buf = _I2C_digital_mbuf;
		I2C_process (_I2C_bin_data[_I2C_binnum].bus, I2C_MASTER, &_I2C_digital_m);


		switch (_I2C_digital_m.status) {
		case I2C_OK:
			printf ("digital device %d at adress 0x%x \n", _I2C_binnum, _I2C_bin_data[_I2C_binnum].address);
			_I2C_bin_data[_I2C_binnum].active = 1;
			_I2C_binnum++;
			break;
		case I2C_TIME_OUT:
		case I2C_NO_BUS:
		case I2C_NACK_ON_ADDRESS:
			break;

		default:
			I2C_messagestatus (&_I2C_digital_m);
		}


	}

	return (_I2C_binnum > 0 ? 0 : 1);


}
