/*  src_experimental/kernel/ivtab.c
   CubeOS Version 0.4.90 experimental
   Copyright (C) 1999,2000 Holger Kenn

   CubeOS is free software; you can redistribute it and/or
   modify it under the terms of the GNU Library General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or any later version.

   CubeOS is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Library General Public License for more details.

 */
#include <cubeos.h>
#include <ivtab.h>
/*! \file ivtab.c
\ingroup KERN
*/

void (*(*_KERN_IVTab)[512]) (void);	/*!< Kernel interrupt vector table */

void *_KERN_IVTab_oldvector[512];	/*!< table for saving old vector settings */

/*! \fn void _KERN_IVTab_init()
   \brief initializes the interrupt vector table
\ingroup KERN
   Reads out the VBR register of the CPU and stores it in the
   _KERN_IVTab pointer. Resets the old vectors to zero. _KERN_IVTab_init() is
   called by crt0.S during initialisation.
 */
void _KERN_IVTab_init ()
{

	int i;

	asm ("movec %vbr,%d0");
      asm ("movel %%d0,%0":"=m" (_KERN_IVTab));

	for (i = 0; i < 512; i++)
		_KERN_IVTab_oldvector[i] = (void *) 0;

}

/*! \fn int _KERN_IVTab_setvector(unsigned short vector, void (*isr) (void))
   \brief  sets a vector in the interrupt vector table
\ingroup KERN
   \param       vector - Number of the vector to set
   \param       isr - memory location of the isr

   _KERN_IVTab_setvector sets the content of the interrupt vector
   table entry vector to isr. It saves the old content of the
   entry to restore it later.
 */
int _KERN_IVTab_setvector (unsigned short vector, void (*isr) (void))
{

	if (_KERN_IVTab_oldvector[vector]) {
		return (-1);
	} else {
		_KERN_IVTab_oldvector[vector] = (*_KERN_IVTab)[vector];
		(*_KERN_IVTab)[vector] = isr;
		return (0);
	}

}

/*! \fn int _KERN_IVTab_clearvector(unsigned short vector, void (*isr) (void))
   \brief resets the interrupt vector table
\ingroup KERN
   \param       vector - the vector number to clear
   \param       isr - the previously installed isr

   Resets the interrupt vector table entry to its original setting.
   The isr pointer is used to verify that a driver can only remove an
   isr that it has set before.
 */
int _KERN_IVTab_clearvector (unsigned short vector, void (*isr) (void))
{
	if ((*_KERN_IVTab)[vector] != isr) {
		return (-1);
	} else {
		(*_KERN_IVTab)[vector] = _KERN_IVTab_oldvector[vector];
		_KERN_IVTab_oldvector[vector] = 0;
		return (0);
	}

}
