/*  src/kernel/list.c
   CubeOS Version 0.4.90
   Copyright (C) 1999,2000 Holger Kenn

   CubeOS is free software; you can redistribute it and/or
   modify it under the terms of the GNU Library General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or any later version.

   CubeOS is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Library General Public License for more details.

 */
#include <stddef.h>
#include <list.h>
/*! \file list.c
\ingroup LIST
*/

/*!
\brief returns a pointer to the head entry of a list
\param l is a pointer to a list
\ingroup LIST
*/
entry *
  LIST_head (list * l)
{
	if (l == NULL)
		return NULL;
	return l->head;
}

/*!
\brief returns a pointer to the tail entry of a list
\param l is a pointer to a list
\ingroup LIST
*/
entry *
  LIST_tail (list * l)
{
	if (l == NULL)
		return NULL;
	return l->tail;
}

/*!
\brief initializes a list for first use
\param l is a pointer to a list
\ingroup LIST
*/
void LIST_init (list * l)
{
	if (l == NULL)
		return;
	l->head = NULL;
	l->tail = NULL;
	l->entries = 0;
	l->type = LIST_TYPE_USER;
}

/*!
\brief returns the number of entries in a list
\param l is a pointer to a list
\ingroup LIST
*/
int LIST_entries (list * l)
{
	if (l == NULL)
		return -1;
	return l->entries;
}

/*!
\brief inserts an entry after another entry in a list
\param e is a pointer to the entry to be inserted
\param x is a pointer to the entry after which e is to be inserted
\ingroup LIST

The list for e to be inserted to is determined by dereferencing x's list field.
If e is already in a list, the function fails.
*/
void LIST_insert_after (entry * e, entry * x)
{
	if (e == NULL)
		return;
	if (e->list)
		return;
	if (x == NULL)
		return;
	if (x->list == NULL)
		return;
	e->prev = x;
	e->next = x->next;
	x->next = e;
	if (x->list->tail == x)
		x->list->tail = e;
	e->list = x->list;
	e->list->entries++;
}


/*!
\brief inserts an entry before another entry in a list
\param e is a pointer to the entry to be inserted
\param x is a pointer to the entry before which e is to be inserted
\ingroup LIST

The list for e to be inserted to is determined by dereferencing x's list field.
If e is already in a list, the function fails.
*/
void LIST_insert_before (entry * e, entry * x)
{
	if (e == NULL)
		return;
	if (e->list)
		return;
	if (x == NULL)
		return;
	if (x->list == NULL)
		return;
	e->prev = x->prev;
	e->next = x;
	x->prev = e;
	if (x->list->head == x)
		x->list->head = e;
	e->list = x->list;
	e->list->entries++;
}


/*!
\brief inserts an entry into a list as head entry and moves the previous head entry.
\param l is a pointer to the list
\param e is a pointer to the entry
\ingroup LIST

If e is already in a list, the function fails.
*/
void LIST_insert_head (list * l, entry * e)
{
	if (l == NULL)
		return;
	if (e == NULL)
		return;
	if (e->list)
		return;
	if (l->head == NULL) {
		l->head = e;
		l->tail = e;
		e->prev = NULL;
		e->next = NULL;
		e->list = l;
		l->entries++;
	} else
		LIST_insert_before (e, l->head);
}

/*!
\brief inserts an entry into a list as tail entry and moves the previous tail entry.
\param l is a pointer to the list
\param e is a pointer to the entry
\ingroup LIST

If e is already in a list, the function fails.
*/
void LIST_insert_tail (list * l, entry * e)
{
	if (l == NULL)
		return;
	if (e == NULL)
		return;
	if (e->list)
		return;
	if (l->tail == NULL) {
		l->head = e;
		l->tail = e;
		e->prev = NULL;
		e->next = NULL;
		e->list = l;
		l->entries++;
	} else
		LIST_insert_after (e, l->tail);
}

/*!
\brief deletes an entry from its current list
\param e is a pointer to the entry
\ingroup LIST

If e is not in a list, the function fails.
*/

void LIST_delete (entry * e)
{
	if (e == NULL)
		return;
	if (e->list == NULL)
		return;
	if (e->prev)
		e->prev->next = e->next;
	if (e->next)
		e->next->prev = e->prev;
	if (e == e->list->tail)
		e->list->tail = e->prev;
	if (e == e->list->head)
		e->list->head = e->next;
	e->list->entries--;
	e->list = NULL;
}

/*!
\brief prepares an entry for (re-)use.
\param e is a pointer to the entry
\ingroup LIST
*/
void LIST_makeentry (entry * e)
{
	if (e == NULL)
		return;
	e->list = NULL;
	e->prev = NULL;
	e->next = NULL;
	e->data = NULL;
	e->len = 0;
}
