/*  src/kernel/ptimer.c
   CubeOS Version 0.4.90
   Copyright (C) 1999,2000 Holger Kenn

   CubeOS is free software; you can redistribute it and/or
   modify it under the terms of the GNU Library General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or any later version.

   CubeOS is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Library General Public License for more details.

 */
/*! \file ptimer.c
\ingroup KERN
*/

#include <ptimer.h>
#include <cubeos.h>
#include <sys_var.h>
#include <io_duart.h>
#include <schedule.h>
#include <sys/time.h>


void (*_KERN_call_dispatcher) ();

/*!
\brief installs a timer dispatcher routine that is called every tick
\ingroup KERN
*/
int KERN_installdispatcher(void (*dispatcher) ())
{
if (_KERN_call_dispatcher)
	return (-1);
disable();
_KERN_call_dispatcher = dispatcher;
enable();
return(0);
}

/*!
\brief removes the installed timer dispatcher routine
\ingroup KERN
*/
int KERN_removedispatcher()
{
if (!_KERN_call_dispatcher)
	return(-1);
disable();
_KERN_call_dispatcher = NULL;
enable();
return (0);
}


/*
 * The periodic timer is a internal device of the 68332
 */

/*!
\brief periodic timer interrupt routine
\ingroup KERN

This function is called by the periodic timer interrupt.
*/
int KERN_ptint (void)
{


	/* Advance the system clock */
	if ((++_time_ticks) == TICKS_PER_SECOND) {
		_time_seconds++;
		_time_ticks = 0;
	}
#ifdef DUART_BASE
	if ((DUART_BASE) != 0)
		_DUART_duart_bugfix ();
#endif
	if (_KERN_call_dispatcher)
		_KERN_call_dispatcher ();

	if (KERN_delta_handler ()) {
		/* time has run out for at least one thread, 
		   delta_handler has woken up the threads, we need to call the
		   scheduler immediately */
		_KERN_quantum_count = 0;	/* abort the current quantum! */
		return (1);
	}
	if (++_KERN_quantum_count == QUANTUM) {
		_KERN_quantum_count = 0;
		return (1);	/* and call scheduler */
	}
	return (0);		/* don't call scheduler */
}

/*!
\brief initializes the periodic timer of the '332 SIM
\ingroup KERN
*/
void KERN_init_ptimer (void)
{
	_time_seconds = 0;
	_time_ticks = 0;
	writeshort (SIM_PITR, PTIMER_PITR_VAL);

	writeshortpos (SIM_PICR, PTIMER_VECTOR, 0xff, 0x0);
	writeshortpos (SIM_PICR, PTIMER_IRQ_LEVEL, 0x7, 0x8);

//      ptimer_clears_wdt = 1; /* clear WDT on timer int */


// the software watchdog timeout period is determined by 
	//      XTAL_FREQ: The clock frequency of the VCO/external clock source
	//      SIM_SYPCR.SWP: Clock Prescaler, 1: div 512 2: no prescaler
	//      SIM_SYPCR.SWT0/1





}

/*!
\brief returns the second counter
\ingroup KERN
*/
unsigned long _getseconds ()
{
	return _time_seconds;
}


/*!
\brief fills a struct timeval with the current internal time.
\todo gettimeofday does not work
\ingroup KERN
*/
int _gettimeofday (struct timeval *tp, struct timezone *ignore)
{

	tp->tv_sec = _time_seconds;

	tp->tv_usec = (_time_ticks * 1000 / TICKS_PER_SECOND);
	return 0;
}

int gettimeofday (struct timeval *tp, struct timezone *ignore)
{

	tp->tv_sec = _time_seconds;

	tp->tv_usec = (_time_ticks * 1000 / TICKS_PER_SECOND);
	return 0;
}

/*!
\brief sets the internal time from an integer
\ingroup KERN
\todo _settimeofday does not work
*/
unsigned long _settimeofday (unsigned long val)
{
	writeshort (SIM_PITR, 0);	/* Disable timer interrupt */
	_time_ticks = 0;
	_time_seconds = val;	/* set clock */
	writeshort (SIM_PITR, PTIMER_PITR_VAL);		/* and return it */
	return _time_seconds;
}

