/*  src_experimental/kernel/sleep.c
   CubeOS Version 0.4.90 experimental
   Copyright (C) 1999,2000 Holger Kenn

   CubeOS is free software; you can redistribute it and/or
   modify it under the terms of the GNU Library General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or any later version.

   CubeOS is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Library General Public License for more details.

 */
/*! \file sleep.c
\ingroup KERN
*/

#include <cubeos.h>
#include <sys_var.h>
#include <ptimer.h>
#include <schedule.h>
#include <list.h>

/* 
\brief suspends the current thread for a while in the scheduler
\param howlong specifies the number of timer ticks the thread should not run
\ingroup KERN
*/
int KERN_psleep (unsigned int howlong)
{
	int deltatrack = 0;
	int savetime;
	struct process *currentprocess;
	struct process *thisprocess;
	entry *listentry;
	int quit = 0;

	disable ();


	if (LIST_entries (&_KERN_delta) == 0) {
		/* this is the easy case */
		_KERN_ptable[getpid ()].time_delta = howlong;
		LIST_insert_head (&_KERN_delta, &(_KERN_ptable[getpid ()].me));
		enable ();
		return (KERN_suspend (-1));
	}
	listentry = LIST_head (&_KERN_delta);
	thisprocess = &(_KERN_ptable[getpid ()]);
	while (!quit) {
		currentprocess = (struct process *) (listentry->data);
		deltatrack += currentprocess->time_delta;
		if (howlong <= deltatrack) {
			/* if we have an earlier deadline than the process in the delta queue */
			/* we have to put ourselves in front and decrement the current processes time_delta by howlong */

			/* move the running process in front of the current process */
			LIST_insert_before (&(thisprocess->me), listentry);

			/* adjust the times. First: substract howlong from the current process time */
			currentprocess->time_delta = -howlong;

			/* then: set our own delta time */
			thisprocess->time_delta = howlong - deltatrack;
			quit = 1;
		} else {
			if (listentry->next) {
				listentry = listentry->next;	/* if there is another entry, go and check it */
			} else {
				/* we're at the end of the queue, so we append here */
				LIST_insert_after (&(thisprocess->me), listentry);
				thisprocess->time_delta = howlong - deltatrack;
				quit = 1;
			}
		}
	}

	enable ();
	return (-1);
}

/* 
\brief suspends the current thread for a while
\param howlong is the number of seconds the thread is blocking
\ingroup KERN
\todo this is currently implemented by actively watching clockticks and is a waste of cycles.
This has to be changed to use KERN_psleep instead
*/
int KERN_ssleep (unsigned int howlong)
{
	unsigned int seconds;
	unsigned int ticks;

	disable ();
	seconds = _time_seconds;
	ticks = _time_ticks;
	enable ();

	seconds += howlong;

	while (seconds > _time_seconds);
	while ((seconds == _time_seconds) && (ticks > _time_ticks));
	return (0);
}

/* 
\brief suspends the current thread for a while
\param howlong is the number of microseconds the thread is blocking
\ingroup KERN
\todo this is currently implemented by actively watching clockticks and a is waste of cycles.
This has to be changed to use KERN_psleep instead. For shorter times, a timed loop
like in BOGOMIPS of Linux should be used instead.
*/
int KERN_usleep (unsigned int howlong)
{
	unsigned int seconds;
	unsigned int ticks;


	disable ();
	seconds = _time_seconds;
	ticks = _time_ticks;
	enable ();
	ticks += howlong * TICKS_PER_SECOND / 1000000;
	seconds += ticks / TICKS_PER_SECOND;
	ticks = ticks % TICKS_PER_SECOND;
	while (seconds > _time_seconds);
	while ((seconds == _time_seconds) && (ticks > _time_ticks));
	return (0);

}
