/*  src_experimental/net/rsm.c
   CubeOS Version 0.4.90 experimental
   Copyright (C) 1999,2000 Holger Kenn

   CubeOS is free software; you can redistribute it and/or
   modify it under the terms of the GNU Library General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or any later version.

   CubeOS is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Library General Public License for more details.

 */
// 
// Radio State machine for CubeOS
//

/*! \file rsm.c
\ingroup RSM
*/

#include <rsm.h>
#include <rdio.h>
#include <cubeos.h>
//#include <conio.h>
#include <stdio.h>
#include <ttyio.h>

//
// A frame for the RSM looks like this:
//
// 00 ff 00 ff ... 00 ff SOH E1 E2 E1 E2 E1 E2 ... E1 E2 C1 C2
//
// There are exactly RMS_FRAMELEN E1/E2 pairs, each one encoding one byte
// The C1/C2 bytes represent the encoded 8-bit checksum of the frame
//
// At the moment, no escaping of the SOH byte is done, so the protocol
// may get stuck.
//

// States of the RSM

#define RSM_OFF 0
#define RSM_IDLE 1
#define RSM_INFRAME1 2
#define RSM_INFRAME2 3
#define RSM_CHECKSUM1 4
#define RSM_CHECKSUM2 5

// Start of Header
#define RSM_SOH 0xA5

#define RSM_PREHEADER 6		//Number of 0x00 0xff in preheader

//#define RSM_DEBUG

// variables

// buffers
char _RSM_buffer0[RSM_FRAMELEN];
char _RSM_buffer1[RSM_FRAMELEN];

// internal
unsigned char _RSM_state = 0;
//unsigned char _RSM_enabled = 0;
short _RSM_count;
char RSM_new;
char *_RSM_writebuf;
char *RSM_getbuf;
unsigned char _RSM_checksum;
unsigned char _RSM_charcount;

// statistics
int RSM_good;
int _RSM_bad;
int _RSM_ugly;
int _RSM_bytes;
int _RSM_uglycount;

// function prototypes: user interface

int RSM_init_rx ();
int RSM_init_tx ();
int RSM_getframe (unsigned char *buffer);
int RSM_send_frame (unsigned char *buffer);


// function prototypes: for internal use only

void _RSM_doutchar (unsigned char c);
void _RSM_douthex (unsigned char c);
void RSM_process (unsigned char c);
void RSM_tty_rdioput_block (char byte);
unsigned char *RSM_rxframe (unsigned char *frame);


// function implementations

/*!
\brief prepares RSM for reception
\ingroup RSM
*/
int RSM_init_rx ()
{
#ifdef DUART_BASE
	_RSM_writebuf = _RSM_buffer0;
	RSM_getbuf = _RSM_buffer1;
	_RSM_state = RSM_IDLE;
	RSM_good = 0;
	_RSM_bad = 0;
	_RSM_ugly = 0;
	_RSM_bytes = 0;
	_RSM_uglycount = 0;
	RSM_new = 0;
	_RSM_charcount = 0;
//	_RSM_enabled = 1;	/* redirect to RSM_process() */
	TTY_tty[1].char_process = RSM_process; /* enable char processor */
	clearRTXSEL ();		/* disable transmitter */
	setRRXSEL ();		/* enable receiver */
	RSM_rdio_enable_rx ();	/* Enable reception IRQ */
	return 0;
#else
	return -1;
#endif
}


/*!
\brief disables reception
\ingroup RSM
*/
int RSM_deinitrx ()
{
#ifdef DUART_BASE
	clearRRXSEL ();		/* disable receiver */
//	_RSM_enabled = 0;	/* redirect to RSM_process() */
	TTY_tty[1].char_process = NULL; /* disable char processor */
	_RSM_state = RSM_IDLE;	/* back to idle state */
	RSM_rdio_disable_rx ();	/* disable reception IRQ */
	return 0;
#else
	return -1;
#endif
}

/*!
\brief initializes RSM for transmission
\ingroup RSM
*/
int RSM_init_tx ()
{
#ifdef DUART_BASE
	clearRRXSEL ();		/* disable receiver */
	setRTXSEL ();		/* enable transmitter */
	RSM_rdio_disable_rx ();	/* disable RX ISR */
//	_RSM_enabled = 0;	/* and redirect to rdio queue */
	TTY_tty[1].char_process = NULL; /* disable char processor */

	return 0;
#else
	return -1;
#endif
}

/*!
\brief disables transmission
\ingroup RSM
*/
int RSM_deinittx ()
{
#ifdef DUART_BASE
	RSM_rdio_flush ();
	clearRTXSEL ();		/* disable transmitter */
	return 0;
#else
	return -1;
#endif
}

#ifdef RSM_DEBUG

/*!
\brief outputs a debug char for RSM
\ingroup RSM
*/
void _RSM_doutchar (unsigned char c)
{
	iTTY_outchar (c);
}

/*!
\brief outputs a debug hex character for RSM
\ingroup RSM
*/
void _RSM_douthex (unsigned char c)
{
	char l;
	char h;

	h = (c & 0xf0) >> 4;
	if (h < 10) {
		_RSM_doutchar ('0' + h);
	} else {
		_RSM_doutchar ('A' + h - 10);
	}
	l = (c & 0x0f);
	if (l < 10) {
		_RSM_doutchar ('0' + l);
	} else {
		_RSM_doutchar ('A' + l - 10);
	}
}

#else
void _RSM_doutchar (unsigned char c)
{
}
void _RSM_douthex (unsigned char c)
{
}

#endif

#ifdef DUART_BASE

/*!
\brief process a new char in the RSM
\ingroup RSM
*/
void RSM_process (unsigned char c)
{				/* this is called from the duart irq */
	static unsigned char save;
	unsigned char cx;

	_RSM_bytes++;		// byte statistics

	switch (_RSM_state) {
	case RSM_IDLE:
		if (c == RSM_SOH) {
			_RSM_state = RSM_INFRAME1;
			_RSM_count = 0;
			_RSM_checksum = 0;
			_RSM_doutchar ('X');
		}
		break;

	case RSM_INFRAME1:
		if ((c & 0xAA) != ((~((c << 1) | 0x55)) & 0xff)) {
			_RSM_state = RSM_IDLE;
			_RSM_uglycount = _RSM_count;
			_RSM_ugly++;
			_RSM_doutchar ('u');
			break;
		}
		save = c;
		_RSM_state = RSM_INFRAME2;
		break;

	case RSM_INFRAME2:
		if ((c & 0xAA) != ((~((c << 1) | 0x55)) & 0xff)) {
			_RSM_state = RSM_IDLE;
			_RSM_uglycount = _RSM_count;
			_RSM_ugly++;
			_RSM_doutchar ('U');
			break;
		}
		cx = ((save & 0xAA) | (c & 0x55));
		_RSM_writebuf[_RSM_count++] = cx;
		_RSM_checksum += cx;

		if (_RSM_count >= RSM_FRAMELEN) {
			_RSM_state = RSM_CHECKSUM1;
			break;
		}
		_RSM_state = RSM_INFRAME1;
		break;

	case RSM_CHECKSUM1:
		save = c;
		_RSM_state = RSM_CHECKSUM2;
		break;

	case RSM_CHECKSUM2:
		if (
		      ((save & 0xAA) != ((~((save << 1) | 0x55)) & 0xff)) ||
			   ((c & 0xAA) != ((~((c << 1) | 0x55)) & 0xff))
			) {
			_RSM_state = RSM_IDLE;
			_RSM_uglycount = _RSM_count;
			_RSM_ugly++;
			_RSM_doutchar ('U');
			_RSM_doutchar ('c');
			break;
		}
		cx = ((save & 0xAA) | (c & 0x55));
		if (cx == _RSM_checksum) {
			_RSM_doutchar ('G');
			_RSM_douthex (cx);
			_RSM_doutchar ('\n');
			RSM_good++;

//                      bufsave = RSM_getbuf;
			//                      RSM_getbuf = _RSM_writebuf;
			//                      _RSM_writebuf = bufsave;
			//                      RSM_new = 1;
			_RSM_writebuf = RSM_rxframe (_RSM_writebuf);
			if (_RSM_writebuf == NULL) {	/* we did not get a new buffer, so we 
							   disable the reception */
				RSM_rdio_disable_rx ();
				_RSM_doutchar ('N');
			}
		} else {
			_RSM_bad++;
			_RSM_doutchar ('B');
			_RSM_douthex (cx);
			_RSM_doutchar ('e');
			_RSM_douthex (_RSM_checksum);
			_RSM_doutchar ('\n');
		}
		_RSM_state = RSM_IDLE;
	}
}


// this implements a simple double buffer scheme. One gets written, the other 
// can be read
#endif


/*!
\brief receive one RSM frame
\ingroup RSM
*/
unsigned char *RSM_rxframe (unsigned char *rxframebuf)
{
	unsigned char *bufsave;

	bufsave = RSM_getbuf;
	RSM_getbuf = rxframebuf;	// this is the pointer to the frame just received

	RSM_new = 1;		// new frame available

	return (bufsave);	// we return the previous getbuf

}

/*!
\brief copy one RSM frame into "user space"
\ingroup RSM
*/
int RSM_getframe (unsigned char *buffer)
{
	int i;

	if (!RSM_new)
		return (-1);
	disable ();
	for (i = 0; i < RSM_FRAMELEN; i++)
		buffer[i] = RSM_getbuf[i];
	RSM_new = 0;
	enable ();

	return (0);

}

/*!
\brief do media encoding for one byte
\param c is the byte to be encoded
\ingroup RSM
*/
unsigned char *RSM_encode (unsigned char c)
{
	static unsigned char buf[2];

	unsigned char t, hi, lo;

	t = (c & 0xaa);
	hi = t | ((~(t >> 1)) & 0x55);
	t = (c & 0x55);
	lo = t | ((~(t << 1)) & 0xaa);
	*buf = hi;
	*(buf + 1) = lo;
	return buf;

}

/*!
\brief blocking output on radio TTY
\ingroup RSM
\ingroup TTY
*/
void RSM_tty_rdioput_block (char c)
{
#ifdef DUART_BASE
	int i;
	i = 0;
	while (TTY_rdioput (c)) {
		i++;
		if (i > 1000) {
			//_RSM_doutchar('B');
		}
	}
#endif

}

/*!
\brief send one RSM frame
\ingroup RSM
*/
int RSM_send_frame (unsigned char *buffer)
{

	unsigned char *buf;
	int i;
	unsigned char checksum;


	for (i = 1; i < RSM_PREHEADER; i++);
	{
		RSM_tty_rdioput_block (0x00);
		RSM_tty_rdioput_block (0xff);
	}

	checksum = 0;

	RSM_tty_rdioput_block (RSM_SOH);

	for (i = 0; i < RSM_FRAMELEN; i++) {
		buf = RSM_encode (buffer[i]);
		RSM_tty_rdioput_block (buf[0]);
		RSM_tty_rdioput_block (buf[1]);
		checksum += buffer[i];
	}

	buf = RSM_encode (checksum);

	RSM_tty_rdioput_block (buf[0]);
	RSM_tty_rdioput_block (buf[1]);
	return (0);

}
