#include <stdio.h>
#include <readcmd.h>
#include <fixpoint.h>
#include <drive.h>
#include <config.h>
#include <config_drive.h>
#include <motor.h>
#include <string.h>
#include <tpud.h>
#include <config_motor.h>

extern long long dr_angl_const, dr_dist_const;
extern int dr_angle_malpha, dr_angle_stop;
extern int dr_length_break, dr_length_stop;
extern int dr_length_apspeed, dr_length_approach;
extern int dr_enc_res;
extern float dr_gear_ratio, dr_wheel_radius;

extern drive_t DRIVE[];
extern int dr_drive;


/* 
\brief asks the user interactively which drivesetting to use. 
*/
int ask_type_drive() {
	int drivetype;

        printf("\n");
        printf("\n");
        printf("*********** Base-Type ***********\n");
        printf("\n");
        printf("  Guard V1      = %d\n", DR_GUARD_V1);
        printf("  Guard V2      = %d\n", DR_GUARD_V2);
        printf("  RCup fast forward  = %d\n", DR_RCUP_FASTFORWARD);
        printf("  RCup forward  = %d\n", DR_RCUP_FORWARD);
        printf("  RCup defender = %d\n", DR_RCUP_DEFENDER);
        printf("  RCJ           = %d\n", DR_RCJ);
        printf("\n");
        sscanf(readcmd("make your choice: "),"%d",&drivetype);
        printf("\n");
        printf("\n");

	return(drivetype);
}

/* 
\brief configures the internal set of known drives.
*/
int config_drives() {

/*****************************************************************************/
// default values
/*****************************************************************************/

/* PWM base frequency */
DRIVE[DR_DEFAULT].pwm_basefreq =	20000;

/* vector parameters */
DRIVE[DR_DEFAULT].angle_malpha =	2000;
DRIVE[DR_DEFAULT].angle_stop =		500;
DRIVE[DR_DEFAULT].length_apspeed =	3000;
DRIVE[DR_DEFAULT].length_break =	150;
DRIVE[DR_DEFAULT].length_approach =	50;
DRIVE[DR_DEFAULT].length_stop =		5;

/* constants for dead-reckoning */
DRIVE[DR_DEFAULT].gear_ratio =		0.0;
DRIVE[DR_DEFAULT].wheel_radius =	0.0;
DRIVE[DR_DEFAULT].wheel_dist =		0.0;
DRIVE[DR_DEFAULT].enc_res =		0.0;

/* relate (normalized) max. speed to encoder pulses */
DRIVE[DR_DEFAULT].pid_p2s =		0;
DRIVE[DR_DEFAULT].pid_P =		0.0;
DRIVE[DR_DEFAULT].pid_I =		0.0;
DRIVE[DR_DEFAULT].pid_D =		0.0;
DRIVE[DR_DEFAULT].pid_leaky_q =		-1.0;


/*****************************************************************************/
// Guard Version 2
/*****************************************************************************/

/* initialise with the default values */
bcopy( (void*)&DRIVE[DR_DEFAULT],(void*)&DRIVE[DR_GUARD_V2],sizeof(DRIVE[DR_DEFAULT]));

/* constants for dead-reckoning */
DRIVE[DR_GUARD_V2].gear_ratio =		(17.0+(29.0/35.0));
DRIVE[DR_GUARD_V2].wheel_radius =	49.0;
DRIVE[DR_GUARD_V2].wheel_dist =		325.0;
DRIVE[DR_GUARD_V2].enc_res =		(500.0*4.0);

/* relate (normalized) max. speed to encoder pulses */
DRIVE[DR_GUARD_V2].pid_p2s =		((2500*100)/DR_ISRFREQ);
DRIVE[DR_GUARD_V2].pid_P =		0.6;
DRIVE[DR_GUARD_V2].pid_I =		0.003;
DRIVE[DR_GUARD_V2].pid_D =		0.4;


/*****************************************************************************/
// RoboCup Forward
/*****************************************************************************/

/* initialise with the default values */
bcopy( (void*)&DRIVE[DR_DEFAULT],(void*)&DRIVE[DR_RCUP_FORWARD],sizeof(DRIVE[DR_DEFAULT]));

/* vector parameters */
DRIVE[DR_RCUP_FORWARD].angle_malpha =		3000;
DRIVE[DR_RCUP_FORWARD].angle_stop =		500;
DRIVE[DR_RCUP_FORWARD].length_apspeed =		6000;
DRIVE[DR_RCUP_FORWARD].length_break =		50;
DRIVE[DR_RCUP_FORWARD].length_approach =	20;
DRIVE[DR_RCUP_FORWARD].length_stop =		5;

/* constants for dead-reckoning */
DRIVE[DR_RCUP_FORWARD].gear_ratio =	(16.0+(113.0/196.0));
DRIVE[DR_RCUP_FORWARD].wheel_radius =	14.0;
DRIVE[DR_RCUP_FORWARD].wheel_dist =	96.0;
DRIVE[DR_RCUP_FORWARD].enc_res =	(16.0*4.0);

/* relate (normalized) max. speed to encoder pulses */
DRIVE[DR_RCUP_FORWARD].pid_p2s =	((115*100)/DR_ISRFREQ);
DRIVE[DR_RCUP_FORWARD].pid_P =		0.1;
DRIVE[DR_RCUP_FORWARD].pid_I =		0.0;
DRIVE[DR_RCUP_FORWARD].pid_D =		0.0;


/*****************************************************************************/
// RoboCup Fast Forward
/*****************************************************************************/

/* initialise with the RoboCup Forward as default values */
bcopy( (void*)&DRIVE[DR_RCUP_FORWARD],(void*)&DRIVE[DR_RCUP_FASTFORWARD],sizeof(DRIVE[DR_RCUP_FORWARD]));

DRIVE[DR_RCUP_FASTFORWARD].wheel_radius =	25.0;
DRIVE[DR_RCUP_FASTFORWARD].wheel_dist =		102.0;

/*****************************************************************************/
// RoboCup Defender
/*****************************************************************************/

/* initialise with the RoboCup Forward as default values */
bcopy( (void*)&DRIVE[DR_RCUP_FORWARD],(void*)&DRIVE[DR_RCUP_DEFENDER],sizeof(DRIVE[DR_RCUP_FORWARD]));

DRIVE[DR_RCUP_DEFENDER].wheel_radius =	14.5;
DRIVE[DR_RCUP_DEFENDER].wheel_dist =	104.0;

/*****************************************************************************/

/*****************************************************************************/
// RCJ
/*****************************************************************************/

/* initialise with the default values */
bcopy( (void*)&DRIVE[DR_DEFAULT],(void*)&DRIVE[DR_RCJ],sizeof(DRIVE[DR_DEFAULT]));

/* vector parameters */
DRIVE[DR_RCJ].angle_malpha =		3000;
DRIVE[DR_RCJ].angle_stop =		500;
DRIVE[DR_RCJ].length_apspeed =		6000;
DRIVE[DR_RCJ].length_break =		50;
DRIVE[DR_RCJ].length_approach =	20;
DRIVE[DR_RCJ].length_stop =		5;

/* constants for dead-reckoning */
DRIVE[DR_RCJ].gear_ratio =	(16.0+(113.0/196.0));
DRIVE[DR_RCJ].wheel_radius =	25.0;
DRIVE[DR_RCJ].wheel_dist =	166.0;
DRIVE[DR_RCJ].enc_res =	(16.0*4.0);

/* relate (normalized) max. speed to encoder pulses */
DRIVE[DR_RCJ].pid_p2s =	((115*100)/DR_ISRFREQ);
DRIVE[DR_RCJ].pid_P =		0.1;
DRIVE[DR_RCJ].pid_I =		0.0;
DRIVE[DR_RCJ].pid_D =		0.0;


return 0;
}

/*
\brief initializes the drive controller with the values of one specific drive 
from the driveset and calculates some internal drive-dependent parameters.
\param drive is the selected drive
*/
int init_drive(int drive) {
	TPU_init();
        FBIN_init();

	dr_drive = drive;

	dr_angle_malpha = DRIVE[drive].angle_malpha;
	dr_angle_stop = DRIVE[drive].angle_stop;
	dr_length_apspeed = DRIVE[drive].length_apspeed;
	dr_length_break = DRIVE[drive].length_break;
	dr_length_approach = DRIVE[drive].length_approach;
	dr_length_stop = DRIVE[drive].length_stop;

	dr_enc_res = (int) DRIVE[drive].enc_res;
	dr_gear_ratio = DRIVE[drive].gear_ratio;

	config_motor(0.0,0.0,0.0,-1.0,DRIVE[drive].pid_p2s);
	MOTOR_init(MLEFT,DRIVE[drive].pwm_basefreq);
	MOTOR_init(MRIGHT,DRIVE[drive].pwm_basefreq);

	DRIVE_set_pid(DRIVE[drive].pid_P,DRIVE[drive].pid_I,DRIVE[drive].pid_D,DRIVE[drive].pid_leaky_q);

	dr_angl_const = (long long) (( 
		(36000.0 * DRIVE[drive].wheel_radius) / 
		(DRIVE[drive].gear_ratio * DRIVE[drive].enc_res * DRIVE[drive].wheel_dist) ) 
		* (float) C_FPOINT_SQ);
	dr_dist_const = (long long) ((
		(3.14159265359 * DRIVE[drive].wheel_radius) /
		(DRIVE[drive].gear_ratio * DRIVE[drive].enc_res) )
		* (float) C_FPOINT_SQ);
	return(0);
}


