#include <cubeos.h>
#include <tpu.h>
#include <tpud.h>
#include <string.h>
#include <motor.h>
#include "fixpoint.h"

motor_t MOTOR[3]; 

motor_t emptymotor={0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0}; //!< just to clear out motors before use.

/*! 
\brief Sets the direction bits for a motor 

\param motor specifies the motor to be configured
This function reads the configuration of the motor from MOTOR[] and
prepares the FBIN direction control outputs accordingly.

*/
void MOTOR_set_directionbits(int motor) {
	FBIN_make(MOTOR[motor].f0ch,MOTOR[motor].f0type,MOTOR[motor].f0id);
	FBIN_make(MOTOR[motor].f1ch,MOTOR[motor].f1type,MOTOR[motor].f1id);
}


/*!
\brief Sets the motor pulsewidth control frequency

\param motor specifies the motor to be configured
\param pulse_frequency specifies the pulse frequency in Hz

This function re-configures an already-configured motor to a
new pulsewidth control frequency and also takes care of the right
settings for the hightime parameter.
*/
void MOTOR_set_frequency(int motor,int pulse_frequency) {
	MOTOR[motor].pwmper = TCR1FREQ / pulse_frequency;
        setPAR (MOTOR[motor].pwmch, 2, abs (MOTOR[motor].PW) * MOTOR[motor].pwmper / 10000);
	TPU_setpwmperiod (MOTOR[motor].pwmch, (unsigned short)MOTOR[motor].pwmper);
	setHSRR (MOTOR[motor].pwmch, 2);	/* request init */
}

/*!
\brief prepares the TPU channels for motor control use

\param motor specifies the motor to be configured

This function initialises the control PWM function channel for a motor
and also prepares two QDEC function channels to read motor pulses
for odometry and PID control.
*/
void MOTOR_set_tpchannels(int motor,int pulse_frequency) {
	int period;

	period = TCR1FREQ / pulse_frequency;
	printf("TCR1FREQ = %d\n",TCR1FREQ);
	MOTOR[motor].pwmper=period;
	MOTOR[motor].tcrfreq=TCR1FREQ;
	TPU_makepwm_TCR1 (MOTOR[motor].pwmch);
	TPU_setpwmperiod (MOTOR[motor].pwmch, (unsigned short)period);
	TPU_makeqd (MOTOR[motor].qd1ch,MOTOR[motor].qd2ch);
	setHSRR (MOTOR[motor].pwmch, 2);	/* request init */
}

/*! 
\brief This function prepares one motor to be used according to its configuration.

\param motor specifies the motor to be configured
\param pulse_frequency is an additional parameter that is used to configure the motor TPU channels

This function reads the motor configuration from MOTOR[] and prepares the hardware
to control the motor. It calls the functions MOTOR_set_directionbits() and MOTOR_set_tpchannels()
*/
void MOTOR_init(int motor,int pulse_frequency)
{
MOTOR_set_tpchannels(motor,pulse_frequency);
MOTOR_set_directionbits(motor);
}

/*!
\brief reads out the motor quadrature decoders and stores the value in MOTOR[].

\param motor specifies which motor's QDECs are to be read

This function reads out the quadrature decoder value and stores it into the corresponding MOTOR[] fields.
(qdval gets the raw value, normspeed gets the normailized speed value.)
It then resets the qudec. Further access during one control cycle should only be made through
direct access to MOTOR[].
\internal
*/
int _MOTOR_encoder(int motor) {
	MOTOR[motor].qdval=TPU_getqd (MOTOR[motor].qd2ch);
        MOTOR[motor].normspeed = ((((C_FPOINT * MOTOR[motor].qdval)/MOTOR[motor].p2s) // this gives 0-10000 encoder reading
				   *C_FPOINT)/MOTOR[motor].speedscale); // this scales up encoder reading according to speedscale
	return MOTOR[motor].qdval;
}

/*! 
\brief returns the raw qdec-value of a motor (mainly for debugging purposes)
\param motor specifies the motor
\internal
*/
short MOTOR_get_qdec(int motor) {
	return(MOTOR[motor].qdval);
}

/*!
\brief sets the output PWM of the motor

\param motor selects the motor
\param newPW is the future pulsewidth normalized to 0-10000
\internal
*/
void _MOTOR_setPW(int motor, int newPW){

	MOTOR[motor].PW= newPW;

        if (newPW < 0) {
                FBIN_set (MOTOR[motor].f0ch, 0);
                FBIN_set (MOTOR[motor].f1ch, 1);
        } else {
                FBIN_set (MOTOR[motor].f0ch, 1);~
                FBIN_set (MOTOR[motor].f1ch, 0);
        }

        setPAR (MOTOR[motor].pwmch, 2, abs (newPW) * MOTOR[motor].pwmper / 10000);
}


/*! 
\brief Sets pid parameters for motor pid controller

\param motor selects the motor to be configured
\param P is the proportional gain factor as float
\param I is the integrative gain factor as float
\param D is the differential gain factor as float
 The setting of these PID parameters has severe impact on the
behaviour of the robot. change at your own risk. Good values are usually very small.
*/
void MOTOR_set_pid(int motor, float P, float I, float D, float leaky_q) {

//printf("Setpid called, motor %d P:%f I:%f D:%f\n",motor,P,I,D);
	MOTOR[motor].p = (int)(P * C_FPOINT);
	MOTOR[motor].i = (int)(I * C_FPOINT);
	MOTOR[motor].d = (int)(D * C_FPOINT);
	MOTOR[motor].leaky_q = (leaky_q==-1.0)? -1 : (int)(leaky_q * C_FPOINT);
//printf("Setpid called, setting %d %d %d\n",MOTOR[motor].p,MOTOR[motor].i,MOTOR[motor].d);
}

/*! 
\brief Sets speedscale parameter for motor pid controller
\param motor selects the motor
\param scale is the speed scale factor. 

Setting a speedscale factor between 0 and 1 reduces the maximum speed of a motor.
Setting the speedscale factor to values larger than 1 will just degrade the performance
of the PID controller since the maximum motor speed is measured in a free-running loadless mode
which cannot be obtained under normal operating conditions.
*/
void MOTOR_set_speedscale(int motor, float scale) {
//printf("Setspeedscale called, motor %d scale %f \n",motor,scale);
	MOTOR[motor].speedscale = (int)(scale * C_FPOINT);
//printf("Setspeedscale called, setting %d \n",MOTOR[motor].speedscale);
}

/*! 
\brief configures a motor before use.
\param motor selects the motor.
\param pwmch is the TPU channel to be used as PWM output.
\param qd1ch is the TPU channel to be used as first QDEC channel.
\param qd2ch is the TPU channel to be used as second QDEC channel.
Exchanging first and second QDEC channel will negate the counted pulse number. Adjust fastbins accordingly.
\param f0type is the type code for fastbin 0.
\param f0id is the id code for fastbin 0.
\param f1type is the type code for fastbin 1.
\param f1id is the id code for fastbin 1.
Exchanging fastbins 0 and 1 will run the motor in reverse. Adjust QDECs accordingly.
\param p is the proportional gain factor as float.
\param i is the integrative gain factor as float.
\param d is the differential gain factor as float.
\param speedscale is the integerspeed scale factor. Initialize to 10000 and change via MOTOR_set_speedscale later.
\param p2s is the pulse-to-speed factor. It contains the physical properties of the motor and the motor encoder. p2s can
either be determined by calculation from known load conditions or by a free-run experiment of the final base.
*/
int MOTOR_set_config(int motor,
		 unsigned char pwmch,unsigned char qd1ch,unsigned char qd2ch,
		 char f0type, char f0id,char f1type,char f1id,
		 float p,float i,float d, float leaky_q,int speedscale,int p2s)
{

//printf("this is MOTOR_setconfig %d\n",motor);

if ((motor<0)||(motor>2)) return(-1);


bcopy((void *)&emptymotor,(void *)&MOTOR[motor],sizeof(emptymotor));

MOTOR[motor].pwmch=pwmch;
MOTOR[motor].qd1ch=qd1ch;
MOTOR[motor].qd2ch=qd2ch;
MOTOR[motor].f0type=f0type;
MOTOR[motor].f0id=f0id;
MOTOR[motor].f0ch=motor*2;
MOTOR[motor].f1type=f1type;
MOTOR[motor].f1id=f1id;
MOTOR[motor].f1ch=motor*2+1;
MOTOR[motor].p=(int)(p * C_FPOINT);
MOTOR[motor].i=(int)(i * C_FPOINT);
MOTOR[motor].d=(int)(d * C_FPOINT);
MOTOR[motor].leaky_q=(leaky_q==-1.0)? -1 : (int)(leaky_q * C_FPOINT);
MOTOR[motor].speedscale=speedscale;
MOTOR[motor].p2s=p2s;

/*
printf("config for  %d is \n",motor);
printf("pwmch = %d\n",MOTOR[motor].pwmch);
printf("qd1ch = %d\n",MOTOR[motor].qd1ch);
printf("qd2ch = %d\n",MOTOR[motor].qd2ch);
printf("f0type = %d\n",MOTOR[motor].f0type);
printf("f0id = %d\n",MOTOR[motor].f0id);
printf("f0ch = %d\n",MOTOR[motor].f0ch);
printf("f1type = %d\n",MOTOR[motor].f1type);
printf("f1id = %d\n",MOTOR[motor].f1id);
printf("f1ch = %d\n",MOTOR[motor].f1ch);
printf("p = %d\n",MOTOR[motor].p);
printf("i = %d\n",MOTOR[motor].i);
printf("d = %d\n",MOTOR[motor].d);
printf("speedscale = %d\n",MOTOR[motor].speedscale);
printf("pulse2speed = %d\n",MOTOR[motor].p2s);

*/

return(0);
}


/*! 
\brief motor PID control function

\param motor selects the motor
\param speed is the target speed the controller is supposed to reach

This function implements a pid controller for a motor. 
It has to be called in regular intervals.
The _MOTOR_encoder() function of the motor has to be called prior to calling this
function, otherwise the controller will calculate the pulsewidth changes based on old quadrature values.
*/
void MOTOR_pid_speed(int motor, int speed)
{
	int PW;
	int error_SPEED;
	int dif_error_SPEED;

// calculate and update the different types of errors

	error_SPEED  = speed - MOTOR[motor].normspeed;
	dif_error_SPEED  = error_SPEED - MOTOR[motor].old_err;
	MOTOR[motor].old_err = error_SPEED;
	if(MOTOR[motor].leaky_q == -1) {
		MOTOR[motor].acc_err += (long long) error_SPEED;
	} else {
		MOTOR[motor].acc_err = (long long) ((
			(C_FPOINT-MOTOR[motor].leaky_q) * MOTOR[motor].acc_err * (long long) C_FPOINT
			+ MOTOR[motor].leaky_q * error_SPEED) / C_FPOINT); 
	}

// calculate the pulsewidth compensations
	PW      = MOTOR[motor].old_pw   + (MOTOR[motor].p * error_SPEED)/C_FPOINT
                              + (MOTOR[motor].i * (int)MOTOR[motor].acc_err)/C_FPOINT
                              + (MOTOR[motor].d * dif_error_SPEED)/C_FPOINT;

// clip PW if necessary
	if (PW<-10000) PW=-10000;
	if (PW>10000) PW=10000;

// set motor-pulsewidths
	_MOTOR_setPW(motor,PW);

// store old values
	MOTOR[motor].old_pw = PW;

}

