#ifndef _MOTOR_H
#define _MOTOR_H

#include <fastbin.h>
#define CONTROL_PWM 15

/*! 
\brief this keeps all information about a single motor
*/
typedef struct motor_s {
	int tcrfreq;	//!< TCR base frequency of PWM
	unsigned char pwmch;	//!< PWM channel 
	int pwmper;     //!< PWM period current value
	unsigned char qd1ch;	//!< QD1 TP channel
	unsigned char qd2ch;	//!< QD2 TP channel
	short qdval;	//!< QD value
	char f0type;	//!< Fastbin 0 type
	char f0id;	//!< Fastbin 0 id
	char f0ch;	//!< Fastbin 0 channel
	char f1type;	//!< Fastbin 1 type
	char f1id;	//!< Fastbin 1 id
	char f1ch;	//!< Fastbin 1 channel
	int p2s;	//!< pulsetospeed constant
	int speedscale;  //!< speed scale for motor
	int normspeed;	//!< current normalized speed -10000<x<10000
	int PW;		//!< current PW setting
	int p;		//!< p value, scaled times 10000
	int i;		//!< i value, scaled times 10000
	int d;		//!< d value, scaled times 10000
        int leaky_q;	//!< q value for leaky integration, scaled by 10000
	int old_pw;	//!< internal for PID use
	int old_err;	//!< internal for PID use
	long long acc_err;	//!< internal for PID use
} motor_t;

extern motor_t MOTOR[3];	//!< default motor array

#define MLEFT 0		//!< default left MOTOR[] array index
#define MRIGHT 1	//!< default right MOTOR[] array index
#define MKICK 2		//!< default kick MOTOR[] array index

#ifndef DOXYSKIP
//********* access macros for motor config *****************

#define CFGLEFT_D0		MOTOR[MLEFT].f0ch 
#define CFGLEFT_D1		MOTOR[MLEFT].f1ch
#define CFGRIGHT_D0		MOTOR[MRIGHT].f0ch
#define CFGRIGHT_D1		MOTOR[MRIGHT].f1ch
#define CFGKICK_D0		MOTOR[MKICK].f0ch
#define CFGKICK_D1		MOTOR[MKICK].f1ch

#define CFGRIGHT_PWM		MOTOR[MRIGHT].pwmch
#define CFGRIGHT_QD1		MOTOR[MRIGHT].qd1ch
#define CFGRIGHT_QD2		MOTOR[MRIGHT].qd2ch
#define CFGRIGHT_TYPE_D0	MOTOR[MRIGHT].f0type
#define CFGRIGHT_ID_D0		MOTOR[MRIGHT].f0id
#define CFGRIGHT_TYPE_D1	MOTOR[MRIGHT].f1type
#define CFGRIGHT_ID_D1		MOTOR[MRIGHT].f1id

#define CFGLEFT_PWM	MOTOR[MLEFT].pwmch
#define CFGLEFT_QD1	MOTOR[MLEFT].qd1ch
#define CFGLEFT_QD2	MOTOR[MLEFT].qd2ch
#define CFGLEFT_TYPE_D0	MOTOR[MLEFT].f0type
#define CFGLEFT_ID_D0	MOTOR[MLEFT].f0id
#define CFGLEFT_TYPE_D1	MOTOR[MLEFT].f1type
#define CFGLEFT_ID_D1  	MOTOR[MLEFT].f1id

#define CFGKICK_PWM	MOTOR[MKICK].pwmch
#define CFGKICK_QD1	MOTOR[MKICK].qd1ch
#define CFGKICK_QD2	MOTOR[MKICK].qd2ch
#define CFGKICK_TYPE_D0	MOTOR[MKICK].f0type
#define CFGKICK_ID_D0	MOTOR[MKICK].f0id
#define CFGKICK_TYPE_D1	MOTOR[MKICK].f1type
#define CFGKICK_ID_D1  	MOTOR[MKICK].f1id



/********************************************************************************/
/* definitions for fastbins. not dependent of base */
#ifdef ACCESS_STATIC_MOTOR

#define LEFT_D0 0 
#define LEFT_D1 1  
#define RIGHT_D0 2 
#define RIGHT_D1 3 
#define KICK_D0 4  
#define KICK_D1 5  

/* definitions of the TP channels. base dependent */
#ifdef USE_RGBASE_ONE
#define BASE_DEFINED

#define LEFT_PWM 0  
#define LEFT_QD1 1
#define LEFT_QD2 2 
#define LEFT_TYPE_D0 FBTYPE_TPU
#define LEFT_ID_D0 9
#define LEFT_TYPE_D1 FBTYPE_CPU
#define LEFT_ID_D1 FBCPU_PCS1


#define RIGHT_PWM 3
#define RIGHT_QD1 5
#define RIGHT_QD2 4 
#define RIGHT_TYPE_D0 FBTYPE_TPU
#define RIGHT_ID_D0 13
#define RIGHT_TYPE_D1 FBTYPE_TPU
#define RIGHT_ID_D1 12

#endif

#ifdef USE_RGBASE_TWO
#define BASE_DEFINED

#define RIGHT_PWM 0  
#define RIGHT_QD1 2
#define RIGHT_QD2 1 
#define RIGHT_TYPE_D0 FBTYPE_RCJ
#define RIGHT_ID_D0 0
#define RIGHT_TYPE_D1 FBTYPE_RCJ
#define RIGHT_ID_D1 1


#define LEFT_PWM 3
#define LEFT_QD1 4
#define LEFT_QD2 5 
#define LEFT_TYPE_D0 FBTYPE_RCJ
#define LEFT_ID_D0 3
#define LEFT_TYPE_D1 FBTYPE_RCJ
#define LEFT_ID_D1 2

#endif

#ifdef USE_RCJBASE
#define BASE_DEFINED

#define RIGHT_PWM 3  
#define RIGHT_QD1 5
#define RIGHT_QD2 4 
#define RIGHT_TYPE_D0 FBTYPE_RCJ
#define RIGHT_ID_D0 4
#define RIGHT_TYPE_D1 FBTYPE_RCJ
#define RIGHT_ID_D1 5


#define LEFT_PWM 6
#define LEFT_QD1 7
#define LEFT_QD2 8 
#define LEFT_TYPE_D0 FBTYPE_RCJ
#define LEFT_ID_D0 3
#define LEFT_TYPE_D1 FBTYPE_RCJ
#define LEFT_ID_D1 2

#endif


/* definitions of the TP channels. base dependent */
#ifdef USE_RC99BASE
#define BASE_DEFINED

#define LEFT_PWM 0  
#define LEFT_QD1 1
#define LEFT_QD2 2 
#define LEFT_TYPE_D0 FBTYPE_CPU
#define LEFT_ID_D0 FBCPU_PCS1
#define LEFT_TYPE_D1 FBTYPE_TPU
#define LEFT_ID_D1 9


#define RIGHT_PWM 3
#define RIGHT_QD1 4
#define RIGHT_QD2 5 
#define RIGHT_TYPE_D0 FBTYPE_TPU
#define RIGHT_ID_D0 13
#define RIGHT_TYPE_D1 FBTYPE_TPU
#define RIGHT_ID_D1 12

#define KICK_PWM 6 
#define KICK_TYPE_D0 FBTYPE_TPU
#define KICK_ID_D0 14
#define KICK_TYPE_D1 FBTYPE_TPU
#define KICK_ID_D1 15

#endif

#ifdef USE_RC98BASE
#define BASE_DEFINED

#define LEFT_PWM 0  
#define LEFT_QD1 1
#define LEFT_QD2 12 /* TP Ext */
#define LEFT_TYPE_D0 FBTYPE_DUART 
#define LEFT_ID_D0 2 /* OP2 DUART */
#define LEFT_TYPE_D1 FBTYPE_DUART
#define LEFT_ID_D1 3 /* OP3 DUART */

#define RIGHT_PWM 4
#define RIGHT_QD1 5
#define RIGHT_QD2 13 /* TP Ext */
#define RIGHT_TYPE_D0 FBTYPE_DUART
#define RIGHT_ID_D0 6 /* OP6 DUART */
#define RIGHT_TYPE_D1 FBTYPE_DUART
#define RIGHT_ID_D1 7 /* OP7 DUART */

#define KICK_PWM 2  
#define KICK_TYPE_D0 FBTYPE_DUART
#define KICK_ID_D0 4 /* OP4 DUART */
#define KICK_TYPE_D1 FBTYPE_DUART
#define KICK_ID_D1 5 /* OP5 DUART */

#endif

#ifdef USE_SMBIIBASE
#define BASE_DEFINED

#define LEFT_PWM 6  
#define LEFT_QD1 7
#define LEFT_QD2 10 /* TP Ext */
#define LEFT_TYPE_D0 FBTYPE_DUART 
#define LEFT_ID_D0 2 /* OP2 DUART */
#define LEFT_TYPE_D1 FBTYPE_DUART
#define LEFT_ID_D1 3 /* OP3 DUART */

#define RIGHT_PWM 8
#define RIGHT_QD1 9
#define RIGHT_QD2 11 /* TP Ext */
#define RIGHT_TYPE_D0 FBTYPE_DUART
#define RIGHT_ID_D0 4 /* OP6 DUART */
#define RIGHT_TYPE_D1 FBTYPE_DUART
#define RIGHT_ID_D1 5 /* OP7 DUART */


#endif

#ifndef BASE_DEFINED
#error No base defined ! Check makefile
#endif

#else // static motor


#define LEFT_D0 CFGLEFT_D0
#define LEFT_D1 CFGLEFT_D1 
#define RIGHT_D0 CFGRIGHT_D0 
#define RIGHT_D1 CFGRIGHT_D1 
#define KICK_D0 CFGKICK_D0 
#define KICK_D1 CFGKICK_D1 

#define RIGHT_PWM CFGRIGHT_PWM 
#define RIGHT_QD1 CFGRIGHT_QD1 
#define RIGHT_QD2 CFGRIGHT_QD2 
#define RIGHT_TYPE_D0 CFGRIGHT_TYPE_D0 
#define RIGHT_ID_D0 CFGRIGHT_ID_D0 
#define RIGHT_TYPE_D1 CFGRIGHT_TYPE_D1 
#define RIGHT_ID_D1 CFGRIGHT_ID_D1 

#define LEFT_PWM CFGLEFT_PWM 
#define LEFT_QD1 CFGLEFT_QD1 
#define LEFT_QD2 CFGLEFT_QD2 
#define LEFT_TYPE_D0 CFGLEFT_TYPE_D0 
#define LEFT_ID_D0 CFGLEFT_ID_D0 
#define LEFT_TYPE_D1 CFGLEFT_TYPE_D1 
#define LEFT_ID_D1 CFGLEFT_ID_D1 

#define KICK_PWM CFGKICK_PWM 
#define KICK_QD1 CFGKICK_QD1 
#define KICK_QD2 CFGKICK_QD2 
#define KICK_TYPE_D0 CFGKICK_TYPE_D0
#define KICK_ID_D0 CFGKICK_ID_D0 
#define KICK_TYPE_D1 CFGKICK_TYPE_D1 
#define KICK_ID_D1 CFGKICK_ID_D1  

#endif

#endif

void MOTOR_set_pid(int motor,float p, float i, float d, float leaky_q);
void MOTOR_set_speedscale(int motor,float scale);
void MOTOR_pid_speed(int motor, int speed);

void _MOTOR_setPW(int motor, int newPW);
int _MOTOR_encoder(int motor);

short MOTOR_get_qdec(int motor);

void MOTOR_set_directionbits(int motor);
void MOTOR_set_frequency(int motor,int pulse_frequency);
void MOTOR_set_tpchannels(int motor,int pulse_frequency);
void MOTOR_init(int motor,int pulse_frequency);
int MOTOR_set_config(int motor,
		 unsigned char pwmch,unsigned char qd1ch,unsigned char qd2ch,
		 char f0type, char f0id,char f1type,char f1id,
		 float p,float i,float d,float leaky_q,int speedscale,int p2s);

#endif



