#include <math.h>
#include <stdio.h>
#include <ttyio.h>
#include "sincos.h"
#include "fixpoint.h"

int sintab[SINCOS_MAX_ANGLE/SINCOS_RES+1];
int costab[SINCOS_MAX_ANGLE/SINCOS_RES+1];
int atantab[C_FPOINT/ATAN_RES+1];
int sqrttab[2*C_FPOINT/SQRT_RES+1];

int tri_init_sincos = -1;
int tri_init_atan = -1;
int tri_init_sqrt = -1;


void TRI_init_sqrt () {
	int i;

	for(i=0;i<=2*C_FPOINT/SQRT_RES;i++) {
		sqrttab[i] = (int)(sqrt((double)i*SQRT_RES/(double)C_FPOINT)*C_FPOINT);	
		TTY_outchar ('.');
	}
	tri_init_sqrt = 1;
//for (i=0;i<= 2*C_FPOINT/SQRT_RES;i++)
//printf ("i=%d, \tsqrt=%d \n",i,sqrttab[i]);
	printf("\n");
}

void TRI_init_atan () {
	int i;

	for (i = 0; i <= C_FPOINT/ATAN_RES; i++) {
		atantab[i] = (int) ( atan( (double)i*ATAN_RES/(double)C_FPOINT )
					/ (2.0*M_PI) * SINCOS_MAX_ANGLE );
		TTY_outchar ('.');
	}
	tri_init_atan = 1;
//for (i=0;i<= C_FPOINT/ATAN_RES;i++)
//printf ("i=%d, \tatan=%d \n",i,atantab[i]);
	printf("\n");
}

/*!
\brief initialization of the trigonometric lookup-tables.

the resolution is the fix-point resolution squared (2x number of digits)
*/
void TRI_init_sincos () {

	int i;

	for (i = 0; i <= SINCOS_MAX_ANGLE/SINCOS_RES; i++) {
		sintab[i] = sin(i/(double)(SINCOS_MAX_ANGLE/SINCOS_RES) * M_PI * 2.0) * C_FPOINT_SQ;
		costab[i] = cos(i/(double)(SINCOS_MAX_ANGLE/SINCOS_RES) * M_PI * 2.0) * C_FPOINT_SQ;
		TTY_outchar ('.');
	}

//for (i=0;i<SINCOS_MAX_ANGLE/SINCOS_RES;i++)
//      printf ("i=%d, \tsin=%d \tcos=%d \n",i,sintab[i],costab[i]);
	printf("\n");

	tri_init_sincos = 1;
}


/*! 
\brief computes sin(x) via fixpoint arithmetic and lookup table

output of TRI_sin and TRI_cos are normalized to +/-C_FPOINT 
both expect values measured in centidegrees and smaller then MAX_ANGLE */
int TRI_sin (int x)
{
	long long y1;
	long long y2;
	long long result;
	int where;

//printf("\t in TRI_sin");
//printf("\tx: %d ",x);

	if(x >= SINCOS_MAX_ANGLE){
		printf("sERROR: angle (=%d) in TRI_sin larger than MAX_ANGLE (=%d)\n",x,SINCOS_MAX_ANGLE);
		return(-1);
	}

//printf("\tx: %d ",x);
//printf("\n");

	where = x / SINCOS_RES;

	y1 = sintab[where];
	y2 = sintab[where + 1];

	result = (x % 100) * ((y2 - y1) / 100) + y1; 
	result = result / C_FPOINT;
	return ((int)result);
}


/*! 
\brief computes cos(x) via fixpoint arithmetic and lookup table

output of TRI_sin and TRI_cos are normalized to +/-C_FPOINT 
both expect values measured in centidegrees and smaller then MAX_ANGLE */
int TRI_cos (int x) {
	long long y1;
	long long y2;
	long long result;
	int where;

//printf("\t in TRI_cos");
//printf("\tx: %d ",x);

	if(x >= SINCOS_MAX_ANGLE){
		printf("cERROR: angle (=%d) in TRI_cos larger than MAX_ANGLE (=%d)\n",x,SINCOS_MAX_ANGLE);
		return(-1);
	}

//printf("\tx: %d ",x);
//printf("\n");

	where = x / SINCOS_RES;

	y1 = costab[where];
	y2 = costab[where + 1];

	result = (x % 100) * ((y2 - y1) / 100) + y1; 
	result = result / C_FPOINT;
	return ((int)result);
}

/*! expects a value between 0 and 2*C_FPOINT (virtual [0.0;2.0]) and returns
its square-root scaled by C_FPOINT */
int TRI_sqrt (int x) {
	int y1;
	int y2;
	int result;
	int where;

	if((x > 2*C_FPOINT)||(x < 0)){
		printf("rERROR: x (=%d) in TRI_sqrt larger than 2*C_FPOINT (=%d) or negative !!!\n",x,2*C_FPOINT);
		return(-1);
	}

	where = x / SQRT_RES;

	y1 = sqrttab[where];
	y2 = sqrttab[where + 1];

	result = (x % 100) * ((y2 - y1) / 100) + y1; 
	return (result);
}

/*! expects a value between 0 and C_FPOINT (virtual [0.0;1.0]) and returns the
arcustangens in centideg */
int TRI_atan(int x) {
	int y1;
	int y2;
	int result;
	int where;

	if((x > C_FPOINT)||(x < 0)){
		printf("aERROR: x (=%d) in TRI_atan larger than C_FPOINT (=%d) or negative !!!\n",x,C_FPOINT);
		return(-1);
	}

	where = x / ATAN_RES;

	y1 = atantab[where];
	y2 = atantab[where + 1];

	result = (x % 100) * ((y2 - y1) / 100) + y1; 
	return (result);
}

/*!
\brief returns the length of a vector in mm
\param dx in mm
\param dy in mm

TRI_vector_length expects dx(mm), dy(mm); it returns the length (mm) 
*/
int TRI_vector_length(int dx, int dy) {
	long long hx, hy, hl;

	if(abs(dx) > abs(dy)) {
		hy = (dy * (long long)C_FPOINT)/(long long)dx;
		hy = (hy*hy) /(long long)C_FPOINT; 
		hl = (long long) TRI_sqrt(C_FPOINT + (int)hy);
		hl = hl * (long long)abs(dx);
		return((int)(hl/C_FPOINT));
	} else {
		hx = (dx * (long long)C_FPOINT)/(long long)dy;
		hx = (hx*hx) /(long long)C_FPOINT; 
		hl = (long long) TRI_sqrt(C_FPOINT + (int)hx);
		hl = hl * (long long)abs(dy);
		return((int)(hl/C_FPOINT));
	}

	return(0);
}


/*!
\brief computes angle from dx and dy
\param dx in mm
\param dy in mm

TRI_vector_angle expects x/y-lengths (in mm) and returns an angle (in centideg) 
*/
int TRI_vector_angle(int dx, int dy) {
	long long h;
	int ang;
	int adx, ady, sdx, sdy;

	if(dx == 0) {
		if(dy > 0) return(9000);
		if(dy < 0) return(27000);
	}
	if(dy == 0) {
		if(dx >= 0) return(0);
		if(dx < 0) return(18000);
	}

	sdx = SIGN(dx);
	sdy = SIGN(dy);
	adx = abs(dx);
	ady = abs(dy);

	if((sdx == 1) && (sdy == 1)) {
		if(adx>ady) {
			h = (ady * (long long)C_FPOINT) / (long long)adx;
			ang = TRI_atan((int)h);
		}else{
			h = (adx * (long long)C_FPOINT) / (long long)ady;
			ang = 9000 - TRI_atan((int)h);
		}
		return(ang);
	} 
			
	if((sdx == -1) && (sdy == 1)) {
		if(adx>ady) {
			h = (ady * (long long)C_FPOINT) / (long long)adx;
			ang = 18000 - TRI_atan((int)h);
		}else{
			h = (adx * (long long)C_FPOINT) / (long long)ady;
			ang = 9000 + TRI_atan((int)h);
		}
		return(ang);
	} 
			
	if((sdx == -1) && (sdy == -1)) {
		if(adx>ady) {
			h = (ady * (long long)C_FPOINT) / (long long)adx;
			ang = 18000 + TRI_atan((int)h);
		}else{
			h = (adx * (long long)C_FPOINT) / (long long)ady;
			ang = 27000 - TRI_atan((int)h);
		}
		return(ang);
	} 
			
	if((sdx == 1) && (sdy == -1)) {
		if(adx>ady) {
			h = (ady * (long long)C_FPOINT) / (long long)adx;
			ang = 36000 - TRI_atan((int)h);
		}else{
			h = (adx * (long long)C_FPOINT) / (long long)ady;
			ang = 27000 + TRI_atan((int)h);
		}
		return(ang);
	} 
			
	return(-1);
}

int TRI_check_sincos() {
	return(tri_init_sincos);
}
int TRI_check_atan() {
	return(tri_init_atan);
}
int TRI_check_sqrt() {
	return(tri_init_sqrt);
}

