/** WirelessWInspect Source Code
 * Author: Thomas Foss (tafoxx@gmail.com)
 * Date: 29-7-08
 * Version: 2.9
 * License: ???
 */

/** Program Notes
 * >>The comments in this program are roughly organized in order of importance.
 *   Comments formatted with with the double-asterisk (as this one is) are more
 *   important than double-slash comments. Single-asterisk commenting is used
 *   for temporary disabling sections of code, so there shouldn't be any in the
 *   finished product. This was done for convenience, because the different
 *   types of commenting is colored differently in KWrite.
 * >>This program is split into two source files, "final.c" and "functions.c". SDCC
 *   will write final.c into the first code page of the Pic, and functions.c into
 *   the second code page. Each code page has room for approximately 2000 instructions.
 *   The Makefile assembles them into a single hex file. Function prototypes and
 *   variable declarations with the keyword "extern" must be present in a source
 *   file in order to use variables declared in the other source file. Also, macros
 *   must be redefined in each source file to be used within that file.
 * >>The sound settings should probably be saved in EEPROM, so that they are retained
 *   when the device resets. I didn't have time to implement this, though.
 * >>I added the command "DISABLEALL", for convenience, which simply disables all the
 *   functions of the device.
 */

#define MAJORVERSION  2
#define MINORVERSION  9

//Prevents bits from being addressed independently of their register bytes.
#define NO_BIT_DEFINES
#include <pic14regs.h>

/** General Purpose Defines */
#define FOSC 14745600
#define SPBRGVAL(b) ((FOSC)/64/(b)-1)
#define TMR0RATE   (255+2-(FOSC/4/2/9600))
//After setting TMR0, Timer 0 is not incremented for 2 clock cycles. Clock rate
//is 14.7456 MHz. PIC16F88 executes 1 instruction cycle every 4 clock cycles,
//prescaler is at 1:2, and desired timer overflow rate is 9600 Hz.

#define SETBIT(p,b)	(p) |= 1<<(b)
#define CLEARBIT(p,b)	(p) &= ~(1<<(b))
#define TOGGLEBIT(p,b)	(p) ^= 1<<(b)
#define CHECKBIT(p,b)	((p & (1<<(b))) > 0 ? 1:0)
/** End General Purpose Defines */

/** Program Flags:
 * Bit 0 (TRANSMITSTATE): Determines whether the next transmission is a type byte (cleared)
 *   or a data byte (set). Toggled with every transmission (in normal mode).
 * Bit 1 (MSBX): Stores the most significant bit of Accelerometer X's data.
 * Bit 2 (MSBY): Stores the most significant bit of Accelerometer Y's data.
 * Bit 3 (MSBZ): Stores the most significant bit of Accelerometer Z's data.
 * Bit 4 (NEWRFIDDATA): RFID data is only sent when a new value is received (so that even
 *   if that value is 0, it will still show up as a newly read byte).
 * Bit 5 (ADRUNNING): Keeps a note of whether the A/D Converter is busy, or doing nothing.
 *   (This is important, because there is a pause between setting the channel and actually
 *   running a conversion, during which time the GO/DONE bit won't be set.)
 */
volatile unsigned char programFlags = 0;
#define TRANSMITSTATE      0
#define MSBX               1
#define MSBY               2
#define MSBZ               3
#define NEWRFIDDATA        4
#define ADRUNNING          5

#define SETPROGRAMFLAG(f)       programFlags |= 1<<(f)
#define CLEARPROGRAMFLAG(f)     programFlags &= ~(1<<(f))
#define TOGGLEPROGRAMFLAG(f)    programFlags ^= 1<<(f)
#define CHECKPROGRAMFLAG(f)     ((programFlags & (1<<(f))) > 0 ? 1:0)
/** End Program Flags definitions. */

/** Sound Command Flags (Contains flags to keep track of receiving keywords for Sound
 *    and Sound choices):
 * Bit 0: Set when SOUND has been received.
 * Bit 1: Set when CONN has been received.
 * Bit 2: Set when DISCONN has been received.
 * Bit 3: Set when RFID has been received.
 * Bit 4: Set when BUTTON has been received.
 * Bit 5: Set when ATTENTION has been received.
 * Bit 6: Set when a value for the frequency has been received.
 * Bit 7: Set when a value for the duration has been received.
 */
volatile unsigned char soundCommandFlags = 0;
#define SOUNDRECEIVED      0
#define CONNRECEIVED       1
#define DISCONNRECEIVED    2
#define RFIDRECEIVED       3
#define BUTTONRECEIVED     4
#define ATTENTIONRECEIVED  5
#define FREQRECEIVED       6
#define DURATIONRECEIVED   7

#define SETSOUNDCOMMANDFLAG(p)     soundCommandFlags |= 1<<(p)
#define CLEARSOUNDCOMMANDFLAG(p)   soundCommandFlags &= ~(1<<(p))
#define TOGGLESOUNDCOMMANDFLAG(p)  soundCommandFlags ^= 1<<(p)
#define CHECKSOUNDCOMMANDFLAG(p)   ((soundCommandFlags & (1<<(p))) > 0 ? 1:0)
/** End Sound Command Flags definitions */

/** Protocol Command Flags (Contains flags to keep track of receiving keywords for Protocol
 *    and Protocol choices):
 * Bit 0: Set when PROTOCOL has been received.
 * Bit 1: Set when SENDX has been received.
 * Bit 2: Set when SENDY has been received.
 * Bit 3: Set when SENDZ has been received.
 * Bit 4: Set when SENDB has been received.
 * Bit 5: Set when SENDRFID has been received.
 * Bit 6: Set when SENDTYPE has been received.
 * Bit 7: Set when SENDAUXIN has been received.
 */
volatile unsigned char protocolCommandFlags = 0;
#define PROTOCOLRECEIVED   0
#define SENDXRECEIVED      1
#define SENDYRECEIVED      2
#define SENDZRECEIVED      3
#define SENDBRECEIVED      4
#define SENDRFIDRECEIVED   5
#define SENDTYPERECEIVED   6
#define SENDAUXINRECEIVED  7

#define SETPROTOCOLCOMMANDFLAG(p)    protocolCommandFlags |= 1<<(p)
#define CLEARPROTOCOLCOMMANDFLAG(p)  protocolCommandFlags &= ~(1<<(p))
#define TOGGLEPROTOCOLCOMMANDFLAG(p) protocolCommandFlags ^= 1<<(p)
#define CHECKPROTOCOLCOMMANDFLAG(p)  ((protocolCommandFlags & (1<<(p))) > 0 ? 1:0)
/** End Protocol Command Flags definitions */

/** Protocol Toggles (Transmissions enabled when bit set):
 * Bit 0: Send type byte.
 * Bit 1: Send X data.
 * Bit 2: Send Y data.
 * Bit 3: Send Z data.
 * Bit 4: Send button data.
 * Bit 5: Send RFID data.
 * Bit 6: Send Auxillary A/D data.
 */
//These bits keep track of which transmissions are enabled or disabled with the
//PROTOCOL command. Default value enables the sending of the type byte, X and Y
//values, button data, and RFID data.
volatile unsigned char protocolToggles = 1|2|4|16|32;
#define SENDTYPEDATA      0
#define SENDXDATA         1
#define SENDYDATA         2
#define SENDZDATA         3
#define SENDBUTTONSDATA   4
#define SENDRFIDDATA      5
#define SENDAUXDATA       6

#define SETPROTOCOL(p)          protocolToggles |= 1<<(p)
#define CLEARPROTOCOL(p)        protocolToggles &= ~(1<<(p))
#define TOGGLEPROTOCOL(p)       protocolToggles ^= 1<<(p)
#define CHECKPROTOCOL(p)        ((protocolToggles & (1<<(p))) > 0 ? 1:0)
/** End Protocol Toggles definitions. */

//A container for the incoming value from the RFID reader.
volatile unsigned char rfidBuffer = 0;
//A container for incoming commands from the USART.
volatile unsigned char commandBuffer[16];
//A checksum for the incoming commands from the USART.
volatile unsigned char commandChecksum;

/** Counters */
//A counter to keep track of bits coming in from the RFID reader. Counts up to 8.
volatile unsigned char rfidCounter = 0;
//A counter to keep track of which A/D data is being handled. 0=X, 1=Y, 2=Z, and 4=Aux.
volatile unsigned char adCounter = 0;
//A counter to keep track of how many characters have been received from the serial
//bluetooth connection. Resets whenever whitespace is received, or when more than
//29 characters have been received without any whitespace.
volatile unsigned char commandCounter = 0;
//A counter to keep track of the frequency of the sounds. Increments each time the Timer
//1 interrupt goes off. The buzzer pin is toggled every time this counter reaches the
//value in soundFreqValue.
volatile unsigned char soundFreqCounter = 0;
//A counter to keep track of the duration of the sounds. Increments each time soundFreqCounter
//reaches soundFreqValue. When this counter reaches the value in soundDurationValue,
//the timer interrupt is disabled and the sound ends.
volatile unsigned char soundDurationCounter = 0;
/** End Counters */

/** Sound Variables */
volatile unsigned char connFreqValue = 20;
volatile unsigned char connDurationValue = 127;
volatile unsigned char disconnFreqValue = 3;
volatile unsigned char disconnDurationValue = 127;
volatile unsigned char rfidFreqValue = 6;
volatile unsigned char rfidDurationValue = 127;
volatile unsigned char buttonFreqValue = 3;
volatile unsigned char buttonDurationValue = 127;
volatile unsigned char attentionFreqValue = 15;
volatile unsigned char attentionDurationValue = 127;

volatile unsigned char soundFreqValue = 0; //When a sound goes off, the Timer 1
//interrupt is turned on, and the appropriate frequency value is copied into
//this variable. The appropriate duration value is copied into soundDurationCounter,
//to count down.
/** End Sound Variables */


//Keeps track of which data is being transmitted. The type byte is also sent before
//each transmission, to indicate the type of data. Values range from 0x41 to 0x49.
//For 9-bit acceleration values, the type byte signifies the most significant bit.
/** Type Byte Values:
 * 0x41:  X acceleration (MSB low)
 * 0x42:  X acceleration (MSB high)
 * 0x43:  Y acceleration (MSB low)
 * 0x44:  Y acceleration (MSB high)
 * 0x45:  Button states
 * 0x46:  RFID data
 * 0x47:  Auxillary A/D data
 * 0x48:  Z acceleration (MSB low)
 * 0x49:  Z acceleration (MSB high)
 */
volatile unsigned char typeByte = 0x41;
//Contains the next value to be transmitted. This is updated according to the
//state of the type byte.
volatile unsigned char dataByte = 0;

//Variables to hold the data, ready to be transmitted. The values in these variables will
//remain until new values are available, or until transmission of that type of data is
//disabled (when they will be cleared).
volatile unsigned char dataX = 0;
volatile unsigned char dataY = 0;
volatile unsigned char dataZ = 0;
volatile unsigned char dataButtons = 0;
volatile unsigned char dataRFID = 0;
volatile unsigned char dataAux = 0;

/** Function Prototypes */
extern void initializeTimer0();
extern void initializeBluetooth();
extern void initializeTRISbits();
extern void initializeADConverter();
extern void startAD();
extern void checkADConverter();
extern void startSound(unsigned char, unsigned char);
extern void startRFID();
extern void enableRFID();
extern void disableRFID();
extern void enableAux();
extern void disableAux();
extern void disableAll();
extern void oneWire();
extern void versionMajor();
extern void versionMinor();
extern void programMode();

volatile unsigned char temp = 0;
volatile unsigned char interruptTemp = 0; //Could run into problems if same temporary
                //register is accessed in interrupt while it's being used in the main.
volatile unsigned char i = 0;



void Interrupt(void) __interrupt 0
{
  if(PIR1_bits.TMR2IF && PIE1_bits.TMR2IE)
  {
    ADCON0_bits.GO = 1; //Start the conversion.
    PIE1_bits.TMR2IE = 0; //Disable Timer 2 interrupt.
    PIR1_bits.TMR2IF = 0; //Clear Timer 2 flag.
    T2CON_bits.TMR2ON = 0; //Turn off Timer 2.
  }
  
  if(INTCON_bits.TMR0IF && INTCON_bits.TMR0IE) //Timer 0 Interrupt enabled only within
  {                                            //the Port B change interrupt (RBIF).
    TMR0 += TMR0RATE; //Set value in Timer 0 register to cause it to
                      //overflow at 9600 Hz.
    if(rfidCounter < 9) //When less than 9 bits have been received. (The 9th is the
    {                   //start bit, which gets bumped out by the last bit shift.)
      //Add the new bit onto the variable.
      rfidBuffer = rfidBuffer>>1;
      if(CHECKBIT(PORTB,7)) //Is the bit high...
        SETBIT(rfidBuffer,7);
      else                  //... or low?
        CLEARBIT(rfidBuffer,7);
      rfidCounter++;
    }
    else //When the last bit has been received.
    {
      if(CHECKBIT(PORTB,7)) //As long as the stop bit is high, as it should be.
      {
        rfidCounter = 0;
        INTCON_bits.TMR0IE = 0; //Disable Timer 0 interrupt.
        startRFID();
        dataRFID = rfidBuffer;
        rfidBuffer = 0;
        SETPROGRAMFLAG(NEWRFIDDATA);
      }
      else //If the stop bit is not high, something went wrong, and the current
      {    //transmission is scrapped.
        rfidBuffer = 0;
        rfidCounter = 0;
        INTCON_bits.TMR0IE = 0; //Disable Timer 0 interrupt.
        startRFID();
      }
    }
    INTCON_bits.TMR0IF = 0; //Clear Timer 0 flag.
  }
  
  if(INTCON_bits.RBIE) //This interrupt is enabled only when the program is
  {                    //waiting for the start bit of the RFID data.
    if(INTCON_bits.RBIF)
    {
      interruptTemp = PORTB;
      INTCON_bits.RBIF = 0;
    }
    if(!CHECKBIT(PORTB,7)) //When the line is low (for the start bit).
    {
      //Start listening for transmission:
      //--Port B change interrupt disabled.
      //--Timer 0 interrupt is enabled, to poll RFID reader at the baud rate
      //  (9600 times a second) (Timer 0 flag is cleared first).
      //--Timer 0 counter is set to 175 here, so the first overflow will occur
      //  approximately halfway through the start bit.
      TMR0 = 175;
      INTCON_bits.TMR0IF = 0;
      INTCON_bits.TMR0IE = 1;
      INTCON_bits.RBIF = 0;
      INTCON_bits.RBIE = 0;
      //The contents of startSound(), placed inline here for optimization.
      if(!CHECKBIT(rfidDurationValue,7)) //If the duration value is less than 128.
      {
        soundFreqValue = rfidFreqValue;
        soundDurationCounter = rfidDurationValue;
        if(!T1CON_bits.TMR1ON) //If a sound is already playing, don't reset the timer.
        {
          T1CON_bits.TMR1ON = 1; //Turn on Timer 1.
          TMR1H = 0xFD;  //This 16-bit value will cause Timer 1 to overflow at 6400 Hz.
          TMR1L = 0xBF;
          PIR1_bits.TMR1IF = 0; //Clear flag first.
          PIE1_bits.TMR1IE = 1; //Enable Timer 1 interrupts.
        }
      }
    }
  }
  
  if(PIR1_bits.TMR1IF && PIE1_bits.TMR1IE) //This interrupt toggles the buzzer pin
  { //(Port B, pin 3) at the given frequency and duration (which are copied into
    //their respective variables whenever this interrupt is enabled.
    TMR1H = 0xFD;  //This 16-bit value will cause Timer 1 to overflow at 6400 Hz.
    TMR1L += 0xBF;
    
    if(!soundFreqCounter)
    {
      soundDurationCounter++;
      soundFreqCounter = soundFreqValue;
      TOGGLEBIT(PORTB,3);
    }
    else
    {
      soundFreqCounter--;
    }
    
    if(!soundDurationCounter)
    {
      PIE1_bits.TMR1IE = 0; //Sound goes off.
      CLEARBIT(PORTB,3); //Light goes off.
      soundFreqCounter = 0;
      soundFreqValue = 0;
      T1CON_bits.TMR1ON = 0; //Turn off Timer 1.
    }
    
    PIR1_bits.TMR1IF = 0; //Clear the flag.
  }
}

void main()
{
  //Initialize Bluetooth Serial Port.
  initializeBluetooth();

  CLEARBIT(OPTION_REG,7); //Enable internal pull-ups for Port B.
  
  //Setting Data Direction Registers for ports (to determine whether they will be
  //used for input or output).
  initializeTRISbits();
  
  SETBIT(PORTB,6); //Turn on LED on accelerometer board.

  //Initialize Timer 0. Note: Timer is not enabled until Port B Change
  //interrupt goes off.
  initializeTimer0();
  
  initializeADConverter();
  
  CLEARBIT(ANSEL,4); //Set Port A, pin 4 (Auxillary A/D) to digital I/O. (Note: This pin
                     //must be set as a digital high bit to turn on the RFID reader if
                     //enabled. When Aux A/D is enabled, this pin will change to analog
                     //input, and the RFID reader will be disabled.
  CLEARBIT(PORTA,4); //Turn on RFID reader.
  startRFID();
  INTCON_bits.PEIE = 1; //Enable peripheral interrupts.
  INTCON_bits.GIE = 1; //Enable global interrupts.

  while(1)
  {
    
    switch(adCounter)
    {
      case 0:
        if(CHECKPROTOCOL(SENDXDATA))
        {
          if(!CHECKPROGRAMFLAG(ADRUNNING))
          {
            ADCON0 &= ~(111<<3);    //Set the correct
            ADCON0 |= adCounter<<3; //input channel.
            SETPROGRAMFLAG(ADRUNNING);
            startAD();
          }
          if(PIR1_bits.ADIF)
          {
            dataX = ADRESH << 1; //First 7 bits of the most significant byte.
            if(CHECKBIT(ADRESL,7)) //Add the second-to-least significant bit.
              SETBIT(dataX,0);
            else
              CLEARBIT(dataX,0);
            if(CHECKBIT(ADRESH,7)) //Set the MSB.
              SETPROGRAMFLAG(MSBX);
            else
              CLEARPROGRAMFLAG(MSBX);
            
            PIR1_bits.ADIF = 0; //Clear the flag.
            CLEARPROGRAMFLAG(ADRUNNING);
            adCounter++; //Advance to Y.
          }
        }
        else
          adCounter++; //Advance to Y.
        break;
      case 1:
        if(CHECKPROTOCOL(SENDYDATA))
        {
          if(!CHECKPROGRAMFLAG(ADRUNNING)) //Start the conversion if it is not already running.
          {
            ADCON0 &= ~(111<<3);    //Set the correct
            ADCON0 |= adCounter<<3; //input channel.
            SETPROGRAMFLAG(ADRUNNING);
            startAD();
          }
          if(PIR1_bits.ADIF) //When the data has arrived.
          {
            dataY = ADRESH << 1; //First 7 bits of the most significant byte.
            if(CHECKBIT(ADRESL,7)) //Add the second-to-least significant bit.
              SETBIT(dataY,0);
            else
              CLEARBIT(dataY,0);
            if(CHECKBIT(ADRESH,7)) //Set the MSB.
              SETPROGRAMFLAG(MSBY);
            else
              CLEARPROGRAMFLAG(MSBY);
            
            PIR1_bits.ADIF = 0; //Clear the flag.
            CLEARPROGRAMFLAG(ADRUNNING);
            adCounter++; //Advance to Z.
          }
        }
        else
          adCounter++; //Advance to Z.
        break;
      case 2:
        if(CHECKPROTOCOL(SENDZDATA))
        {
          if(!CHECKPROGRAMFLAG(ADRUNNING))
          {
            ADCON0 &= ~(111<<3);    //Set the correct
            ADCON0 |= adCounter<<3; //input channel.
            SETPROGRAMFLAG(ADRUNNING);
            startAD();
          }
          if(PIR1_bits.ADIF)
          {
            dataZ = ADRESH << 1; //First 7 bits of the most significant byte.
            if(CHECKBIT(ADRESL,7)) //Add the second-to-least significant bit.
              SETBIT(dataZ,0);
            else
              CLEARBIT(dataZ,0);
            if(CHECKBIT(ADRESH,7)) //Set the MSB.
              SETPROGRAMFLAG(MSBZ);
            else
              CLEARPROGRAMFLAG(MSBZ);
            
            PIR1_bits.ADIF = 0; //Clear the flag.
            CLEARPROGRAMFLAG(ADRUNNING);
            adCounter = 4; //Advance to Aux.
          }
        }
        else
          adCounter = 4; //Advance to Aux.
        break;
      case 4:
        if(CHECKPROTOCOL(SENDAUXDATA))
        {
          if(!CHECKPROGRAMFLAG(ADRUNNING))
          {
            ADCON0 &= ~(111<<3);    //Set the correct
            ADCON0 |= adCounter<<3; //input channel.
            SETPROGRAMFLAG(ADRUNNING);
            startAD();
          }
          if(PIR1_bits.ADIF)
          {
            dataAux = ADRESH;
            
            PIR1_bits.ADIF = 0; //Clear the flag.
            CLEARPROGRAMFLAG(ADRUNNING);
            adCounter = 0; //Loop back to X.
          }
        }
        else
          adCounter = 0; //Loop back to X.
        break;
    }
    
    if(PIR1_bits.TXIF) //Toggles between sending a type byte and a data byte using a
    {                  //program flag, and cycles through the available data to be sent.
      if(!CHECKPROGRAMFLAG(TRANSMITSTATE)) //When TRANSMITSTATE is clear, type byte is sent.
      {
        //The RFID data is sent when available. If it waits too long, it could
        //get overwritten.
        if((CHECKPROTOCOL(SENDRFIDDATA)) && (CHECKPROGRAMFLAG(NEWRFIDDATA)))
        {
          if(CHECKPROTOCOL(SENDTYPEDATA))
            TXREG = 0x46; //The RFID type byte.
          dataByte = dataRFID;
          CLEARPROGRAMFLAG(NEWRFIDDATA);
          SETPROGRAMFLAG(TRANSMITSTATE);
        }
        else
        {
          switch(typeByte)
          {
            case 0x41:
              if(CHECKPROTOCOL(SENDXDATA))
              {
                if(CHECKPROTOCOL(SENDTYPEDATA))
                {
                  if(CHECKPROGRAMFLAG(MSBX))
                    TXREG = typeByte + 1;
                  else
                    TXREG = typeByte;
                }
                dataByte = dataX;
                SETPROGRAMFLAG(TRANSMITSTATE);
              }
              typeByte = typeByte + 2;
              break;
            case 0x43:
              if(CHECKPROTOCOL(SENDYDATA))
              {
                if(CHECKPROTOCOL(SENDTYPEDATA))
                {
                  if(CHECKPROGRAMFLAG(MSBY))
                    TXREG = typeByte + 1;
                  else
                    TXREG = typeByte;
                }
                dataByte = dataY;
                SETPROGRAMFLAG(TRANSMITSTATE);
              }
              typeByte = typeByte + 2;
              break;
            case 0x45:
              if(CHECKPROTOCOL(SENDBUTTONSDATA))
              {
                if(CHECKPROTOCOL(SENDTYPEDATA))
                  TXREG = typeByte;
                dataButtons = 0;
                if(!CHECKBIT(PORTB,4)) //Button 1 status.
                  SETBIT(dataButtons,0);
                if(!CHECKBIT(PORTB,0)) //Button 2 status.
                  SETBIT(dataButtons,1);
                if(!CHECKBIT(PORTB,1)) //Button 3 status.
                  SETBIT(dataButtons,2);
                
                dataByte = dataButtons;
                SETPROGRAMFLAG(TRANSMITSTATE);
              }
              typeByte = typeByte + 2;
              break;
// RFID data gets priority. Old code:
//            case 0x46:
//              if(CHECKPROTOCOL(SENDRFIDDATA))
//              {
//                if(CHECKPROGRAMFLAG(NEWRFIDDATA))
//                {
//                  if(CHECKPROTOCOL(SENDTYPEDATA))
//                    TXREG = typeByte;
//                  dataByte = dataRFID;
//                  CLEARPROGRAMFLAG(NEWRFIDDATA);
//                  SETPROGRAMFLAG(TRANSMITSTATE);
//                }
//              }
//              typeByte++;
//              break;
            case 0x47:
              if(CHECKPROTOCOL(SENDAUXDATA))
              {
                if(CHECKPROTOCOL(SENDTYPEDATA))
                  TXREG = typeByte;
                dataByte = dataAux;
                SETPROGRAMFLAG(TRANSMITSTATE);
              }
              typeByte++;
              break;
            case 0x48:
              if(CHECKPROTOCOL(SENDZDATA))
              {
                if(CHECKPROTOCOL(SENDTYPEDATA))
                {
                  if(CHECKPROGRAMFLAG(MSBZ))
                    TXREG = typeByte + 1;
                  else
                    TXREG = typeByte;
                }
                dataByte = dataZ;
                SETPROGRAMFLAG(TRANSMITSTATE);
              }
              typeByte = 0x41;
              break;
          }
        }
      }
      else //When TRANSMITSTATE is set, a data byte is sent.
      {
        TXREG = dataByte;
        CLEARPROGRAMFLAG(TRANSMITSTATE);
      }
    }
    
    if(PIR1_bits.RCIF)
    {
      commandBuffer[15] = RCREG; //A temporary storage place for the current value
                          //(because RCREG might change as soon as it's read once).
      if(commandBuffer[15] == ' ' || commandBuffer[15] == '\n'
         || commandBuffer[15] == '\r')
      {
        if(commandCounter > 0) //Don't start processing if only whitespace has been
        {
          if(!soundCommandFlags && !protocolCommandFlags) //No keywords previously
          {                                               //received.
            switch(commandCounter) //The string is only checked if it is the right
            {  //length for acceptable input. If the string matches, the appropriate
               //actions are taken, or subroutines called, in this switch.
              case 5: //Could be "1WIRE" or "SOUND".
                if(commandChecksum == 137) //If is "SOUND".
                  SETSOUNDCOMMANDFLAG(SOUNDRECEIVED);
                if(commandChecksum == 104) //If is "1WIRE".
                  oneWire();
                break;
              case 7: //Could be "PROGEXT" or "CONNECT".
                if(commandChecksum == 41) //If is "PROGEXT".
                  programMode(); //Exits to Program Mode.
                if(commandChecksum == 10) //If is "CONNECT".
                  startSound(connFreqValue,connDurationValue);
                break;
              case 8: //Could be "PROTOCOL".
                if(commandChecksum == 114) //If is "PROTOCOL".
                  SETPROTOCOLCOMMANDFLAG(PROTOCOLRECEIVED);
                break;
              case 9: //Could be "ATTENTION".
                if(commandChecksum == 182) //If is "ATTENTION".
                  startSound(attentionFreqValue,attentionDurationValue);
                break;
              case 10: //Could be "DISABLEALL" OR "DISCONNECT".
                if(commandChecksum == 205) //If is "DISABLEALL".
                  disableAll();
                if(commandChecksum == 234) //If is "DISCONNECT".
                  startSound(disconnFreqValue,disconnDurationValue);
                break;
              case 12: //Could be "VERSIONMAJOR" or "VERSIONMINOR".
                if(commandChecksum == 159) //If is "VERSIONMAJOR".
                  versionMajor();
                if(commandChecksum == 171) //If is "VERSIONMINOR".
                  versionMinor();
                break;
            }
          }
          else
          {
            if(CHECKPROTOCOLCOMMANDFLAG(PROTOCOLRECEIVED))
            {
              if(protocolCommandFlags == 1) //If only the "Protocol Received" flag
              { //is set, check for the Protocol command's payload.
                switch(commandCounter)
                {
                  case 5:
                    if(commandChecksum == 130) //If is "SENDX".
                    {
                      SETPROTOCOLCOMMANDFLAG(SENDXRECEIVED);
                      break;
                    }
                    if(commandChecksum == 131) //If is "SENDY".
                    {
                      SETPROTOCOLCOMMANDFLAG(SENDYRECEIVED);
                      break;
                    }
                    if(commandChecksum == 132) //If is "SENDZ".
                    {
                      SETPROTOCOLCOMMANDFLAG(SENDZRECEIVED);
                      break;
                    }
                    if(commandChecksum == 108) //If is "SENDB".
                    {
                      SETPROTOCOLCOMMANDFLAG(SENDBRECEIVED);
                      break;
                    }
                    protocolCommandFlags = 0; //When no command has been found.
                    break;
                  case 8:
                    if(commandChecksum == 79) //If is "SENDRFID".
                    {
                      SETPROTOCOLCOMMANDFLAG(SENDRFIDRECEIVED);
                      break;
                    }
                    if(commandChecksum == 108) //If is "SENDTYPE".
                    {
                      SETPROTOCOLCOMMANDFLAG(SENDTYPERECEIVED);
                      break;
                    }
                    protocolCommandFlags = 0; //When no command has been found.
                    break;
                  case 9:
                    if(commandChecksum == 175) //If is "SENDAUXIN".
                    {
                      SETPROTOCOLCOMMANDFLAG(SENDAUXINRECEIVED);
                      break;
                    }
                    protocolCommandFlags = 0; //When no command has been found.
                    break;
                  default: //When no command has been found.
                    protocolCommandFlags = 0;
                    break;
                }
              }
              else //When one of the other flags is set.
              {
                switch(protocolCommandFlags) //Check which flag has been set (defined
                { //at the top of the source). Protocol is set by this point, so
                  //the possible values are 2^n + 1, where n is between 1 and 7 inclusive.
                  case 3:   //SENDXRECEIVED
                    if(commandChecksum == 157) //If is "ON".
                    {
                      SETPROTOCOL(SENDXDATA);
                      if(adCounter == 5) //If the A/D Converter was not being used.
                      {
                        adCounter = 0; //Set the counter to the X channel.
                        ADCON0_bits.ADON = 1; //Turn it back on.
                      }
                    }
                    if(commandChecksum == 219) //If is "OFF".
                    {
                      CLEARPROTOCOL(SENDXDATA);
                      checkADConverter(); //Turn off the A/D Converter if it isn't being used.
                    }
                    break;
                  case 5:   //SENDYRECEIVED
                    if(commandChecksum == 157) //If is "ON".
                    {
                      SETPROTOCOL(SENDYDATA);
                      if(adCounter == 5) //If the A/D Converter was not being used.
                      {
                        adCounter = 1; //Set the counter to the Y channel.
                        ADCON0_bits.ADON = 1; //Turn it back on.
                      }
                    }
                    if(commandChecksum == 219) //If is "OFF".
                    {
                      CLEARPROTOCOL(SENDYDATA);
                      checkADConverter(); //Turn off the A/D Converter if it isn't being used.
                    }
                    break;
                  case 9:   //SENDZRECEIVED
                    if(commandChecksum == 157) //If is "ON".
                    {
                      SETPROTOCOL(SENDZDATA);
                      if(adCounter == 5) //If the A/D Converter was not being used.
                      {
                        adCounter = 2; //Set the counter to the Z channel.
                        ADCON0_bits.ADON = 1; //Turn it back on.
                      }
                    }
                    if(commandChecksum == 219) //If is "OFF".
                    {
                      CLEARPROTOCOL(SENDZDATA);
                      checkADConverter(); //Turn off the A/D Converter if it isn't being used.
                    }
                    break;
                  case 17:  //SENDBRECEIVED
                    if(commandChecksum == 157) //If is "ON".
                      SETPROTOCOL(SENDBUTTONSDATA);
                    if(commandChecksum == 219) //If is "OFF".
                      CLEARPROTOCOL(SENDBUTTONSDATA);
                    break;
                  case 33:  //SENDRFIDRECEIVED
                    if(commandChecksum == 157) //If is "ON".
                      enableRFID();
                    if(commandChecksum == 219) //If is "OFF".
                      disableRFID();
                    break;
                  case 65:  //SENDTYPERECEIVED
                    if(commandChecksum == 157) //If is "ON".
                      SETPROTOCOL(SENDTYPEDATA);
                    if(commandChecksum == 219) //If is "OFF".
                      CLEARPROTOCOL(SENDTYPEDATA);
                    break;
                  case 129: //SENDAUXINRECEIVED
                    if(commandChecksum == 157) //If is "ON".
                      enableAux();
                    if(commandChecksum == 219) //If is "OFF".
                      disableAux();
                    break;
                }
                protocolCommandFlags = 0; //Whether or not a valid command is received,
                //input processing will now start fresh.
              }
            }
            if(CHECKSOUNDCOMMANDFLAG(SOUNDRECEIVED))
            {
              if(soundCommandFlags == 1) //If only the "Sound Received" flag is
              //set, check for the Sound command's payload.
              {
                switch(commandCounter)
                {
                  case 4: //Could be CONN or RFID.
                    if(commandChecksum == 46) //If is "CONN".
                    {
                      SETSOUNDCOMMANDFLAG(CONNRECEIVED);
                      break;
                    }
                    if(commandChecksum == 37) //If is "RFID".
                    {
                      SETSOUNDCOMMANDFLAG(RFIDRECEIVED);
                      break;
                    }
                    soundCommandFlags = 0; //When no command has been received.
                    break;
                  case 6: //Could be BUTTON.
                    if(commandChecksum == 220) //If is "BUTTON".
                    {
                      SETSOUNDCOMMANDFLAG(BUTTONRECEIVED);
                      break;
                    }
                    soundCommandFlags = 0; //When no command has been received.
                    break;
                  case 7: //Could be DISCONN.
                    if(commandChecksum == 14) //If is "DISCONN".
                    {
                      SETSOUNDCOMMANDFLAG(DISCONNRECEIVED);
                      break;
                    }
                    soundCommandFlags = 0; //When no command has been received.
                    break;
                  case 9: //Could be ATTENTION.
                    if(commandChecksum == 182) //If is "ATTENTION".
                    {
                      SETSOUNDCOMMANDFLAG(ATTENTIONRECEIVED);
                      break;
                    }
                    soundCommandFlags = 0; //When no command has been received.
                    break;
                  default: //When no command has been received.
                    soundCommandFlags = 0;
                    break;
                }
              }
              else //When one of the other flags is set.
              {
                if(!CHECKSOUNDCOMMANDFLAG(FREQRECEIVED)) //Receiving frequency.
                { //Converting ASCII characters to numbers.
                  if(commandCounter <= 3) //Not expecting values above 255.
                  {
                    temp = 0;
                    for(i=0; i < commandCounter; i++)
                    {
                      temp = temp * 10; //Shift up one decimal place.
                      if((47 < commandBuffer[i])      //Only numbers
                          && (commandBuffer[i] < 58)) //are accepted.
                      {
                        temp += (commandBuffer[i] & 0xF); //The first nibble of an ASCII
                                                       //number corresponds to binary.
                        SETSOUNDCOMMANDFLAG(FREQRECEIVED);
                      }
                      else //When it is not a number, drop the whole thing.
                      {
                        soundCommandFlags = 0;
                        i = commandCounter;
                      }
                    }
                  }
                  if(CHECKSOUNDCOMMANDFLAG(FREQRECEIVED)) //This routine won't execute
                  {                                       //if the data was invalid.
                    switch(soundCommandFlags) //Check which flag has been set (defined
                    { //at the top of the source). SOUNDRECEIVED and FREQRECEIVED are
                      //set by this point, so the possible values are 2^n + 65, where
                      //n is between 1 and 5 inclusive.
                      case 67:  //CONNRECEIVED
                        connFreqValue = temp;
                        break;
                      case 69:  //DISCONNRECEIVED
                        disconnFreqValue = temp;
                        break;
                      case 73:  //RFIDRECEIVED
                        rfidFreqValue = temp;
                        break;
                      case 81: //BUTTONRECEIVED
                        buttonFreqValue = temp;
                        break;
                      case 97: //ATTENTIONRECEIVED
                        attentionFreqValue = temp;
                        break;
                    }
                  }
                }
                else //When FREQRECEIVED is set, receive the duration data.
                {
                  if(commandCounter <= 3) //Not expecting values above 255.
                  {
                    temp = 0;
                    for(i=0; i < commandCounter; i++)
                    {
                      temp = temp * 10; //Shift up one decimal place.
                      if((47 < commandBuffer[i])      //Only numbers
                          && (commandBuffer[i] < 58)) //are accepted.
                      {
                        temp += (commandBuffer[i] & 0xF); //The first nibble of an ASCII
                                                        //number corresponds to binary.
                        SETSOUNDCOMMANDFLAG(DURATIONRECEIVED);
                      }
                      else //When it is not a number, drop the whole thing.
                      {
                        soundCommandFlags = 0;
                        i = commandCounter;
                      }
                    }
                  }
                  if(CHECKSOUNDCOMMANDFLAG(DURATIONRECEIVED)) //This routine won't
                  {                              //execute if the data was invalid.
                    switch(soundCommandFlags) //Check which flag has been set (defined
                    { //at the top of the source). SOUNDRECEIVED, FREQRECEIVED, and
                      //DURATIONRECEIVED are set by this point, so the possible values
                      //are 2^n + 193, where n is between 1 and 5 inclusive.
                      case 195:  //CONNRECEIVED
                        connDurationValue = temp;
                        break;
                      case 197:  //DISCONNRECEIVED
                        disconnDurationValue = temp;
                        break;
                      case 201:  //RFIDRECEIVED
                        rfidDurationValue = temp;
                        break;
                      case 209: //BUTTONRECEIVED
                        buttonDurationValue = temp;
                        break;
                      case 225: //ATTENTIONRECEIVED
                        attentionDurationValue = temp;
                        break;
                    }
                  }
                  soundCommandFlags = 0; //Clear the flags when the input has
                                         //been received.
                }
              }
            }
          }
        }
        commandChecksum = 0;
        commandCounter = 0;
      }
      else //If it's not whitespace, it gets stored and commandCounter is incremented.
      {
        if(soundCommandFlags > 1) //If a number is being received, the bytes are stored
        {  //into a buffer array. Otherwise, the bytes are computed into a checksum.
          commandBuffer[commandCounter] = RCREG;
        }
        else
        {
          commandChecksum += RCREG; //The most pitiful checksum ever, but it provides
        } //a unique value for each command (note that the number of characters in each
          //command is also checked.
        commandCounter++;
      }
      if(commandCounter > 14) //When the buffer reaches 14 characters with no whitespace,
      {                       //it resets.
        commandChecksum = 0;
        commandCounter = 0;
      }
    }
  }
}

