#define MAJORVERSION  2
#define MINORVERSION  9


//Prevents bits from being addressed independently of their register bytes.
#define NO_BIT_DEFINES
#include <pic14regs.h>

/** General Purpose Defines */
#define FOSC 14745600
#define SPBRGVAL(b) ((FOSC)/64/(b)-1)
#define TMR0RATE   (255+2-(FOSC/4/2/9600))
//After setting TMR0, Timer 0 is not incremented for 2 clock cycles. Clock rate
//is 14.7456 MHz. PIC16F88 executes 1 instruction cycle every 4 clock cycles,
//prescaler is at 1:2, and desired timer overflow rate is 9600 Hz.

#define SETBIT(p,b)	(p) |= 1<<(b)
#define CLEARBIT(p,b)	(p) &= ~(1<<(b))
#define TOGGLEBIT(p,b)	(p) ^= 1<<(b)
#define CHECKBIT(p,b)	((p & (1<<(b))) > 0 ? 1:0)
/** End General Purpose Defines */

/** Program Flags: Defined fully in final.c */
extern volatile unsigned char programFlags;
#define TRANSMITSTATE      0
#define MSBX               1
#define MSBY               2
#define MSBZ               3
#define NEWRFIDDATA        4
#define ADRUNNING          5

#define SETPROGRAMFLAG(f)       programFlags |= 1<<(f)
#define CLEARPROGRAMFLAG(f)     programFlags &= ~(1<<(f))
#define TOGGLEPROGRAMFLAG(f)    programFlags ^= 1<<(f)
#define CHECKPROGRAMFLAG(f)     ((programFlags & (1<<(f))) > 0 ? 1:0)
/** End Program Flags definitions. */

/** Protocol Toggles (Transmissions enabled when bit set):
 * Bit 0: Send type byte.
 * Bit 1: Send X data.
 * Bit 2: Send Y data.
 * Bit 3: Send Z data.
 * Bit 4: Send button data.
 * Bit 5: Send RFID data.
 * Bit 6: Send Auxillary A/D data.
 */
//These bits keep track of which transmissions are enabled or disabled with the
//PROTOCOL command. Default value enables the sending of the type byte, X and Y
//values, button data, and RFID data.
extern volatile unsigned char protocolToggles;
#define SENDTYPEDATA      0
#define SENDXDATA         1
#define SENDYDATA         2
#define SENDZDATA         3
#define SENDBUTTONSDATA   4
#define SENDRFIDDATA      5
#define SENDAUXDATA       6

#define SETPROTOCOL(p)          protocolToggles |= 1<<(p)
#define CLEARPROTOCOL(p)        protocolToggles &= ~(1<<(p))
#define TOGGLEPROTOCOL(p)       protocolToggles ^= 1<<(p)
#define CHECKPROTOCOL(p)        ((protocolToggles & (1<<(p))) > 0 ? 1:0)
/** End Protocol Toggles definitions. */

volatile unsigned char oneWireWorkingByte = 0;
volatile unsigned char oneWireReadingByte = 0;

/** One Wire States
 * These flags are mutually exclusive. Default is LISTEN.
 */
#define LISTEN          0
#define READBYTE        1
#define WRITEBYTE       2
#define EXITONEWIREMODE 3
volatile unsigned char oneWireStates = 1;
#define SETONEWIRESTATE(s)    oneWireStates = 1<<(s)
#define CHECKONEWIRESTATE(s)  ((oneWireStates & (1<<(s)))?1:0)

/** End One Wire States */

/**
 *Oscillator speed is 14.7456 MHz, and the clock cycle is a quarter of that.
 *A timer operates with a prescaler of 4, putting it at 921,600 Hz. The timer
 *increments every 1.085 microseconds. This provides a timer that goes slightly
 *over 1 millisecond, so the minimum time increments can be used.
 */
#define WAIT(time)   {PR2 = (time); \
                     TMR2 = 0; \
                     PIR1_bits.TMR2IF = 0; \
                     while(!PIR1_bits.TMR2IF) {} }


/** Function Prototypes */
void initializeTimer0();
void initializeBluetooth();
void reinitializePromiSD();
unsigned char waitForResponse();
unsigned char waitForOk();
void initializeTRISbits();
void initializeADConverter();
void startAD();
void checkADConverter();
void startSound(unsigned char, unsigned char);
void startRFID();
void enableRFID();
void disableRFID();
void enableAux();
void disableAux();
void disableAll();
void oneWire();
void versionMajor();
void versionMinor();
void programMode();

extern volatile unsigned char soundFreqValue;
extern volatile unsigned char soundDurationCounter;
extern volatile unsigned char temp;
extern volatile unsigned char i;
extern volatile unsigned char rfidCounter;
extern volatile unsigned char rfidBuffer;
extern volatile unsigned char dataRFID;
extern volatile unsigned char adCounter;
extern volatile unsigned char disconnFreqValue;
extern volatile unsigned char disconnDurationValue;
extern volatile unsigned char connFreqValue;
extern volatile unsigned char connDurationValue;
volatile unsigned char j = 0;
volatile unsigned char k = 0;
volatile unsigned char inputReady = 0;

/** Reinitialize Promi SD Commands */
code unsigned char atCommand1[20] = {'A','T','+','B','T','N','A','M','E','=','"','S',
                                     'c','i','p','i','o','"','\r'};
code unsigned char atCommand2[13] = {'A','T','+','B','T','M','O','D','E',',','3','\r'};
code unsigned char atCommand3[15] = {'A','T','+','B','T','S','C','A','N',',','3',',',
                                     '0','\r'};
code unsigned char atCommand4[27] = {'A','T','+','U','A','R','T','C','O','N','F','I',
                                     'G',',','3','8','4','0','0',',','n',',','1',',',
                                     '0','\r'};
code unsigned char atCommand5[5] = {'A','T','Z','\r'};
/** End Reinitialize Promi SD Commands */

/** Program Mode Variables */
volatile unsigned char portBReadBuffer = 0;
volatile unsigned char receivedByte = 0;
volatile unsigned char sendByte = 0;
volatile unsigned char numberOfBits = 0;

/** Program Mode States: These bits determine which state program mode is in at
 *    any given time, therefore one of them must always be set. THESE COMMANDS
 *    ARE DIFFERENT: WHEN ONE IS SET, THE OTHERS ARE AUTOMATICALLY CLEARED.
 *    For that reason there are no "Clear" or "Toggle" commands for this set.
 * Bit 0 (COMMANDMODE): This flag starts out set. This state reads a command
 *   byte from the bluetooth USART connection and follows its instructions.
 * Bit 1 (READMODE): This state sends a byte back through the bluetooth which
 *   contains 1-8 bits of information from the data line.
 * Bit 2 (WRITEMODE): This state receives a byte from the bluetooth, and programs
 *   up to 8 bits of data to a second Scipio.
 */
volatile unsigned char progmodeStates = 1;
#define COMMANDMODE      0
#define READMODE         1
#define WRITEMODE        2

#define SETPROGMODESTATE(p)    progmodeStates = (1<<(p))
#define CHECKPROGMODESTATE(p)  ((progmodeStates & (1<<(p))) > 0 ? 1:0)
/** End Program Mode States */
/** End Program Mode Variables */


void initializeTimer0()
{
  OPTION_REG_bits.PSA = 0; //Assign prescaler to Timer 0 (not Watchdog Timer).
  OPTION_REG_bits.T0CS = 0; //Enable Timer 0 in Timer mode.

  OPTION_REG_bits.PS2 = 0; //////////////////////
  OPTION_REG_bits.PS1 = 0; //Set prescaler to 1:2.
  OPTION_REG_bits.PS0 = 0; //////////////////////
}

void initializeBluetooth()
{
  /** The following line call the function to re-initialize the bluetooth module
   *  if it has been reset to its default settings (9600 baud).
   */
  reinitializePromiSD();

  TXSTA_bits.SYNC = 0; //Asynchronous mode.
  RCSTA_bits.SPEN = 1; //Enable serial interface.
  TXSTA_bits.TXEN = 1; //Enable serial transmit.
  RCSTA_bits.CREN = 1; //Enable continuous receive.
  SPBRG = SPBRGVAL(38400); //Enables baud rate of 38400 in low speed mode (BRGH = 0)
                           //with a internal clock of 14.7456 MHz.
}

void reinitializePromiSD()
{
  TXSTA_bits.SYNC = 0; //Asynchronous mode.
  RCSTA_bits.SPEN = 1; //Enable serial interface.
  TXSTA_bits.TXEN = 1; //Enable serial transmit.
  RCSTA_bits.CREN = 1; //Enable continuous receive.
  SPBRG = SPBRGVAL(9600); //Enables baud rate of 38400 in low speed mode (BRGH = 0)
                           //with a internal clock of 14.7456 MHz.
  

  waitForOk();
  while(!PIR1_bits.TXIF) {}
  TXREG = '\r';
  while(!PIR1_bits.TXIF) {}
  TXREG = 'A';
  while(!PIR1_bits.TXIF) {}
  TXREG = 'T';
  while(!PIR1_bits.TXIF) {}
  TXREG = '\r';

  if (!waitForOk()) return;

  for(i=0; i<=18; i++)
  {
    while(!PIR1_bits.TXIF) {}
    TXREG = atCommand1[i];
  }
  waitForOk();
  for(i=0; i<=11; i++)
  {
    while(!PIR1_bits.TXIF) {}
    TXREG = atCommand2[i];
  }
  waitForOk();
  for(i=0; i<=13; i++)
  {
    while(!PIR1_bits.TXIF) {}
    TXREG = atCommand3[i];
  }
  waitForOk();
  for(i=0; i<=25; i++)
  {
    while(!PIR1_bits.TXIF) {}
    TXREG = atCommand4[i];
  }
  waitForOk();
  for(i=0; i<=3; i++)
  {
    while(!PIR1_bits.TXIF) {}
    TXREG = atCommand5[i];
  }
  waitForOk();
}

unsigned char waitForResponse()
{
  for(i=0;i<254;i++)
  {
    for(j=0;j<254;j++)
    {
      for(k=0;k<10;k++)
      {
        if(PIR1_bits.RCIF)
        {
          return RCREG;
        }
      }
    }
  }
  return 0;
}

unsigned char waitForOk()
{
  unsigned char responseCharacter;
  while(1)
  {
    responseCharacter = waitForResponse();
    if (!responseCharacter)
      return 0;
    if (responseCharacter == 'O')
    {
      startSound(disconnFreqValue,disconnDurationValue);
      responseCharacter = waitForResponse();
      if (responseCharacter != 'K')
        return 0;
      return 1;
    }
  } 
}

void initializeTRISbits()
{
  SETBIT(TRISA,0);   //Set Port A, pin 0 (Accelerometer X) to input.
  SETBIT(TRISA,1);   //Set Port A, pin 1 (Accelerometer Y) to input.
  SETBIT(TRISA,2);   //Set Port A, pin 2 (Accelerometer Z) to input.
  CLEARBIT(TRISA,4); //Set Port A, pin 4 (Aux A/D / RFID on) to output. (Note: This pin is
                     //set to turn on the RFID reader, enabled by default. When Aux A/D is
                     //enabled, the RFID reader is turned off and this pin is set to analog
                     //input.
  SETBIT(TRISA,5); //Set Port A, pin 5 (MCLR) to input.
  SETBIT(TRISB,0); //Set Port B, pin 0 (Button 2) to input.
  SETBIT(TRISB,1); //Set Port B, pin 1 (Button 3) to input.
  SETBIT(TRISB,2);   //Set Port B, pin 2 (Bluetooth Receive) to input.
  CLEARBIT(TRISB,3); //Set Port B, pin 3 (status LED and buzzer) to output.
  SETBIT(TRISB,4); //Set Port B, pin 4 (Button 1) to input.
  CLEARBIT(TRISB,5); //Set Port B, pin 5 (Bluetooth Transmit) to output.
  CLEARBIT(TRISB,6); //Set Port B, pin 6 (Accelerometer board LED) to output.
  SETBIT(TRISB,7); //Set Port B, pin 7 (RFID scanner input) to input.
}

void initializeADConverter()
{
  CLEARBIT(ANSEL,6); //Set Port B, pin 7 (RFID scanner input) to digital I/O.
  CLEARBIT(ANSEL,5); //Set Port B, pin 6 () to digital I/O.
  CLEARBIT(ANSEL,3); //Set Port A, pin 3 () to digital I/O.
  SETBIT(ANSEL,2); //Set Port A, pin 2 (Accelerometer Z) to analog input.
  SETBIT(ANSEL,1); //Set Port A, pin 1 (Accelerometer Y) to analog input.
  SETBIT(ANSEL,0); //Set Port A, pin 0 (Accelerometer X) to analog input.
  ADCON1_bits.ADFM = 0;//Left-justify the data (the two LSBs are in ADRESL).
  ADCON1_bits.ADCS2 = 0; ///////////////////////////
  ADCON0_bits.ADCS1 = 1; //Set A/D Clock to FOSC/32.
  ADCON0_bits.ADCS0 = 0; ///////////////////////////
  
  ADCON0_bits.ADON = 1; //Turn on Analog/Digital Converter.
}

void startAD() //This routine starts Timer 2 to overflow after 80 cycles. This is to allow
{     //the A/D Converter sufficient time to sample the channel, and charge the capacitor.
  //I'm not *entirely* sure about this value, but it can be easily changed if necessary.
  //The formula for calculating the acquisition time (in milliseconds, not in cycles) is
  //in figure 12.1 of the PIC16F87/88 Data Sheet.
  T2CON_bits.TMR2ON = 1; //Turn on Timer 2.
  TMR2 = 0; //Reset the counter.
  PR2 = 80; //Timer 2 will overflow after 80 cycles.
  PIR1_bits.TMR2IF = 0; //Clear the flag.
  PIE1_bits.TMR2IE = 1; //Enable Timer 2 interrupt.
}

void checkADConverter()
{
  if(!(CHECKPROTOCOL(SENDXDATA) || CHECKPROTOCOL(SENDYDATA) || CHECKPROTOCOL(SENDZDATA) ||
      CHECKPROTOCOL(SENDAUXDATA)))
  {
    adCounter = 5;
    ADCON0_bits.ADON = 0; //Turn off the A/D Converter when it's not being used, to
                          //conserve battery life.
  }
}

void startSound(unsigned char frequencyValue, unsigned char durationValue)
{
  if(!CHECKBIT(durationValue,7)) //If the duration is less than 128.
  {
      soundFreqValue = frequencyValue;
      soundDurationCounter = durationValue;
      if(T1CON_bits.TMR1ON) //If a sound is already playing, don't reset the timer.
        return;
      T1CON_bits.TMR1ON = 1; //Turn on Timer 1.
      TMR1H = 0xFD;  //This 16-bit value will cause Timer 1 to overflow at 6400 Hz.
      TMR1L = 0xBF;
      PIR1_bits.TMR1IF = 0; //Clear flag first.
      PIE1_bits.TMR1IE = 1; //Enable Timer 1 interrupts.
  }
}

void startRFID()
{
  temp = PORTB;
  INTCON_bits.RBIF = 0;
  INTCON_bits.RBIE = 1;
}

void enableRFID()
{
  CLEARBIT(ANSEL,4); //Set port A, pin 4 to digital I/O.
  CLEARBIT(TRISA,4); //Set port A, pin 4 to output.
  CLEARBIT(PORTA,4); //Turn on the RFID reader.
  disableAux();
  startRFID();
  SETPROTOCOL(SENDRFIDDATA);
}

void disableRFID()
{
  INTCON_bits.RBIE = 0; //Turn off Port B change interrupt.
  INTCON_bits.TMR0IE = 0; //Turn off Timer 0 interrupt.
  rfidCounter = 0;
  rfidBuffer = 0;
  dataRFID = 0;
  SETBIT(PORTA,4); //Turn off the RFID reader.
  CLEARPROTOCOL(SENDRFIDDATA);
}

void enableAux()
{
  disableRFID();
  SETBIT(ANSEL,4); //Set port A, pin 4 (Aux A/D) to analog input.
  SETPROTOCOL(SENDAUXDATA);
  if(adCounter == 5) //If the A/D Converter was not being used.
  {
    adCounter = 4; //Set the counter to the Aux channel.
    ADCON0_bits.ADON = 1; //Turn it back on.
  }
}

void disableAux()
{
  CLEARPROTOCOL(SENDAUXDATA);
  checkADConverter(); //Turn off the A/D Converter if it isn't being used.
}

void disableAll()
{
  disableRFID();
  protocolToggles = 0;
  adCounter = 5;
  ADCON0_bits.ADON = 0;
  T2CON_bits.TMR2ON = 0;
}

void oneWire()
{
  INTCON_bits.GIE = 0; //Disable interrupts.
  T2CON_bits.T2CKPS0 = 1; //Set timer prescaler to 4.
  T2CON_bits.TMR2ON = 1; //Turn on Timer 2.
  
  SETONEWIRESTATE(LISTEN);
  
  while(!CHECKONEWIRESTATE(EXITONEWIREMODE))
  {
    if(CHECKONEWIRESTATE(LISTEN))
    {
      if(PIR1_bits.RCIF)
      {
        temp = RCREG;
        switch(temp)
        {
          case 0xa0: //Send the reset signal.
            SETONEWIRESTATE(READBYTE);
            oneWireReadingByte=1;
            SETBIT(TRISA,3);
            CLEARBIT(PORTA,3);
            CLEARBIT(TRISA,3);
            WAIT(250) //WAIT() can only take values up to 254.
            WAIT(230)
            SETBIT(TRISA,3);
            WAIT(60)
            if(CHECKBIT(PORTA,3)) //If the return pulse is not detected, exit.
            {
            oneWireReadingByte=0;
              break;
            }
            WAIT(220)
            WAIT(200)
            break;
          case 0x80: //Write the following byte on the 1-Wire bus.
            SETONEWIRESTATE(WRITEBYTE);
            break;
          case 0xc0: //Read one byte from the 1-Wire bus.
            for(i=8; i > 0; i--)
            {
              CLEARBIT(PORTA,3);
              CLEARBIT(TRISA,3);
              WAIT(5)
              SETBIT(TRISA,3);
              WAIT(3)
              oneWireReadingByte = oneWireReadingByte >> 1;
              if(CHECKBIT(PORTA,3))
                SETBIT(oneWireReadingByte,7);
              WAIT(60)
            }
            SETONEWIRESTATE(READBYTE);
            break;
          case 0x43:
          case 0x90: //Exit 1-Wire mode.
            SETONEWIRESTATE(EXITONEWIREMODE);
            break;
        }
      }
    }
    if(CHECKONEWIRESTATE(WRITEBYTE))
    {
      if(PIR1_bits.RCIF)
      {
        temp = RCREG;
        for(i = 8; i > 0; i--)
        {
          CLEARBIT(PORTA,3);
          CLEARBIT(TRISA,3);
          WAIT(3)
          if(CHECKBIT(temp,0))
            SETBIT(TRISA,3);
          temp = temp >> 1;
          WAIT(60)
          SETBIT(TRISA,3);
          WAIT(2)
        }
        SETONEWIRESTATE(LISTEN);
      }
    }

    if(CHECKONEWIRESTATE(READBYTE))
    {
      if(PIR1_bits.TXIF)
      {
        TXREG = oneWireReadingByte;
        SETONEWIRESTATE(LISTEN);
      }
    }
  }
  T2CON_bits.TMR2ON = 0; //Turn off Timer 2.
  PIR1_bits.TMR2IF = 0; //Clear Timer 2 flag.
  T2CON_bits.T2CKPS0 = 0; //Clear Timer 2 prescaler.
  INTCON_bits.GIE = 1; //Re-enable interrupts.
}

void versionMajor()
{
  while(!PIR1_bits.TXIF) {} //Wait until the transmit buffer is ready.
  TXREG = MAJORVERSION;
}

void versionMinor()
{
  while(!PIR1_bits.TXIF) {} //Wait until the transmit buffer is ready.
  TXREG = MINORVERSION;
}

void programMode()
{
  ADCON0_bits.ADON = 0; //Turn off A/D Converter.
  SETBIT(PORTA,4); //Turn off the RFID reader.
  INTCON_bits.GIE = 0; //Disable interrupts.
  CLEARBIT(OPTION_REG,7); //Enable internal pull-ups for Port B.

  CLEARBIT(TRISA,0);
  SETBIT(PORTA,1);
  CLEARBIT(TRISA,1);
  CLEARBIT(TRISB,0);
  CLEARBIT(TRISB,1);
  CLEARBIT(TRISB,6);
  ANSEL = 0;


  while(1)
  {
    if(CHECKPROGMODESTATE(COMMANDMODE))
    {
      if(PIR1_bits.RCIF)
      {
        receivedByte = RCREG;
        if(CHECKBIT(receivedByte,7)) //If bit 7 is set, read the command byte. Else, ignore
        {                            //the byte.
          if(CHECKBIT(receivedByte,6)) //Bit 6 determines the MCLR pin.
            SETBIT(PORTB,6);
          else
            CLEARBIT(PORTB,6);
          if(CHECKBIT(receivedByte,5)) //If bit 5 is set, read. If clear, write.
          {
            SETBIT(TRISB,1);
            if(CHECKBIT(receivedByte,4)) //If bit 4 is set (and 5 is set), read 0-8 following
            {                //bits (set by bits 0-3). If greater than 8, exit program mode.
              numberOfBits = (receivedByte & 0xF);
              if(numberOfBits > 8)
              {
                _asm
                goto 0
                _endasm;
              }
              else
                SETPROGMODESTATE(READMODE);
            }
            else //If bit 4 is clear (and 5 is set), apply the settings from bits 0-2, and
            {    //send a byte back with the status of the data line.
              if(CHECKBIT(receivedByte,2))   //If bit 2 is set, set the power line (this low bit
                CLEARBIT(PORTA,1); // is inverted in the cross-programmer board.)
              else                         //If bit 2 is clear, clear the power line (this
                SETBIT(PORTA,1); //high bit is inverted in the cross-programmer board.)
              if(CHECKBIT(receivedByte,1)) //If bit 1 is set, set the clock line.
                SETBIT(PORTB,0);
              else                         //If bit 1 is clear, clear the clock line.
                CLEARBIT(PORTB,0);
              if(CHECKBIT(receivedByte,0)) //If bit 0 is set, set the data line.
                SETBIT(PORTB,1);
              else                         //If bit 0 is clear, clear the data line.
                CLEARBIT(PORTB,1);

              if(CHECKBIT(PORTB,1)) //Send a byte with the status of the data line.
                sendByte = 1;
              else
                sendByte = 0;
              numberOfBits = 9;
              SETPROGMODESTATE(READMODE);
            }
          }
          else //If bit 5 is clear, write.
          {
            CLEARBIT(TRISB,1);
            if(CHECKBIT(receivedByte,4)) //If bit 4 is set (and 5 is clear), write 0-8
            { //0-8 following bits (set in bits 0-3). If greater than 8, exit program mode.
              numberOfBits = (receivedByte & 0xF);
              SETPROGMODESTATE(WRITEMODE);
            }
            else //If bit 4 is clear (and 5 is clear), apply the settings from bits 0-2.
            {
              if(CHECKBIT(receivedByte,2))   //If bit 2 is set, set the power line (this low bit
                CLEARBIT(PORTA,1); // is inverted in the cross-programmer board.)
              else                         //If bit 2 is clear, clear the power line (this
                SETBIT(PORTA,1); //high bit is inverted in the cross-programmer board.)
              if(CHECKBIT(receivedByte,1)) //If bit 1 is set, set the clock line.
                SETBIT(PORTB,0);
              else                         //If bit 1 is clear, clear the clock line.
                CLEARBIT(PORTB,0);
              if(CHECKBIT(receivedByte,0)) //If bit 0 is set, set the data line.
                SETBIT(PORTB,1);
              else                         //If bit 0 is clear, clear the data line.
                CLEARBIT(PORTB,1);
            }
          }
        }
        else //When bit 7 is clear.
        {
          receivedByte = 0;
        }
      }
      else
        TOGGLEBIT(PORTA,0);
    }
    if(CHECKPROGMODESTATE(READMODE))
    {
      if(PIR1_bits.TXIF)
      {
        if(numberOfBits == 9) //Only set when bit 5 is set and bit 4 is clear.
        {
          TXREG = sendByte;
          numberOfBits = 0;
          sendByte = 0;
        }
        if(numberOfBits != 0)
        {
          temp = 0;
          portBReadBuffer = 0;
          sendByte = 0;
          while(numberOfBits > 0) //Read the set number of bits from the data line.
          {
            SETBIT(PORTB,0); //Set the clock bit.
            _asm
            nop
            nop
            nop
            nop
            nop
            nop
            nop
            nop
            _endasm; //Waiting.
            CLEARBIT(PORTB,0); //Clear the clock bit.
            portBReadBuffer = PORTB;
            _asm
            nop
            nop
            nop
            nop
            nop
            nop
            nop
            nop
            _endasm; //Waiting.
            if(CHECKBIT(portBReadBuffer,1)) //If data line is set, set a bit in
              SETBIT(sendByte,temp);        //sendByte.
            temp++;
            TOGGLEBIT(PORTA,0);
            numberOfBits--;
          }
          TXREG = sendByte;
        }
        SETPROGMODESTATE(COMMANDMODE);
      }
    }
    
    if(CHECKPROGMODESTATE(WRITEMODE))
    {
      if(PIR1_bits.RCIF)
      {
        if(numberOfBits == 0)
          SETPROGMODESTATE(COMMANDMODE);
        else
        {
          receivedByte = RCREG;
          while(numberOfBits > 0) //Write the set number of bits into the data line.
          {
            if(CHECKBIT(receivedByte,0)) //Put the first bit into the data line.
              SETBIT(PORTB,1);
            else
              CLEARBIT(PORTB,1);
            SETBIT(PORTB,0); //Set the clock bit.
            _asm
            nop
            nop
            nop
            nop
            nop
            nop
            nop
            nop
            _endasm; //Waiting.
            CLEARBIT(PORTB,0); //Clear the clock bit.
            _asm
            nop
            nop
            nop
            nop
            nop
            nop
            nop
            nop
            _endasm; //Waiting.
            CLEARBIT(PORTB,1); //Clear the data line.
            receivedByte = receivedByte>>1; //Shift the next bit into place.
            TOGGLEBIT(PORTA,0);
            numberOfBits--;
          }
        }
      }
      else
        TOGGLEBIT(PORTA,0);
    }
  }
}
