#include <unistd.h>
#include <fcntl.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>
#include <sys/time.h>
#include <sys/select.h>
#include <signal.h>
#include <termios.h>
#include <syslog.h>
#include <sys/ioctl.h>
#include <netinet/in.h>
#include <netdb.h>
#include <arpa/inet.h>
#include <sys/socket.h>

#include <sys/types.h>
#include <sys/un.h>
#include <syslog.h>

#include <bluetooth/bluetooth.h>
#include <bluetooth/hci.h>
#include <bluetooth/hci_lib.h>
#include <bluetooth/rfcomm.h>
#include "lowlevel_input.h"
#include "xscipio.h"

#define INITZCMD "PROTOCOL SENDZ ON\n"
#define SHUNT_VALUE 0.05

static int battery_listen_sock = -1;

struct t_battery_state {
  int raac;
  int raac_p;
  int rsac;
  int rsac_p;
  int rarc;
  int rarc_p;
  int rsrc;
  int rsrc_p;
  short iavg;
  double iavg_p;
  short temp;
  double temp_p;
  int volt;
  double volt_p;
  short current;
  double current_p;
  int acr;
  double acr_p;
  int acrl;
  double acrl_p;
  int as;
  int as_p;
  int full;
  int full_p;
  int ae;
  int ae_p;
  int se;
  int se_p;
};

void init_battery_sock(int port)
{
  int x;
  struct sockaddr_in addr;
  addr.sin_family = AF_INET;
  addr.sin_port = htons(port);
  addr.sin_addr.s_addr = htonl(INADDR_LOOPBACK);
  battery_listen_sock = socket(AF_INET, SOCK_STREAM, 0);
  x=1;
  setsockopt(battery_listen_sock,SOL_SOCKET,SO_REUSEADDR,&x,sizeof (x));
  if (bind(battery_listen_sock, (struct sockaddr *) &addr, sizeof(addr))) {
    fprintf(stderr,"cannot bind battery port\n");  
    close(battery_listen_sock);
    battery_listen_sock=-1;
  } else {
    listen(battery_listen_sock,2);
  }
}

static int search_btname(char *name, bdaddr_t *result, int timeout) 
{
  char addr[80];
  inquiry_info *info=NULL;
  int i;
  int found=0;
  static int found_class=0;
  int dd=hci_open_dev(0);
  /* get a list of all available devices */
  int num_dev;
  if (use_syslog)
    syslog(LOG_INFO,"starting to search for devices with major class=0x1f and name: %s\n",name);
  num_dev=hci_inquiry(0,timeout*8/10,100,NULL,&info,found_class?0:IREQ_CACHE_FLUSH);
  if (num_dev<0){
    fprintf(stderr,"searching devices failed\n");
    sleep(1);
  }
  found_class=0;
  fprintf(stderr,"found %d devices\n",num_dev);
  for(i=0;i<num_dev;i++) {
    ba2str(&info[i].bdaddr,addr);
    fprintf(stderr,"%s\n",addr);
  }
  for(i=0;i<num_dev;i++){
    char buf[80];
    /* check major device class, the value is only valid
     * for PromiESD-Devices */
    if (info[i].dev_class[1]==0x1f) { 
      found_class=1;
      ba2str(&info[i].bdaddr,addr);
      fprintf(stderr,"getting name for %s: ",addr);
      /* get the bluetooth name */
      if (hci_read_remote_name(dd,&info[i].bdaddr,sizeof(buf),buf,100000)<0) {
        fprintf(stderr,"failed\n");
      } else {
        fprintf(stderr,"%s: ",buf)
	  /* compare the device name */;
        if (strcmp(name,buf)==0) {
          found=1;
          *result=info[i].bdaddr;
          fprintf(stderr,"found!\n");
          /* right device, done */
          break;
        } else {
          fprintf(stderr,"nothing interesting\n");
        }
      }
              
    } 
  }
  if (!found) {
    syslog(LOG_INFO,"no devices found");
    sleep(1);
  }
  if (!found_class)  {
    fprintf(stderr,"no devices with the right class found\n");
  }
  hci_close_dev(dd);
  return found;
}


/* open serial device, set the baudrate and set mode to raw mode */

static int open_ser(char *fname, int bps)
{
  struct termios mytermios;
  int b;
  int fd=open(fname,O_RDWR);
  if (fd<0) {
    fprintf(stderr,"Cannot open connection to %s\n",fname);
    return -1;
  }
  tcgetattr(fd, &mytermios);
  cfmakeraw(&mytermios);
  mytermios.c_cflag&=(~CRTSCTS);
  /* mytermios.c_cflag&=(~CLOCAL); */
  b=B115200;
  switch(bps) {
  case 9600: b=B9600; break;
  case 19200: b=B19200; break;
  case 38400: b=B38400; break;
  case 57600: b=B57600; break;
  case 115200: b=B115200; break;
  default: fprintf(stderr,"invalid rate: %d\n",bps);
  }
  cfsetospeed(&mytermios,b);
  cfsetispeed(&mytermios,B0);
  tcsetattr(fd,TCSANOW,&mytermios);
  tcflush(fd,TCIFLUSH);
  tcflush(fd,TCOFLUSH); 
  return fd;
}

/* create a tcp connection */
int open_tcp(char *host, int port)
{
  struct hostent *ph=gethostbyname(host);
  int sock;
  struct sockaddr_in addr;
  addr.sin_family=AF_INET;
  addr.sin_port=htons(port);
  if (!ph) {
    return -1;
  }
  sock=socket(AF_INET,SOCK_STREAM,0);
  memcpy((char *) &addr.sin_addr,ph->h_addr,ph->h_length); 
  if (sock<0) {
    return -1;
  } 
  if (connect(sock,(struct sockaddr *)&addr,sizeof(addr))) {
    close(sock);
    return -1;
  }
  return sock;
  
}

/* create a rfcomm connection */
int open_bluetooth(char *mac, int channel)
{
  bdaddr_t local;
  int fd;
  char buf[80];
  struct sockaddr_rc rcaddr;  
  fd=socket(PF_BLUETOOTH,SOCK_STREAM,BTPROTO_RFCOMM);
  if (fd<0) {
    perror("socket(): ");
    return -1;
  }
  hci_devba(0, &local);
  bacpy(&rcaddr.rc_bdaddr,&local);  
  rcaddr.rc_family=AF_BLUETOOTH;  
  rcaddr.rc_channel=0;
  if (bind(fd,(struct sockaddr *)&rcaddr,sizeof(rcaddr))) {
    perror("bind: ");
    close(fd);
    return -1;
  }
  rcaddr.rc_channel=channel;
  if (mac[0]==':') {
    /* try to find out the bluetooth addr */ 
    while(!search_btname(mac+1,&rcaddr.rc_bdaddr,10));
  } else {
    str2ba(mac,&rcaddr.rc_bdaddr);
  }
  if (connect(fd,(struct sockaddr *)&rcaddr,sizeof(rcaddr))) {
    perror("connect: ");
    close(fd);
    return -1;
  }
  ba2str(&rcaddr.rc_bdaddr,buf);
  if (use_syslog)
    syslog(LOG_INFO,"sucessfully connected to %s", buf);

  if(enable_z)
    {
      fd_set rfds;
      struct timeval tv;
      if (beverbose) {printf("Enabling Z Mode\n"); fflush(stdout);}
      usleep(300000);
      // make sure the command parsing state machine is
      // in a sane state
      write(fd,"\nCONNECT\n",strlen("\nCONNECT\n"));
      usleep(300000);
      // and now enable the z axis
      write(fd,INITZCMD,strlen(INITZCMD));
      usleep(300000);
      write(fd,INITZCMD,strlen(INITZCMD));
      usleep(300000);
      // flush input received before commands are processed
      while(1) {
        char buf[512];
        FD_ZERO(&rfds);
        FD_SET(fd,&rfds);
        tv.tv_sec=0;
        tv.tv_usec=0;
        if (select(fd+1,&rfds,NULL,NULL,&tv)>0) {
          printf("throw away %d bytes\n",read(fd,buf,sizeof(buf)));
        } else {
          break;
        }
      }
    }


  return fd;
}


/* handle input from the serial interface on the scipio board */
static void handle_ser_input(char ser)
{
   static char serbuf[128];
   static int readpos;
   static int pos;
   serbuf[pos]=ser;
   pos++;
   /* full line read -> a rfid tag was scanned */
   if ((ser==0xa)&&(pos>=2)&&(serbuf[pos-2]==0xd)) {
     handle_rfid_input(serbuf+readpos,pos-readpos);
     readpos=pos;
   }
   /*  buffer is full */
   if (pos>(sizeof(serbuf)-1)) {
     /* delete stuff which was already read */
     if (readpos) {
       memmove(serbuf,serbuf+readpos,pos-readpos);
       pos-=readpos;
       readpos=0;
     } else {
       /* nothing read, drop whole buffer */
       fputs("buffer overrun, dropping: \n",stderr);
       fwrite(serbuf,1,pos,stderr);
       pos=0;
     }
   }
}

/* checks for a type number */
//#define CHKI(x) (((x)&0xf8)==0x40)
#define CHKI(x) (((x)&0xf0)==0x40) /* test more bits */

static int timed_read_byte(int fd, unsigned char *out)
{
  fd_set rfds;
  struct timeval tv;
  tv.tv_sec=1;
  tv.tv_usec=0;
  FD_ZERO(&rfds);
  FD_SET(fd,&rfds);
  if (select(fd+1,&rfds,NULL,NULL,&tv)>0) {
    return read(fd,out,1);
  } else {
    return 0;
  }
}

/* output battery state */
static void output_bs(struct t_battery_state *bs, char *buf, int len)
{
  snprintf(buf,len,"RAAC (Remaining Active Absolute Capacity): %d\n"
		  "RSAC (Remaining Standby Absolute Capacity): %d\n"
		  "RARC (Remaining Active Relative Capacity): %d (%d%%)\n"
		  "RSRC (Remainint Standby Relative Capacity): %d (%d%%)\n"
		  "IAVG (Average Current): %d (%.2f mA)\n"
		  "TEMP (Temperature): %d (%.2f C)\n"
		  "VOLT (Voltage): %d (%.3f V)\n"
		  "CURRENT (Current): %d (%.2f mA)\n"
		  "ACR (Accumulated Current): %d\n"
		  "ACRL (Low Accumulated Current): %d\n"
		  "AS (Age Scalar): %d\n"
		  "FULL (Full Capacity): %d\n"
		  "AE (Active Empty): %d\n"
		  "SE (Standby Empty): %d\n",
		  bs->raac,
		  bs->rsac,
		  bs->rarc,bs->rarc_p,
		  bs->rsrc,bs->rsrc_p,
		  bs->iavg,bs->iavg_p,
		  bs->temp,bs->temp_p,
		  bs->volt,bs->volt_p,
		  bs->current,bs->current_p,
		  bs->acr,
		  bs->acrl,
		  bs->as,
		  bs->full,
		  bs->ae,
		  bs->se);
}



/* handle onewire traffic to the battery port */
static void handle_batteryport()
{
  int i;
  char outbuf[4096];
  unsigned char inbuf[32];
  int s=accept(battery_listen_sock,NULL,NULL); 
  if (s<0) {
    return;
  }
  fcntl(s,F_SETFL,fcntl(s,F_GETFL)|O_NONBLOCK); 
  write(destfd,"\n1WIRE\n1WIRE\n",13);
  usleep(100000);
  while(read(destfd,outbuf,sizeof(outbuf))>0);
  usleep(100000); 
  if (read(destfd,outbuf,sizeof(outbuf))>0) {
    char *answ="battery access not supported by firmware\n";
    write(s,answ,strlen(answ));
    close(s);
  write(destfd,"\x90",1);
  write(destfd,"\x90",1);
    return;
  }
  write(destfd,"\xa0",1);
  if (1!=timed_read_byte(destfd,inbuf)) {
    char *answ="read failed\n";
    write(s,answ,strlen(answ));
    close(s);
    write(destfd,"\x90",1);
    return;
  } else if (inbuf[0]==0) {
    char *answ="no 1wire device found\n";
    write(s,answ,strlen(answ));
    close(s);
    write(destfd,"\x90",1);
    return;
  }
  write(destfd,"\x80\xcc",2);
  write(destfd,"\x80\x69",2);
  write(destfd,"\x80\x02",2);
  for(i=2;i<0x1c;i++) {
    outbuf[(i-2)*4]=0;
    outbuf[(i-2)*4+1]=0xc0;
    outbuf[(i-2)*4+2]=0;
    outbuf[(i-2)*4+3]=0;
  }
  write(destfd,outbuf,4*(0x1c-2));
  for(i=2;i<0x1c;i++) {
    if (1!=timed_read_byte(destfd,inbuf+i)) {
      char *answ="read failed\n";
      write(s,answ,strlen(answ)); 
      break;
    }
  }
  if (i==0x1c) {
    struct t_battery_state bs;
    bs.raac=inbuf[2]*256+inbuf[3];
    bs.rsac=inbuf[4]*256+inbuf[5];
    bs.rarc=inbuf[6];
    bs.rarc_p=bs.rarc*100/255;
    bs.rsrc=inbuf[7];
    bs.rsrc_p=bs.rsrc*100/255;
    bs.iavg=inbuf[8]*256+inbuf[9];
    bs.iavg_p=bs.iavg;
    bs.iavg_p=bs.iavg_p*1.5625/1000000.0/SHUNT_VALUE*1000.0;
    bs.temp=inbuf[0xa]*256+inbuf[0xb];
    bs.temp_p=bs.temp;
    bs.temp_p=bs.temp_p/256.0;
    bs.volt=inbuf[0xc]*256+inbuf[0xd];
    bs.volt_p=bs.volt/32;
    bs.volt_p=bs.volt_p*0.00488*2;
    bs.current=inbuf[0xe]*256+inbuf[0xf];
    bs.current_p=bs.current;
    bs.current_p=bs.current_p*1.5625/1000000.0/SHUNT_VALUE*1000.0;
    bs.acr=inbuf[0x10]*256+inbuf[0x11];
    bs.acrl=inbuf[0x12]*256+inbuf[0x13];
    bs.as=inbuf[0x14];
    bs.full=inbuf[0x16]*256+inbuf[0x17];
    bs.ae=inbuf[0x18]*256+inbuf[0x19];
    bs.se=inbuf[0x1a]*256+inbuf[0x1b];
    output_bs(&bs, outbuf,sizeof(outbuf));
    write(s,outbuf,strlen(outbuf));
  }
  close(s);
  write(destfd,"\x90",1);
}


/* extract x, y and button values from the input */
void get_values(char *name, int val, int *x, int *y, int *button)
{
   fd_set rfds;
   struct timeval tv;
   static char buf[128];
   static int len;
   static int pos;
   int foundmask=0; /* bit mask for the input found */
   /* repeat  until x, y and button values are found */
   while(foundmask!=7) {
    /* try to read enough so that 64 bytes are in the buffer */
    while(len<64) {
      int l;
      /* don't give up to open a connection to the scipio */
      while(destfd<0) {
        
	if (name[0]=='/') {
          destfd=open_ser(name,val);
        } else if (strchr(name,':')) {
          destfd=open_bluetooth(name,val);
        } else {
	  destfd=open_tcp(name,val);
	}
	if (destfd>=0) {
	  fprintf(stderr,"successfully connected\n");
	  fcntl(destfd,F_SETFL,fcntl(destfd,F_GETFL)|O_NONBLOCK);
	} else {
	   /* don't try to often to connect */
	  sleep(2);
	}
      }
      FD_ZERO(&rfds);
      tv.tv_sec=5;
      tv.tv_usec=0;
      FD_SET(destfd,&rfds);
      l=-1;
      if (select(destfd+1,&rfds,NULL,NULL,&tv)) {
        l=read(destfd,buf+len,128-len);
      } else if (use_syslog) {
        syslog(LOG_ERR,"no data received for 5 seconds, disconnecting");
      }
      if (l<=0) {
	char *btrestartcmd;
	close(destfd);
	destfd=-1;
	if (use_syslog) {
	  syslog(LOG_ERR,"connection terminated\n");
	}
	btrestartcmd=getenv("BTRESTART");
	if (btrestartcmd)
          system(btrestartcmd);
        continue;
      }
      if (battery_listen_sock>0) {
        FD_ZERO(&rfds);
	FD_SET(battery_listen_sock,&rfds);
	tv.tv_sec=0;
	tv.tv_usec=0;
	if (select(battery_listen_sock+1,&rfds,NULL,NULL,&tv)>0) {
	  handle_batteryport();
	}
      }
      len+=l;
    }
    if (pos>32) {
      memmove(buf,buf+pos,len-pos);
      len-=pos;
      pos=0;
      continue;
    }
    /* synchronize with the input data */
    if (CHKI(buf[pos])&&CHKI(buf[pos+2])&&CHKI(buf[pos+4])&&CHKI(buf[pos+6])) {
      /* successfully found one type/value pair */
      switch(buf[pos]) {
      case 0x41: *x=(buf[pos+1]&0xff)-0x100; foundmask|=1; 
	if( beverbose ) printf( "X: %d\n", *x );
	break;
      case 0x42: *x=(buf[pos+1]&0xff);       foundmask|=1; 
	if( beverbose ) printf( "X: %d\n", *x );
	break;
      case 0x43: *y=(buf[pos+1]&0xff)-0x100; foundmask|=2; break;
      case 0x44: *y=(buf[pos+1]&0xff);       foundmask|=2; break;
      case 0x45: *button=buf[pos+1]&0xff;    foundmask|=4; break; 
      case 0x46: handle_ser_input(buf[pos+1]);             break;
      case 0x47:	break;
      }
      pos+=2;
    } else {
      /* nothing valid found, try to resynchronize */
      pos+=1;
    }
  }
   
}

void get_values_z(char *name, int val, int *x, int *y, int *z, int *button)
{
   fd_set rfds;
   struct timeval tv;
   static char buf[128];
   static int len;
   static int pos;
   int pcount=0;
   int foundmask=0; /* bit mask for the input found */
   /* repeat  until x, y, z and button values are found */
   while(foundmask!= 15) {
    /* try to read enough so that 64 bytes are in the buffer */
    while(len<64) {
      int l;
      /* don't give up to open a connection to the scipio */
      while(destfd<0) {
        
	if (name[0]=='/') {
          destfd=open_ser(name,val);
        } else if (strchr(name,':')) {
          destfd=open_bluetooth(name,val);
        } else {
	  destfd=open_tcp(name,val);
	}
	if (destfd>=0) {
	  fprintf(stderr,"successfully connected\n");
	  request_process_init(); // Request new initialisation of data processing functions
	  fcntl(destfd,F_SETFL,fcntl(destfd,F_GETFL)|O_NONBLOCK);
	} else {
	   /* don't try to often to connect */
	  sleep(2);
	}
      }
      FD_ZERO(&rfds);
      tv.tv_sec=5;
      tv.tv_usec=0;
      FD_SET(destfd,&rfds);
      l=-1;
      if (select(destfd+1,&rfds,NULL,NULL,&tv)) {
        l=read(destfd,buf+len,128-len);
      } else if (use_syslog) {
        syslog(LOG_ERR,"no data received for 5 seconds, disconnecting");
      }
      if (l<=0) {
	char *btrestartcmd;
	close(destfd);
	destfd=-1;
	if (use_syslog) {
	  syslog(LOG_ERR,"connection terminated\n");
	}
	btrestartcmd=getenv("BTRESTART");
	if (btrestartcmd)
          system(btrestartcmd);
        continue;
      }
      if (battery_listen_sock>0) {
        FD_ZERO(&rfds);
	FD_SET(battery_listen_sock,&rfds);
	tv.tv_sec=0;
	tv.tv_usec=0;
	if (select(battery_listen_sock+1,&rfds,NULL,NULL,&tv)>0) {
	  handle_batteryport();
	}
      }
      len+=l;
    }
    if (pos>32) {
      memmove(buf,buf+pos,len-pos);
      len-=pos;
      pos=0;
      continue;
    }
    /* synchronize with the input data */
    if (CHKI(buf[pos])&&CHKI(buf[pos+2])&&CHKI(buf[pos+4])&&CHKI(buf[pos+6])) {
      /* successfully found one type/value pair */
      switch(buf[pos]) {
        case 0x41: *x=(buf[pos+1]&0xff)-0x100; foundmask|=1;pcount++;if (beverbose) {printf("x"); fflush(stdout);} break;
	case 0x42: *x=(buf[pos+1]&0xff);       foundmask|=1;pcount++;if (beverbose) {printf("X"); fflush(stdout);} break;
	case 0x43: *y=(buf[pos+1]&0xff)-0x100; foundmask|=2;pcount++;if (beverbose) {printf("y"); fflush(stdout);}break;
	case 0x44: *y=(buf[pos+1]&0xff);       foundmask|=2;pcount++;if (beverbose) {printf("Y"); fflush(stdout);} break;
	case 0x45: *button=buf[pos+1]&0xff;    foundmask|=4;pcount++;if (beverbose) {printf("b"); fflush(stdout);} break; 
	case 0x46: handle_ser_input(buf[pos+1]);                   if (beverbose) {printf("r"); fflush(stdout);} break;
        case 0x47: /* This is the aux input. Not handled */        
	  break;
	case 0x48: *z=(buf[pos+1]&0xff)-0x100; foundmask|=8;pcount++;if (beverbose) {printf("z"); fflush(stdout);} break;
	case 0x49: *z=(buf[pos+1]&0xff);       foundmask|=8;pcount++;if (beverbose) {printf("Z"); fflush(stdout);} break;
      }
      pos+=2;
    } else {
      if (beverbose) {printf("."); fflush(stdout);}
      /* nothing valid found, try to resynchronize */
      pos+=1;
    }
    if (pcount>32) {
      fprintf(stderr,"Z-Axis enable failed: Still missing data after more than 32 packets\n");
	if (use_syslog) {
	  syslog(LOG_ERR,"Z-Axis enable failed: Still missing data after more than 32 packets\n");
	}
      close(destfd);
      destfd=-1;
      continue;
    }
  }

   if (beverbose) {printf("\n"); fflush(stdout);}

   
}
