/********************************************************************** 
 *  picprog - Copyright (C) 2003-2006 - Andreas Kemnade
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *                  
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied
 *  warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 ************************************************************************/
#include <sys/io.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>
#include <sys/time.h>
#include <unistd.h>
#include <fcntl.h>
#include <sys/socket.h>
#include <bluetooth/bluetooth.h>
#include <bluetooth/hci.h>
#include <bluetooth/hci_lib.h>
#include <bluetooth/rfcomm.h>
#include "out.h"

#define PIC_IN 0
#define PIC_OUT 0
#define PIC_CLK 1
#define PIC_RST 6
#define PIC_POWER 2
#define INBUFSIZE 131072

static unsigned char port_st[2]={0,0};
#define port_state (port_st[1])
static int sock;
static int do_delay;
static unsigned char out_buf[512];
static int out_bufpos;
static unsigned char *in_buf;
static int inbuf_readpos;
static int inbuf_writepos;

#define SET_BIT(bit,state) port_state=(state)?(port_state|(1<<(bit))):(port_state&~(1<<(bit)))
static void set_block();
							     
static int search_btname(char *name, bdaddr_t *result, int timeout)
{ 
  char addr[80];
  inquiry_info *info=NULL;
  int i;
  int found=0;
  static int found_class=0;
  int dd=hci_open_dev(0);
  /* get a list of all available devices */
  int num_dev;
  num_dev=hci_inquiry(0,timeout*8/10,100,NULL,&info,found_class?0:IREQ_CACHE_FLUSH);
  if (num_dev<0){
    fprintf(stderr,"searching devices failed\n");
    sleep(1);
  }
  found_class=0;
  fprintf(stderr,"found %d devices\n",num_dev);
  for(i=0;i<num_dev;i++) {
    ba2str(&info[i].bdaddr,addr);
    fprintf(stderr,"%s\n",addr);
  }
  for(i=0;i<num_dev;i++){
    char buf[80];
    /* check major device class, the value is only valid
     * for PromiESD-Devices */
    if (info[i].dev_class[1]==0x1f) {
      found_class=1;
      ba2str(&info[i].bdaddr,addr);
      fprintf(stderr,"getting name for %s: ",addr);
      /* get the bluetooth name */
      if (hci_read_remote_name(dd,&info[i].bdaddr,sizeof(buf),buf,100000)<0) {
        fprintf(stderr,"failed\n");
      } else {
        fprintf(stderr,"%s: ",buf)
          /* compare the device name */;
        if (strcmp(name,buf)==0) {
          found=1;
          *result=info[i].bdaddr;
          fprintf(stderr,"found!\n");
          /* right device, done */
          break;
        } else {
          fprintf(stderr,"nothing interesting\n");
        }
      }
              
    } 
  }
  if (!found) {
    sleep(1);
  }
  if (!found_class)  {
    fprintf(stderr,"no devices with the right class found\n");
  }
  hci_close_dev(dd);
  return found;
}
/* create a rfcomm connection */
int open_bluetooth(char *mac, int channel)
{
  bdaddr_t local;
  int fd;
  char buf[80];
  struct sockaddr_rc rcaddr;  
  fd=socket(PF_BLUETOOTH,SOCK_STREAM,BTPROTO_RFCOMM);
  if (fd<0) {
    perror("socket(): ");
    return -1;
  }
  hci_devba(0, &local);
  bacpy(&rcaddr.rc_bdaddr,&local);  
  rcaddr.rc_family=AF_BLUETOOTH;  
  rcaddr.rc_channel=0;
  if (bind(fd,(struct sockaddr *)&rcaddr,sizeof(rcaddr))) {
    perror("bind: ");
    close(fd);
    return -1;
  }
  rcaddr.rc_channel=channel;
  if (mac[0]==':') {
    /* try to find out the bluetooth addr */ 
    while(!search_btname(mac+1,&rcaddr.rc_bdaddr,10));
  } else {
    str2ba(mac,&rcaddr.rc_bdaddr);
  }
  set_block(fd);
  if (connect(fd,(struct sockaddr *)&rcaddr,sizeof(rcaddr))) {
    perror("connect: ");
    close(fd);
    return -1;
  }
  ba2str(&rcaddr.rc_bdaddr,buf);
  return fd;
}



static void out_sync()
{
  set_block();
  if (do_delay) {
    
    do_delay=0;
    int len=write(sock,out_buf,out_bufpos);
    if (len!=out_bufpos)
      fprintf(stderr,"written %d/%d bytes\n",len,out_bufpos);
    out_bufpos=0;
  } else {
    write(sock,port_st,2);
  }
}

static void out_freeze()
{
  do_delay=1;
}

static void out_append(unsigned char *buf, int len)
{
  if (!do_delay) {
    write(sock,buf,len);
  } else {
    if ((out_bufpos+len)>sizeof(out_buf)) {
      int l;
      l=write(sock,out_buf,out_bufpos);
      if (l!=out_bufpos) {
	fprintf(stderr,"wrote only %d/%d bytes\n",l,out_bufpos);
      }
      out_bufpos=0;
      
    }
    memcpy(out_buf+out_bufpos,buf,len);
    out_bufpos+=len;
  }
}
static void my_usleep(int usecs)
{
  int i;
  int bytes=usecs/300;
  for(i=0;i<bytes;i++) {
    out_append("\0",1);
  }
}
static void clk_pic(int state)
{
  SET_BIT(PIC_CLK,state);
}

static void out_pic(int state)
{
  port_state&=(~32);
  SET_BIT(PIC_OUT,state);
}

static void power_pic(int state)
{
  SET_BIT(PIC_POWER,state);
}

static void mclr_pic(int state)
{
  SET_BIT(PIC_RST,state);
}

static void set_nonblock()
{
  fcntl(sock,F_SETFL,fcntl(sock,F_GETFL)|O_NONBLOCK);
}

static void set_block()
{
  fcntl(sock,F_SETFL,fcntl(sock,F_GETFL)&(~O_NONBLOCK));
}

static int  in_pic()  /* read paper out */
{
  char buf[4];
  fcntl(sock,F_SETFL,fcntl(sock,F_GETFL)|O_NONBLOCK);
  while(read(sock,buf,4)>0) {
    fprintf(stderr,"got junk\n");
  }
  port_state|=32;
  out_sync(); 
  fcntl(sock,F_SETFL,fcntl(sock,F_GETFL)&(~O_NONBLOCK));
  read(sock,buf,1);
  return(buf[0]&1);
 
}

static int pic_setup(char *arg)
{
  char *initcmd="\nPROGEXT\n";
  int channel=1;
  char *sl;
  sl=strchr(arg,'/');
  if (sl) {
    sl[0]=0;
    sl++;
    channel=atoi(sl);
  }
  sock=open_bluetooth(arg,channel);
  if (sock<0) {
    printf("cannot connect to %s\n",arg);
    return 0;
  }
  sleep(1);
  write(sock,"\nCONNECT\n",strlen("\nCONNECT\n"));
  sleep(1);
  in_buf=malloc(INBUFSIZE);
  write(sock,initcmd,strlen(initcmd));
  sleep(2);
  
  port_state|=128;
  clk_pic(0);
  out_pic(0);
  set_nonblock();
  while(read(sock,in_buf,sizeof(in_buf))>0) {
    fprintf(stderr,"flushing input\n");
  }
  set_block();
  return 1;
}


static void append_to_input_buf(int len)
{
  int l;
  if (len<0) {
    set_nonblock();
    len=INBUFSIZE-inbuf_writepos;
  } else {
    set_block();
    l=inbuf_writepos-inbuf_readpos;
    len=len-l;
  }
  while((len>0)&&((l=read(sock,in_buf+inbuf_writepos,len))>0)) {
    inbuf_writepos+=l;
    if ((inbuf_writepos>INBUFSIZE/2)&&(inbuf_readpos>INBUFSIZE/2)) {
      memmove(in_buf,in_buf+INBUFSIZE/2,INBUFSIZE/2);
      inbuf_writepos-=INBUFSIZE/2;
      inbuf_readpos-=INBUFSIZE/2;
    }
    len-=l;
  }
}

static int copy_bits(int count)
{
  int ret=0;
  if (do_delay)
    out_sync();
  append_to_input_buf(count/8);
  ret+=in_buf[inbuf_readpos];
  inbuf_readpos++;
  if (count>8) {
    ret+=(256*in_buf[inbuf_readpos]);
    inbuf_readpos++;
  }
  return ret;
}

static void read_bits(int count)
{
  int p;
  int l;
  unsigned char buf[4];
  unsigned char ps=port_state;
  p=0;
  my_usleep(500);
  ps=ps&(~15);
  ps|=16;
  /* out_sync(); */
  append_to_input_buf(-1);
  
  buf[0]=ps|32;
  buf[1]=buf[0];
  if (count>8) {
    buf[0]|=8;
    p++;
    count-=8;
  } else {
    buf[1]=0;
  }
  buf[p]|=count;
  
  out_append(buf,1);
  my_usleep(500);
  if (p)
    out_append(buf+1,1);
  my_usleep(500);
  /* out_sync(); */
#if 0  
  read(sock,buf,1);
  out_sync(); 
  if (p) {
    /* write(sock,buf+1,1); */
    read(sock,buf+1,1);
  }
  out_sync();
  
  return buf[0]+(buf[1]*256);
#endif
}


static void write_bits(unsigned short word, int count)
{
  unsigned char ps=port_state&(~0x7);
  unsigned char buf[4];
  int p;  
  p=0;
  ps|=16;
  if (count>8) {
    buf[0]=ps|8;
    buf[1]=(word&255);
    word/=256;
    p=2;
    count-=8;
  }
  buf[p]=ps|count;
  buf[p+1]=(word&255);
  out_append(buf,p+2);
 
  /* out_sync(); */
}



struct out_funcs bt_out_funcs ={
  "bt",
  pic_setup,
  out_sync,
  out_freeze,
  power_pic,
  mclr_pic,
  copy_bits,
  read_bits,
  write_bits,
  my_usleep,
  clk_pic,
  out_pic,
  in_pic
};
