/********************************************************************** 
 *  picprog - Copyright (C) 2003-2006 - Andreas Kemnade
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *                  
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied
 *  warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 ************************************************************************/

#include <sys/io.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>
#include <sys/time.h>
#include <unistd.h>
#include <fcntl.h>
#include "out.h"

extern struct out_funcs bt_out_funcs;
extern struct out_funcs par_out_funcs;

struct out_funcs *out_funcs_list[]={&bt_out_funcs, &par_out_funcs, NULL};

static struct out_funcs *outdevice;

#define LDCONF 0
#define LDPROG 2
#define RDPROG 4
#define INCADD 6
#define CHIPDEL 9
#define BEGPRG 8
#define BULKMEM 0x9
#define BULKDDATA 0xb
#define ENDPRG 0xe
/* #define CHIPDEL 0x1f */
#define LDDATA 3
#define RDDATA 5
#define EPMODE 10

#define VPPDLY 200000
#define PRGDLY  10000
#define TSET 1
#define THLD 1
#define TDLY 2

#define PROGRAM 0
#define DATA 1
#define PSIZE 512
#define DSIZE 64


#define FUSE_CP 16
#define FUSE_PWRTE 8
#define FUSE_WDTE 4
#define FUSE_RC 3
#define FUSE_HS 2
#define FUSE_XT 1
#define FUSE_LP 0

#define NO_OP 0x20

char inbuf[0x2200];
short progbuf[PSIZE];
short progbuf2[PSIZE];
short fusewords[1]={0};
short fusewords_found=0;



static void reset_pic()
{
  /*  clk_pic(0);
      out_pic(0); */
  outdevice->reset(1);
  outdevice->out_sync();
  outdevice->power(0);
  outdevice->out_sync();
  usleep(1000000);
  outdevice->power(1);
#if 0
  outdevice->out_sync();
  outdevice->sleepu(50);
#endif
  outdevice->reset(0);
  outdevice->out_sync();
  
}



static void prog_mode()
{
  /*
  clk_pic(0);
  out_pic(0); */
  reset_pic();
  usleep(VPPDLY);
  sleep(1);
}




static void out_word(short word) 
{
  unsigned short w;
  w=word;
  w*=2;
  outdevice->out_bits(w,16);
  
}

static void in_word_prep()
{
  outdevice->in_bits_request(16);
}

static int in_word()
{
 
  short ret;
  ;
  ret=outdevice->in_bits(16)>>1;
  ret&=0x3fff;
  return ret;

}

static void command(int cmd)
{
  outdevice->sleepu(TDLY);
  outdevice->out_bits(cmd,6);
  return;

}

static void erase()
{
  prog_mode();
  command(INCADD);
  command(CHIPDEL);
}

#if 0
static void erase()
{
  int i;
  prog_mode();
 command(LDCONF);
  out_word(0x3fff);
  for(i=0;i<7;i++)
    command(INCADD);
  command(1);
  command(7);
  command(BEGPRG);
  usleep(PRGDLY);
  command(1);
  command(7);
}
#endif

static void program(int do_verify)
{
  int i,j;
  prog_mode();
  printf("\n");
  command(INCADD);
  for(i=0,j=0;i<PSIZE;i++) {
    short ret;
    outdevice->out_freeze();
    command(LDPROG);
    out_word(progbuf[i]);
    command(BEGPRG);
    outdevice->sleepu(PRGDLY);
    if (!do_verify) {
      outdevice->out_freeze();
    }
    command(ENDPRG);
    if (do_verify) {
      command(RDPROG);
      in_word_prep();
      
    
      
    } else {
      outdevice->out_freeze();
    }
    command(INCADD);
    
    if (!(i&0x3f)) {
      fprintf(stderr,"\rPosition: %4d",i);
      if (do_verify)
	for(;j<i;j++) {
	  if (((ret=in_word())&0x3fff)!=progbuf[j]) {
	    fprintf(stderr,"verify error at %d: %x!=%x\n",j,(int)ret,(int)progbuf[j]);
	    return;
	    /*
	      if (ret==0x3fff)
	      return ;  */
	  }
	}
    } 
    
  }
  outdevice->out_sync();
  if (do_verify)
    for(;j<i;j++) {
      short ret;
      if (((ret=in_word())&0x3fff)!=progbuf[j]) {
	fprintf(stderr,"verify error at %d: %x!=%x\n",i,(int)ret,(int)progbuf[i]);
	return;
	/*
	  if (ret==0x3fff)
	  return ;  */
      }
    }
  fprintf(stderr,"\rPosition: %4d",i);
}

static char hexcode[512]="000102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f202122232425262728292a2b2c2d2e2f303132333435363738393a3b3c3d3e3f404142434445464748494a4b4c4d4e4f505152535455565758595a5b5c5d5e5f606162636465666768696a6b6c6d6e6f707172737475767778797a7b7c7d7e7f808182838485868788898a8b8c8d8e8f909192939495969798999a9b9c9d9e9fa0a1a2a3a4a5a6a7a8a9aaabacadaeafb0b1b2b3b4b5b6b7b8b9babbbcbdbebfc0c1c2c3c4c5c6c7c8c9cacbcccdcecfd0d1d2d3d4d5d6d7d8d9dadbdcdddedfe0e1e2e3e4e5e6e7e8e9eaebecedeeeff0f1f2f3f4f5f6f7f8f9fafbfcfdfeff";

static void out_hexbyte(char *str,int inp,int *sum)
{
  str[0]=hexcode[inp*2];
  str[1]=hexcode[inp*2+1];
  *sum=*sum+inp;
}

static void readprog_bin(short *buf)
{
  int i;
  prog_mode();
  outdevice->out_freeze();
  for(i=0;i<PSIZE;i++) {
    command(RDPROG);
    in_word_prep();
    command(INCADD);
    if (!(i&0xf)) {
      fprintf(stderr,"%cPosition: %4d",0xd,i);
      
    }
  }
  for(i=0;i<PSIZE;i++) {
    buf[i]=in_word()&0x3fff;
    if (!(i&0xf)) {
      fprintf(stderr,"%cPosition: %4d",0xd,i);
    } 
  }

}

static void readprog(char *fname)
{
  int i;
  int fd;
  int lbpos;
  int sum;
  short buf[PSIZE];
  char linebuf[1024];
  readprog_bin(progbuf2);
  fd=open(fname,O_CREAT|O_WRONLY,0666);
  lbpos=0;
  sum=0;
  for(i=0;i<PSIZE;i++) {
    int w;
    if (!(i&0x7)) {
      int ll=2*(PSIZE-i);
      if (ll>0x10) 
	ll=0x10;
      if (lbpos) {
	sum=0-sum;
	out_hexbyte(linebuf+lbpos,sum&255,&sum);
	lbpos+=2;
        linebuf[lbpos]='\n';
	write(fd,linebuf,lbpos+1);
      }
      sum=0;
      lbpos=0;
      linebuf[0]=':';
      lbpos++;
      out_hexbyte(linebuf+lbpos,ll,&sum);
      lbpos+=2;
      out_hexbyte(linebuf+lbpos,(i*2)/256,&sum);
      lbpos+=2;
      out_hexbyte(linebuf+lbpos,(i*2)&255,&sum);
      lbpos+=2;
      /* type */
      out_hexbyte(linebuf+lbpos,0,&sum);
      lbpos+=2;
    }
    w=progbuf2[i];
    out_hexbyte(linebuf+lbpos,w&255,&sum);
    lbpos+=2;
    out_hexbyte(linebuf+lbpos,w/256,&sum);
    lbpos+=2;
  }
  if (lbpos) {
    sum=0-sum;
    out_hexbyte(linebuf+lbpos,sum&255,&sum);
    lbpos+=2;
    linebuf[lbpos]='\n';
    write(fd,linebuf,lbpos+1);
  }
  write(fd,":00000001FF\n",11);
  close(fd);
}

static void verify()
{
  int i;
  readprog_bin(progbuf2);
  for(i=0;i<PSIZE;i++) {
    if (progbuf2[i]!=progbuf[i]) {
      fprintf(stderr,"verify error at %d: %x!=%x\n",i,(int)progbuf2[i],(int)progbuf[i]);
      /*fprintf(stderr,"verify error at %d\n",i);
	return ; */
    }
    
    if (!(i&0xf)) {
      fprintf(stderr,"%cPosition: %4d",0xd,i);
      
    }
    
  }
}

static int hexbyte(char *buf) 
{
  char p[3];
  p[2]=0;
  p[0]=buf[0];
  p[1]=buf[1];
  return strtol(p,NULL,16);
}

static int hexword(char *buf) 
{
  char p[5];
  p[4]=0;
  p[3]=buf[3];
  p[2]=buf[2];
  p[1]=buf[1];
  p[0]=buf[0];
  return strtol(p,NULL,16);
}

static int hexword_swap(char *buf)
{
  char p[5];
  p[4]=0;
  p[3]=buf[1];
  p[2]=buf[0];
  p[1]=buf[3];
  p[0]=buf[2];
  return strtol(p,NULL,16);
}

static void config_pic()
{
  int i;
  short ret;
  prog_mode();
  command(LDPROG);
  out_word(fusewords[0]);
  command(BEGPRG);
  usleep(PRGDLY);
  command(ENDPRG);
  command(RDPROG);
  in_word_prep();
  if ((ret=(in_word()&0x1f))!=fusewords[0]) {
    fprintf(stderr,"setting config (2007) failed %x!=%x\n",
	    (int)ret,(int)fusewords[0]);
  }
}

static int load_file(char *fname)
{
  int len;
 
  int inpos=0;
  unsigned int type=0;
  int line=0;
  FILE *f=fopen(fname,"r");
  if (!f)
    return 0;
  fseek(f,0,SEEK_END);
  len=ftell(f);
  fseek(f,0,SEEK_SET);
  if (len>sizeof(inbuf)) {
    fclose(f);
    return 0;
  }
  fread(inbuf,1,len,f);
  fclose(f);
  while((type!=1)&&(inpos+12<=sizeof(inbuf))) {
    int i;
    unsigned int linelen;
    unsigned int adr;
    line++;
    if (inbuf[inpos]!=':') {
      fprintf(stderr,"error in hex file\n");
      return 0;
    }
    inpos++;
    linelen=hexbyte(inbuf+inpos);
    inpos+=2;
    adr=hexword(inbuf+inpos);
    inpos+=4;
    type=hexbyte(inbuf+inpos);
    inpos+=2;
    printf("type: %x adr: %x len: %x\n",type,adr,linelen);
    adr/=2;
    if ((adr<0x200)&&(adr+linelen/2>PSIZE)) {
      fprintf(stderr,"ignoring type: %x address: %x len: %x\n",(int)type,(int)adr,(int)linelen);
      inpos+=2*linelen;
    } else {
      for(i=0;i<linelen/2;i++) {
	if (adr==0x3ff) {
          fusewords[0]=0x3fff&hexword_swap(inbuf+inpos);
	  fusewords_found=1;
	} else if (adr<PSIZE) {
	  progbuf[adr]=0x3fff&hexword_swap(inbuf+inpos);
	}
	inpos+=4;
	adr++;
      }
    }
    inpos+=2;
    if (inbuf[inpos]==0xd)
      inpos++;
    inpos++;
  }
  return 1;
}

static short get_ident()
{
  int i;
  prog_mode();
  command(LDCONF);
  out_word(0);
  for(i=0;i<6;i++)
    command(INCADD);
  command(RDPROG);
  in_word_prep();
  return in_word();
}

int main(int argc, char **argv)
{
  
  int i;
  char *devarg;
  if (((argc!=3)&&(argc!=4))||(!(devarg=strchr(argv[1],':')))) {
    printf("Usage:\n%s device power on|off\n"
	   "%s device ident\n"
	   "%s device erase\n"
	   "%s device verify hexfile\n"
	   "%s device prog hexfile\n"
	   "%s device progverify hexfile\n"
	   "%s device config configword\n"
	   "\nhexfile is a file in the default format of gpasm\n"
	   "\ndevice can be either parport:<ioport>, e.g. parport:0x3bc\n"
	   "or bt:mac[/channel], e.g. bt:00:11:22:33:44:55\n"
	   "or bt::name[/channel], e.g. bt::Scipio\n",
	   argv[0],argv[0],argv[0],argv[0],argv[0],argv[0],argv[0]);
    return 1;
  }
  devarg[0]=0;
  devarg++;
  outdevice=NULL;
 
  for(i=0;out_funcs_list[i];i++) {
    if (!strcmp(out_funcs_list[i]->prefix,argv[1])) {
      outdevice=out_funcs_list[i];
      if (!outdevice->init(devarg)) {
	fprintf(stderr,"Cannot init output device\n");
	return 1;
      }
    }
  }
  if (!outdevice) {
    fprintf(stderr,"no output device found\n");
    return 1;
  }
  /* setuid(getuid()); */
  if ((argc==4)&&((!strcmp(argv[2],"prog"))||(!strcmp(argv[2],"progverify")))) {
    if (!load_file(argv[3])) {
      fprintf(stderr,"loading %s failed\n",argv[3]);
      return 1;
    }
    if (!strcmp(argv[2],"progverify")) {
      program(1);
    } else {
      program(0);
    }
    get_ident();
    if (fusewords_found) {
      fprintf(stderr,"configuring 0x3ff to: %x\n",fusewords[0]);
      config_pic();
    }
  } else if ((!strcmp(argv[2],"erase"))) {
    erase();
  } else if ((argc==4)&&(!strcmp(argv[2],"verify"))) {
    if (!load_file(argv[3])) {
      fprintf(stderr,"loading %s failed\n",argv[3]);
      return 1;
    }
    verify();
  } else if ((argc==4)&&(!strcmp(argv[2],"read"))) {
    readprog(argv[3]);  
  } else if ((argc==4)&&(!strcmp(argv[2],"config"))) {
    fusewords[0]=strtol(argv[3],NULL,0);
    config_pic();
  } else if ((argc==4)&&(!strcmp(argv[2],"power"))) {
    outdevice->reset(1);
    outdevice->out_sync();
    usleep(10000);
    outdevice->power(0);
    outdevice->out_sync();
    if (strcmp(argv[3],"on")==0) {
      usleep(1000000);
      outdevice->power(1);
      outdevice->out_sync();
    }
  } else if ((!strcmp(argv[2],"ident"))) {
    short identword=get_ident();
    char *str=NULL;
    switch(identword>>5) {
     case 0x39: str="PIC16F87"; break;
     case 0x3b: str="PIC16F88"; break;		      
     case 0x68: str="PIC16F870"; break;
     case 0x69: str="PIC16F871"; break;
     case 0x47: str="PIC16F872"; break;
     case 0x4b: str="PIC16F873"; break;
     case 0x49: str="PIC16F874"; break;
     case 0x4f: str="PIC16F876"; break;
     case 0x4d: str="PIC16F877"; break;
     case 0x3d: str="PIC16F627"; break;
     case 0x3e: str="PIC16F628"; break;
    }
    if (str) {
      printf("Device: %s, Rev: %d\n",str,(int)(identword>>4)&3);
    } else {
      printf("Unknown device, ident=%x\n",identword);
    }
  } else {
    fprintf(stderr,"unknown command\n");
  }
  
  return 0;
}
