#include <unistd.h>
#include <fcntl.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>
#include <sys/time.h>
#include <sys/select.h>
#include <signal.h>
#include <termios.h>
#include <syslog.h>
#include <sys/ioctl.h>
#ifndef DISABLE_UDEV_IFACE
#include <linux/input.h>
#include <linux/uinput.h>
#endif
#include <X11/Xlib.h>
#include <X11/extensions/XTest.h>
#include <netinet/in.h>
#include <netdb.h>
#include <arpa/inet.h>
#include <sys/socket.h>

#include <sys/types.h>
#include <sys/un.h>

#include <bluetooth/bluetooth.h>
#include <bluetooth/hci.h>
#include <bluetooth/hci_lib.h>
#include <bluetooth/rfcomm.h>
#define SHUNT_VALUE 0.05

#include <math.h>
#include <string.h>

// Constants for gesture detection

double smooth=0.2;
double xsmooth=0.02;

#define GXSET -100      // set point X for interaction state detection
#define GYSET -150      // set point Y for interaction state detection
#define GZSET -110      // set point Z for interaction state detection

#define GXLSET -100     // set point X for interaction state detection
#define GYLSET 0        // set point Y for interaction state detection
#define GZLSET -85      // set point Z for interaction state detection

#define VZSET -160      // set point Z for voice state detection

#define GDELTA 30       // maximum difference between set points and value for interaction state detection
#define GDELTA2 5       // minimum difference between xsmooth and smooth signal for gesture start detection
#define GDELTAT 5      // gesture start condition must be true for this number of cycles
#define GDELTATDEAD 150 // gesture detection inhibited for this number of cycles
#define GDELTATSTART 100 // gesture detection begins after this number of cycles




#define INITZCMD "\n PROTOCOL SENDZ ON\n"


static int destfd;
static int use_x11;
Display *dpy;
static int gesturemode=0; /* Recognizes gestures and emits kepress events */
static int recorder=0;    /* Emits motion data to RFID file */ 
static int enable_z=0;    /* Enables Z-Axis mode */
static int lefthand=0;    /* Switches to left hand mode */
static int voicectl=0;    /* Controls gspeakfreely input via unix socket */

static int height=640;
static int width=480;
static int use_abs;
static int last_scanned;
static double exps=0.3;  //Factor for exponential smoothing



// Butterworth coeffizienten
#define OMEGA    0.0726255766802706   // tan(M_PI / 130 * 3)
#define OMEGASQ  0.00527447438814186  // OMEGA * OMEGA
#define W0       0.0726255766802706   // 2 *(OMEGA * cos(M_PI / 3))
#define W1       1.07790005106841     // 1 + W0 + OMEGASQ

static double c[3][5][2], d[3][2][2];

enum gesturestate { idle, maybe, deadtime } gstate;


static int minx=-256; // Ranges of the measurement device
static int maxx=256;
static int miny=-256;
static int maxy=256;
static int listen_sock=-1;
static int beverbose=0;
static int use_syslog=0;

char *rfidfilename;
char lastrfid[256];

// Keyboard Interface
char* keyb_filename=NULL;
int keyb_lastpressed = 0;
int keyb_keycount=17; // number of keys + nothing pressed value
int keyb_analog[]={ -256, // nothing pressed
		    196,  148,  110,   78,
		    50,   27,    9,   -9,
		    -24,  -37,  -50,  -60, 
		    -70,  -78,  -85,  -92 };
char keyb_charcodes[] = { ' ',
			  '1', '2', '3', 'o',
			  '4', '5', '6', 'n',
			  '7', '8', '9', 'a',
			  'l', '0', 'd', 'm' };
int keyb_tolerance = 3;
int keyb_testcounter=0;
int keyb_okcounter=3;

struct t_battery_state {
  int raac;
  int raac_p;
  int rsac;
  int rsac_p;
  int rarc;
  int rarc_p;
  int rsrc;
  int rsrc_p;
  short iavg;
  double iavg_p;
  short temp;
  double temp_p;
  int volt;
  double volt_p;
  short current;
  double current_p;
  int acr;
  double acr_p;
  int acrl;
  double acrl_p;
  int as;
  int as_p;
  int full;
  int full_p;
  int ae;
  int ae_p;
  int se;
  int se_p;
};

void request_process_init();


static int search_btname(char *name, bdaddr_t *result, int timeout) 
{
  char addr[80];
  inquiry_info *info=NULL;
  int i;
  int found=0;
  static int found_class=0;
  int dd=hci_open_dev(0);
  /* get a list of all available devices */
  int num_dev;
  if (use_syslog)
    syslog(LOG_INFO,"starting to search for devices with major class=0x1f and name: %s\n",name);
  num_dev=hci_inquiry(0,timeout*8/10,100,NULL,&info,found_class?0:IREQ_CACHE_FLUSH);
  if (num_dev<0){
    fprintf(stderr,"searching devices failed\n");
    sleep(1);
  }
  found_class=0;
  fprintf(stderr,"found %d devices\n",num_dev);
  for(i=0;i<num_dev;i++) {
    ba2str(&info[i].bdaddr,addr);
    fprintf(stderr,"%s\n",addr);
  }
  for(i=0;i<num_dev;i++){
    char buf[80];
    /* check major device class, the value is only valid
     * for PromiESD-Devices */
    if (info[i].dev_class[1]==0x1f) { 
      found_class=1;
      ba2str(&info[i].bdaddr,addr);
      fprintf(stderr,"getting name for %s: ",addr);
      /* get the bluetooth name */
      if (hci_read_remote_name(dd,&info[i].bdaddr,sizeof(buf),buf,100000)<0) {
        fprintf(stderr,"failed\n");
      } else {
        fprintf(stderr,"%s: ",buf)
	  /* compare the device name */;
        if (strcmp(name,buf)==0) {
          found=1;
          *result=info[i].bdaddr;
          fprintf(stderr,"found!\n");
          /* right device, done */
          break;
        } else {
          fprintf(stderr,"nothing interesting\n");
        }
      }
              
    } 
  }
  if (!found) {
    syslog(LOG_INFO,"no devices found");
    sleep(1);
  }
  if (!found_class)  {
    fprintf(stderr,"no devices with the right class found\n");
  }
  hci_close_dev(dd);
  return found;
}


/* open serial device, set the baudrate and set mode to raw mode */

static int open_ser(char *fname, int bps)
{
  struct termios mytermios;
  int b;
  int fd=open(fname,O_RDWR);
  if (fd<0) {
    fprintf(stderr,"Cannot open connection to %s\n",fname);
    return -1;
  }
  tcgetattr(fd, &mytermios);
  cfmakeraw(&mytermios);
  mytermios.c_cflag&=(~CRTSCTS);
  /* mytermios.c_cflag&=(~CLOCAL); */
  b=B115200;
  switch(bps) {
  case 9600: b=B9600; break;
  case 19200: b=B19200; break;
  case 38400: b=B38400; break;
  case 57600: b=B57600; break;
  case 115200: b=B115200; break;
  default: fprintf(stderr,"invalid rate: %d\n",bps);
  }
  cfsetospeed(&mytermios,b);
  cfsetispeed(&mytermios,B0);
  tcsetattr(fd,TCSANOW,&mytermios);
  tcflush(fd,TCIFLUSH);
  tcflush(fd,TCOFLUSH); 
  return fd;
}

/* create a tcp connection */
int open_tcp(char *host, int port)
{
  struct hostent *ph=gethostbyname(host);
  int sock;
  struct sockaddr_in addr;
  addr.sin_family=AF_INET;
  addr.sin_port=htons(port);
  if (!ph) {
    return -1;
  }
  sock=socket(AF_INET,SOCK_STREAM,0);
  memcpy((char *) &addr.sin_addr,ph->h_addr,ph->h_length); 
  if (sock<0) {
    return -1;
  } 
  if (connect(sock,(struct sockaddr *)&addr,sizeof(addr))) {
    close(sock);
    return -1;
  }
  return sock;
  
}

/* create a rfcomm connection */
int open_bluetooth(char *mac, int channel)
{
  bdaddr_t local;
  int fd;
  char buf[80];
  struct sockaddr_rc rcaddr;  
  fd=socket(PF_BLUETOOTH,SOCK_STREAM,BTPROTO_RFCOMM);
  if (fd<0) {
    perror("socket(): ");
    return -1;
  }
  hci_devba(0, &local);
  bacpy(&rcaddr.rc_bdaddr,&local);  
  rcaddr.rc_family=AF_BLUETOOTH;  
  rcaddr.rc_channel=0;
  if (bind(fd,(struct sockaddr *)&rcaddr,sizeof(rcaddr))) {
    perror("bind: ");
    close(fd);
    return -1;
  }
  rcaddr.rc_channel=channel;
  if (mac[0]==':') {
    /* try to find out the bluetooth addr */ 
    while(!search_btname(mac+1,&rcaddr.rc_bdaddr,10));
  } else {
    str2ba(mac,&rcaddr.rc_bdaddr);
  }
  if (connect(fd,(struct sockaddr *)&rcaddr,sizeof(rcaddr))) {
    perror("connect: ");
    close(fd);
    return -1;
  }
  ba2str(&rcaddr.rc_bdaddr,buf);
  if (use_syslog)
    syslog(LOG_INFO,"sucessfully connected to %s", buf);

  if(enable_z)
    {
      if (beverbose) {printf("Enabling Z Mode\n"); fflush(stdout);}
      write(fd,INITZCMD,strlen(INITZCMD));
      write(fd,INITZCMD,strlen(INITZCMD));
    }


  return fd;
}

/* handle an input line from the rfid scanner */
static void handle_rfid_input(char *buf, int len)
{
   char rfidbuf[256];
   memcpy(rfidbuf,buf,len);
   rfidbuf[strcspn(rfidbuf,"\r\n")]=0;
   if ((time(NULL)>last_scanned)||strcmp(rfidbuf,lastrfid)) {
     FILE *f;
     last_scanned=time(NULL);
     if( beverbose )
       printf("rfid scanned: %s\n",rfidbuf);
     if (use_syslog) {
       syslog(LOG_INFO,"rfid scanned: %s",rfidbuf);
     }
     strcpy(lastrfid,rfidbuf);

     if (!recorder)     unlink(rfidfilename); /* only unlink if we're not in recorder mode */

     //     f=fopen(rfidfilename,"w");
     f=fopen(rfidfilename,"a"); // Same effect if the file was unlinked before.
     if (f) {

       if (recorder) {
	 fprintf(f,"RFID:%s\n",lastrfid);
       } else {
	 fputs(lastrfid,f);
       }
       fclose(f);
     }
   }
}

/* handle input from the serial interface on the scipio board */
static void handle_ser_input(char ser)
{
   static char serbuf[128];
   static int readpos;
   static int pos;
   serbuf[pos]=ser;
   pos++;
   /* full line read -> a rfid tag was scanned */
   if ((ser==0xa)&&(pos>=2)&&(serbuf[pos-2]==0xd)) {
     handle_rfid_input(serbuf+readpos,pos-readpos);
     readpos=pos;
   }
   /*  buffer is full */
   if (pos>(sizeof(serbuf)-1)) {
     /* delete stuff which was already read */
     if (readpos) {
       memmove(serbuf,serbuf+readpos,pos-readpos);
       pos-=readpos;
       readpos=0;
     } else {
       /* nothing read, drop whole buffer */
       fputs("buffer overrun, dropping: \n",stderr);
       fwrite(serbuf,1,pos,stderr);
       pos=0;
     }
   }
}

/* checks for a type number */
//#define CHKI(x) (((x)&0xf8)==0x40)
#define CHKI(x) (((x)&0xf0)==0x40) /* test more bits */

static int timed_read_byte(int fd, unsigned char *out)
{
  fd_set rfds;
  struct timeval tv;
  tv.tv_sec=1;
  tv.tv_usec=0;
  FD_ZERO(&rfds);
  FD_SET(fd,&rfds);
  if (select(fd+1,&rfds,NULL,NULL,&tv)>0) {
    return read(fd,out,1);
  } else {
    return 0;
  }
}

/* output battery state */
static void output_bs(struct t_battery_state *bs, char *buf, int len)
{
  snprintf(buf,len,"RAAC (Remaining Active Absolute Capacity): %d\n"
		  "RSAC (Remaining Standby Absolute Capacity): %d\n"
		  "RARC (Remaining Active Relative Capacity): %d (%d%%)\n"
		  "RSRC (Remainint Standby Relative Capacity): %d (%d%%)\n"
		  "IAVG (Average Current): %d (%.2f mA)\n"
		  "TEMP (Temperature): %d (%.2f C)\n"
		  "VOLT (Voltage): %d (%.3f V)\n"
		  "CURRENT (Current): %d (%.2f mA)\n"
		  "ACR (Accumulated Current): %d\n"
		  "ACRL (Low Accumulated Current): %d\n"
		  "AS (Age Scalar): %d\n"
		  "FULL (Full Capacity): %d\n"
		  "AE (Active Empty): %d\n"
		  "SE (Standby Empty): %d\n",
		  bs->raac,
		  bs->rsac,
		  bs->rarc,bs->rarc_p,
		  bs->rsrc,bs->rsrc_p,
		  bs->iavg,bs->iavg_p,
		  bs->temp,bs->temp_p,
		  bs->volt,bs->volt_p,
		  bs->current,bs->current_p,
		  bs->acr,
		  bs->acrl,
		  bs->as,
		  bs->full,
		  bs->ae,
		  bs->se);
}

/* handle onewire traffic to the battery port */
static void handle_batteryport()
{
  int i;
  char outbuf[4096];
  unsigned char inbuf[32];
  int s=accept(listen_sock,NULL,NULL); 
  if (s<0) {
    return;
  }
  fcntl(s,F_SETFL,fcntl(s,F_GETFL)|O_NONBLOCK); 
  write(destfd,"\n1WIRE\n1WIRE\n",13);
  usleep(100000);
  while(read(destfd,outbuf,sizeof(outbuf))>0);
  usleep(100000); 
  if (read(destfd,outbuf,sizeof(outbuf))>0) {
    char *answ="battery access not supported by firmware\n";
    write(s,answ,strlen(answ));
    close(s);
  write(destfd,"\x90",1);
  write(destfd,"\x90",1);
    return;
  }
  write(destfd,"\xa0",1);
  if (1!=timed_read_byte(destfd,inbuf)) {
    char *answ="read failed\n";
    write(s,answ,strlen(answ));
    close(s);
    write(destfd,"\x90",1);
    return;
  } else if (inbuf[0]==0) {
    char *answ="no 1wire device found\n";
    write(s,answ,strlen(answ));
    close(s);
    write(destfd,"\x90",1);
    return;
  }
  write(destfd,"\x80\xcc",2);
  write(destfd,"\x80\x69",2);
  write(destfd,"\x80\x02",2);
  for(i=2;i<0x1c;i++) {
    outbuf[(i-2)*4]=0;
    outbuf[(i-2)*4+1]=0xc0;
    outbuf[(i-2)*4+2]=0;
    outbuf[(i-2)*4+3]=0;
  }
  write(destfd,outbuf,4*(0x1c-2));
  for(i=2;i<0x1c;i++) {
    if (1!=timed_read_byte(destfd,inbuf+i)) {
      char *answ="read failed\n";
      write(s,answ,strlen(answ)); 
      break;
    }
  }
  if (i==0x1c) {
    struct t_battery_state bs;
    bs.raac=inbuf[2]*256+inbuf[3];
    bs.rsac=inbuf[4]*256+inbuf[5];
    bs.rarc=inbuf[6];
    bs.rarc_p=bs.rarc*100/255;
    bs.rsrc=inbuf[7];
    bs.rsrc_p=bs.rsrc*100/255;
    bs.iavg=inbuf[8]*256+inbuf[9];
    bs.iavg_p=bs.iavg;
    bs.iavg_p=bs.iavg_p*1.5625/1000000.0/SHUNT_VALUE*1000.0;
    bs.temp=inbuf[0xa]*256+inbuf[0xb];
    bs.temp_p=bs.temp;
    bs.temp_p=bs.temp_p/256.0;
    bs.volt=inbuf[0xc]*256+inbuf[0xd];
    bs.volt_p=bs.volt/32;
    bs.volt_p=bs.volt_p*0.00488*2;
    bs.current=inbuf[0xe]*256+inbuf[0xf];
    bs.current_p=bs.current;
    bs.current_p=bs.current_p*1.5625/1000000.0/SHUNT_VALUE*1000.0;
    bs.acr=inbuf[0x10]*256+inbuf[0x11];
    bs.acrl=inbuf[0x12]*256+inbuf[0x13];
    bs.as=inbuf[0x14];
    bs.full=inbuf[0x16]*256+inbuf[0x17];
    bs.ae=inbuf[0x18]*256+inbuf[0x19];
    bs.se=inbuf[0x1a]*256+inbuf[0x1b];
    output_bs(&bs, outbuf,sizeof(outbuf));
    write(s,outbuf,strlen(outbuf));
  }
  close(s);
  write(destfd,"\x90",1);
}

/* extract x, y and button values from the input */
void get_values(char *name, int val, int *x, int *y, int *button)
{
   fd_set rfds;
   struct timeval tv;
   static char buf[128];
   static int len;
   static int pos;
   int foundmask=0; /* bit mask for the input found */
   /* repeat  until x, y and button values are found */
   while(foundmask!=7) {
    /* try to read enough so that 64 bytes are in the buffer */
    while(len<64) {
      int l;
      /* don't give up to open a connection to the scipio */
      while(destfd<0) {
        
	if (name[0]=='/') {
          destfd=open_ser(name,val);
        } else if (strchr(name,':')) {
          destfd=open_bluetooth(name,val);
        } else {
	  destfd=open_tcp(name,val);
	}
	if (destfd>=0) {
	  fprintf(stderr,"successfully connected\n");
	  fcntl(destfd,F_SETFL,fcntl(destfd,F_GETFL)|O_NONBLOCK);
	} else {
	   /* don't try to often to connect */
	  sleep(2);
	}
      }
      FD_ZERO(&rfds);
      tv.tv_sec=5;
      tv.tv_usec=0;
      FD_SET(destfd,&rfds);
      l=-1;
      if (select(destfd+1,&rfds,NULL,NULL,&tv)) {
        l=read(destfd,buf+len,128-len);
      } else if (use_syslog) {
        syslog(LOG_ERR,"no data received for 5 seconds, disconnecting");
      }
      if (l<=0) {
	char *btrestartcmd;
	close(destfd);
	destfd=-1;
	if (use_syslog) {
	  syslog(LOG_ERR,"connection terminated\n");
	}
	btrestartcmd=getenv("BTRESTART");
	if (btrestartcmd)
          system(btrestartcmd);
        continue;
      }
      if (listen_sock>0) {
        FD_ZERO(&rfds);
	FD_SET(listen_sock,&rfds);
	tv.tv_sec=0;
	tv.tv_usec=0;
	if (select(listen_sock+1,&rfds,NULL,NULL,&tv)>0) {
	  handle_batteryport();
	}
      }
      len+=l;
    }
    if (pos>32) {
      memmove(buf,buf+pos,len-pos);
      len-=pos;
      pos=0;
      continue;
    }
    /* synchronize with the input data */
    if (CHKI(buf[pos])&&CHKI(buf[pos+2])&&CHKI(buf[pos+4])&&CHKI(buf[pos+6])) {
      /* successfully found one type/value pair */
      switch(buf[pos]) {
      case 0x41: *x=(buf[pos+1]&0xff)-0x100; foundmask|=1; 
	if( beverbose ) printf( "X: %d\n", *x );
	break;
      case 0x42: *x=(buf[pos+1]&0xff);       foundmask|=1; 
	if( beverbose ) printf( "X: %d\n", *x );
	break;
      case 0x43: *y=(buf[pos+1]&0xff)-0x100; foundmask|=2; break;
      case 0x44: *y=(buf[pos+1]&0xff);       foundmask|=2; break;
      case 0x45: *button=buf[pos+1]&0xff;    foundmask|=4; break; 
      case 0x46: handle_ser_input(buf[pos+1]);             break;
      case 0x47:	break;
      }
      pos+=2;
    } else {
      /* nothing valid found, try to resynchronize */
      pos+=1;
    }
  }
   
}

void get_values_z(char *name, int val, int *x, int *y, int *z, int *button)
{
   fd_set rfds;
   struct timeval tv;
   static char buf[128];
   static int len;
   static int pos;
   int pcount=0;
   int foundmask=0; /* bit mask for the input found */
   /* repeat  until x, y, z and button values are found */
   while(foundmask!= 15) {
    /* try to read enough so that 64 bytes are in the buffer */
    while(len<64) {
      int l;
      /* don't give up to open a connection to the scipio */
      while(destfd<0) {
        
	if (name[0]=='/') {
          destfd=open_ser(name,val);
        } else if (strchr(name,':')) {
          destfd=open_bluetooth(name,val);
        } else {
	  destfd=open_tcp(name,val);
	}
	if (destfd>=0) {
	  fprintf(stderr,"successfully connected\n");
	  request_process_init(); // Request new initialisation of data processing functions
	  fcntl(destfd,F_SETFL,fcntl(destfd,F_GETFL)|O_NONBLOCK);
	} else {
	   /* don't try to often to connect */
	  sleep(2);
	}
      }
      FD_ZERO(&rfds);
      tv.tv_sec=5;
      tv.tv_usec=0;
      FD_SET(destfd,&rfds);
      l=-1;
      if (select(destfd+1,&rfds,NULL,NULL,&tv)) {
        l=read(destfd,buf+len,128-len);
      } else if (use_syslog) {
        syslog(LOG_ERR,"no data received for 5 seconds, disconnecting");
      }
      if (l<=0) {
	char *btrestartcmd;
	close(destfd);
	destfd=-1;
	if (use_syslog) {
	  syslog(LOG_ERR,"connection terminated\n");
	}
	btrestartcmd=getenv("BTRESTART");
	if (btrestartcmd)
          system(btrestartcmd);
        continue;
      }
      if (listen_sock>0) {
        FD_ZERO(&rfds);
	FD_SET(listen_sock,&rfds);
	tv.tv_sec=0;
	tv.tv_usec=0;
	if (select(listen_sock+1,&rfds,NULL,NULL,&tv)>0) {
	  handle_batteryport();
	}
      }
      len+=l;
    }
    if (pos>32) {
      memmove(buf,buf+pos,len-pos);
      len-=pos;
      pos=0;
      continue;
    }
    /* synchronize with the input data */
    if (CHKI(buf[pos])&&CHKI(buf[pos+2])&&CHKI(buf[pos+4])&&CHKI(buf[pos+6])) {
      /* successfully found one type/value pair */
      switch(buf[pos]) {
        case 0x41: *x=(buf[pos+1]&0xff)-0x100; foundmask|=1;pcount++;if (beverbose) {printf("x"); fflush(stdout);} break;
	case 0x42: *x=(buf[pos+1]&0xff);       foundmask|=1;pcount++;if (beverbose) {printf("X"); fflush(stdout);} break;
	case 0x43: *y=(buf[pos+1]&0xff)-0x100; foundmask|=2;pcount++;if (beverbose) {printf("y"); fflush(stdout);}break;
	case 0x44: *y=(buf[pos+1]&0xff);       foundmask|=2;pcount++;if (beverbose) {printf("Y"); fflush(stdout);} break;
	case 0x45: *button=buf[pos+1]&0xff;    foundmask|=4;pcount++;if (beverbose) {printf("b"); fflush(stdout);} break; 
	case 0x46: handle_ser_input(buf[pos+1]);                   if (beverbose) {printf("r"); fflush(stdout);} break;
        case 0x47: /* This is the aux input. Not handled */        
	  break;
	case 0x48: *z=(buf[pos+1]&0xff)-0x100; foundmask|=8;pcount++;if (beverbose) {printf("z"); fflush(stdout);} break;
	case 0x49: *z=(buf[pos+1]&0xff);       foundmask|=8;pcount++;if (beverbose) {printf("Z"); fflush(stdout);} break;
      }
      pos+=2;
    } else {
      if (beverbose) {printf("."); fflush(stdout);}
      /* nothing valid found, try to resynchronize */
      pos+=1;
    }
    if (pcount>4) {
      fprintf(stderr,"Z-Axis enable failed: Still missing data after more than 4 packets\n");
	if (use_syslog) {
	  syslog(LOG_ERR,"Z-Axis enable failed: Still missing data after more than 4 packets\n");
	}
      close(destfd);
      destfd=-1;
      continue;
    }
  }

      if (beverbose) {printf("\n"); fflush(stdout);}

   
}


static void send_button_uinput(int ufd,int key, int state)
{
	
#ifndef DISABLE_UDEV_IFACE
  struct input_event ev;
  switch(key) {
    case 1: ev.code=BTN_LEFT;      break;
    case 2: ev.code=BTN_MIDDLE;    break;
    case 3: ev.code=BTN_RIGHT;     break;
  }     
  ev.value=state;
  ev.type=EV_KEY;
  write(ufd,&ev,sizeof(ev));
#endif
}

static void handle_button_values(Display *dpy, int ufd, int oldbutton, int
newbutton)
{
  int xorval=oldbutton^newbutton;
  if (xorval && use_syslog) {
    syslog(LOG_INFO,"button state changed");
  }
  if (xorval&1) {
    if (!use_x11) {
      send_button_uinput(ufd,1,newbutton&1);
    } else {
      XTestFakeButtonEvent(dpy,1,newbutton&1,0);
    }
  }
  if (xorval&2) {
    if (!use_x11) {
      send_button_uinput(ufd,2,newbutton&2);
    } else {
      XTestFakeButtonEvent(dpy,2,newbutton&2,0);
    }
  }
  if (xorval&4) {
    if (!use_x11) {
      send_button_uinput(ufd,3,newbutton&4);
    } else {
      XTestFakeButtonEvent(dpy,3,newbutton&4,0); 
    }
  }
}

static int init_udev(int use_abs)
{
#ifdef DISABLE_UDEV_IFACE
  return -1;
#else
  struct uinput_user_dev user_dev;
  int ufd=open("/dev/uinput",O_RDWR);
  if (ufd<0)
    return -1;
  memset(&user_dev,0,sizeof(user_dev));
  user_dev.id.bustype=BUS_HOST;
  user_dev.id.vendor=1;
  user_dev.id.product=1;
  user_dev.id.version=1;
  user_dev.absmax[ABS_X]=width;
  user_dev.absmin[ABS_X]=0;
//  user_dev.absfuzz[ABS_X]=2;
//  user_dev.absfuzz[ABS_Y]=2;
  user_dev.absmax[ABS_Y]=height;
  user_dev.absmin[ABS_Y]=0;
  strcpy(user_dev.name,"Scipio");
  write(ufd,&user_dev,sizeof(user_dev));
  if (use_abs) {
    ioctl(ufd,UI_SET_ABSBIT,ABS_X);
    ioctl(ufd,UI_SET_ABSBIT,ABS_Y);
    ioctl(ufd,UI_SET_EVBIT,EV_ABS);
  } else {
    ioctl(ufd,UI_SET_RELBIT,REL_X);
    ioctl(ufd,UI_SET_RELBIT,REL_Y);
    ioctl(ufd,UI_SET_EVBIT,EV_REL);
  }
  ioctl(ufd,UI_SET_EVBIT,EV_KEY);
  ioctl(ufd,UI_SET_KEYBIT,BTN_TOUCH);
  ioctl(ufd,UI_SET_KEYBIT,BTN_LEFT);
  ioctl(ufd,UI_SET_KEYBIT,BTN_RIGHT);
  ioctl(ufd,UI_SET_KEYBIT,BTN_MIDDLE);
  ioctl(ufd,UI_DEV_CREATE,0);

  return ufd;
#endif
}


double xsm,ysm,zsm; // Smoothed values
double xxsm,yxsm,zxsm; // Extra Smoothed Values
int process_init=0;





void request_process_init()
{
  process_init=0;
}

void process_data(int xraw,int yraw,int zraw)
{

  if (!process_init) 
    {
      xsm=xraw;
      ysm=yraw;
      zsm=zraw;

      xxsm=xraw;
      yxsm=yraw;
      zxsm=zraw;

      process_init=1;
    }

    xsm = ((double)xraw)*smooth + xsm*(1.0-smooth);
    ysm = ((double)yraw)*smooth + ysm*(1.0-smooth);
    zsm = ((double)zraw)*smooth + zsm*(1.0-smooth);

    xxsm = ((double)xraw)*xsmooth + xxsm*(1.0-xsmooth);
    yxsm = ((double)yraw)*xsmooth + yxsm*(1.0-xsmooth);
    zxsm = ((double)zraw)*xsmooth + zxsm*(1.0-xsmooth);



}

int detect_interaction()
{

  if (lefthand) {
   if ((abs(xxsm- (GXLSET))<GDELTA) &&
       (abs(yxsm- (GYLSET))<GDELTA) &&
       (abs(zxsm- (GZLSET))<GDELTA)) {
     return 1;
   }

  } else {


   if ((abs(xxsm- (GXSET))<GDELTA) &&
       (abs(yxsm- (GYSET))<GDELTA) &&
       (abs(zxsm- (GZSET))<GDELTA)) {
     return 1;
   }
  } 
  return 0;
 
}

int detect_voice()
{

   if ((abs(zxsm- (VZSET))<GDELTA)) {
     return 1;
   }
  return 0;
 
}

int detect_gesture()
{

  if (lefthand) {
  if (ysm-yxsm<-GDELTA2){ return (2);} // down
  if (ysm-yxsm>GDELTA2) { return (1);} // up
  if (xsm-xxsm>GDELTA2) { return (4);} // right
  if (xsm-xxsm<-GDELTA2){ return (3);} // left

  } else {
  if (ysm-yxsm<-GDELTA2){ return (1);} // up
  if (ysm-yxsm>GDELTA2) { return (2);} // down
  if (xsm-xxsm>GDELTA2) { return (3);} // left
  if (xsm-xxsm<-GDELTA2){ return (4);} // right
  
  }

return (0);

}


void gesture_found(int gesture)
{
  switch (gesture)
    {
    case 1: 
      printf("\n UP \n");
      XTestFakeKeyEvent(dpy,33 ,True,0);// "p"
      usleep(1000);            
      XTestFakeKeyEvent(dpy,33 ,False,0);
      break;
    case 2: 
      printf("\n DOWN \n");
      XTestFakeKeyEvent(dpy,57 ,True,0); // "n"
      usleep(1000);      
      XTestFakeKeyEvent(dpy,57 ,False,0);
      break;
    case 3: 
      printf("\n LEFT \n");
      XTestFakeKeyEvent(dpy,32 ,True,0);// "o"
      usleep(1000);            
      XTestFakeKeyEvent(dpy,32 ,False,0);
      break;
    case 4: 
      printf("\n RIGHT \n");
      XTestFakeKeyEvent(dpy,32 ,True,0);// "o"
      usleep(1000);            
      XTestFakeKeyEvent(dpy,32 ,False,0);
      break;
    default:
      printf("\n STRANGE \n");
    }

}


int voice_control(int voice)
{

  int sock;
  char buf[256];
  struct sockaddr_un addr;
  sock=socket(AF_UNIX,SOCK_STREAM,0);
  addr.sun_family=AF_UNIX;
  snprintf(addr.sun_path,108,"/tmp/gspeakfreely-%s",getenv("USER"));
  if (connect(sock,(struct sockaddr *)&addr,sizeof(addr))) {
    fprintf(stderr,"no gspeakfreely is running\n");
    return 1;
  }

  buf[0]=0;

  strcat(buf," \"btin\" \"");
  if (voice) {
    strcat(buf,"activate");
  } else {
    strcat(buf,"deactivate");
  }
  strcat(buf,"\"");

  write(sock,buf,strlen(buf));
  fprintf(stderr,"sending: %s\n",buf);
  close(sock);

  return(0);
}

static int gesture_kenn(int xraw, int yraw, int zraw, int gesture)
{
  int oldg;
  int interact;
  int intsince;
  int gsince;
  int dtcount;
  int gendcount;
  int voice;
  int vsince=0;
  int novsince=0;
  int vmode=0;

      process_data(xraw,yraw,zraw);

      if (voicectl) { /* voice mode active */

	voice=detect_voice();
	if (voice) {
	  vsince++;
	  novsince=0;
	} else {
	  vsince=0;
	  novsince++;
	}
	
	
	if (vmode) {
	  // voice mode is currently on
	  if (novsince>50) { // check for off
	    printf("\nexiting voice mode\n");
	    vmode=0;
	    voice_control(0);
	  }
	} else {
	  // voice mode is currently off
	  if (vsince>50) { // check for on
	    printf("\nentering voice mode\n");
	    vmode=1;
	    voice_control(1);
	  }
	}
      }

      interact=detect_interaction();
      if (interact) {
	intsince++;
	if (intsince==1) printf("\nInteraction Mode\n");
      } else {
	intsince=0;
      }
      switch (gstate) {
      case idle:
	if (intsince>GDELTATSTART) { // gesture state delay
	  gesture=detect_gesture();
	  if (gesture) {
	    gstate=maybe;
	    oldg=gesture;
	    gsince=0;
	  }
	}

	break;

      case maybe:
	gesture=detect_gesture();
	if (gesture!=oldg) {
	  gstate=idle;
	} else {
	if (++gsince > GDELTAT) { // delta_t
	  gesture_found(gesture);
	  gstate=deadtime;
	  dtcount=0;
	  gendcount=0;
	}
	}

	break;

      case deadtime:
	if(++dtcount >GDELTATDEAD) { // delta_tdead
	  gstate=idle;
	}
	gesture=detect_gesture();
	if (gesture==0) {
	  gendcount++;
	} else {
	  gendcount=0;
	}
	if(gendcount>10) {
	  //gstate=idle;
	}

	break;
      }

      return gesture;
}

static int doLowPass(int number, int value)
{
    int i;
    for(i = 0; i < 2; i++)
    {
	double err = value + c[number][0][i] * d[number][0][i] + c[number][1][i] * d[number][1][i];
	double y = c[number][4][i] * (err + c[number][2][i] * d[number][0][i] + c[number][3][i] * d[number][1][i]);

	
	d[number][1][i] = d[number][0][i];
	d[number][0][i] = err;
	
	value = y;
    }

    return value;
}

static double divWinkel(double a, double b)
{
    a = abs(a - b);
    b = abs(a - 2 * M_PI);

    if(a < b)
	return a;
    else
	return b;
}

static double anglesHelp(double horizontal, double localG, int negative)
{
    double zLocal;

    if(negative)
	zLocal = -sqrt(localG * localG - horizontal * horizontal);
    else
	zLocal = sqrt(localG * localG - horizontal * horizontal);

    return atan2(horizontal, zLocal);
}

static double pitchOld = 1200;
static double rollOld = 1200;

static void calcAngles(int gx, int gy, int gz, double *pitch, double *roll)
{
    double mG = gx * gx + gy * gy + gz * gz;
    double gG = sqrt(mG);

    *pitch = anglesHelp(gy, gG, gz < 0);
    *roll = anglesHelp(gx, gG, gz < 0);

    if(*pitch < 0)
	*pitch += 2 * M_PI;
    if(*roll < 0)
	*roll += 2 * M_PI;

    if(pitchOld > 1000)
    {
	pitchOld--;
	rollOld--;
	return;
    }
    else if(pitchOld == 1000)
    {
	pitchOld = *pitch;
	rollOld = *roll;
	return;
    }

    if(divWinkel(pitchOld, *pitch) > M_PI_2)
	*pitch = anglesHelp(gy, gG, !(gz < 0));

    if(divWinkel(rollOld, *roll) > M_PI_2)
	*roll = anglesHelp(gx, gG, !(gz < 0));

    if(*pitch < 0)
	*pitch += 2 * M_PI;
    if(*roll < 0)
	*roll += 2 * M_PI;

    pitchOld = *pitch;
    rollOld = *roll;
}

#define NULLROLL 0.5
#define NULLPITCH 0.5

static int erkenneNullLage(double pitch, double roll)
{
    if((roll > M_PI_2 && roll < (M_PI + M_PI_2)) || 
       (pitch > M_PI_2 && pitch < (M_PI + M_PI_2)))
	return False;

    if(abs(sin(pitch)) < NULLPITCH && abs(roll) < NULLROLL)
	return True;

    return False;
}

#define MINROLL 0.4
#define MINPITCH 0.15

static double startPitch, startRoll;

static int erkenneGeste(double pitch, double roll)
{
    if(sin(startRoll) + MINROLL < sin(roll))
	return 5; // ROTLINKS
    else if(sin(startRoll) - MINROLL > sin(roll))
	return 6; // ROTRECHTS
    else if(sin(startPitch) + MINPITCH < sin(pitch))
	return 2; // UNTEN
    else if(sin(startPitch) - MINPITCH > sin(pitch))
	return 1; // OBEN

    return -1; // STRANGE
}

#define NULL_X -103
#define NULL_Y -71
#define NULL_Z -108
#define NULLLAGE 20
#define GESTESEIT 10
#define DEADTIME 80

static int nullTimer = 0;
static int deadTimer, moveTimer;
static int alteGeste = -1; // STRANGE

static int gesture_midi(int xraw, int yraw, int zraw, int gesture)
{
    // Achsen Nullen
    xraw -= NULL_X;
    yraw -= NULL_Y;
    zraw -= NULL_Z;

    // Achsen tauschen
    int help = yraw;
    yraw = zraw;
    zraw = help;

    // Filtern
    xraw = doLowPass(0, xraw);
    yraw = doLowPass(1, yraw);
    zraw = doLowPass(2, zraw);

    // Winkel berechnen
    double pitch, roll;
    calcAngles(xraw, yraw, zraw, &pitch, &roll);

    printf("pitch: %3.2f roll: %3.2f gstate %d      \r", pitch * 180 / M_PI, roll * 180 / M_PI, gstate);

    int erkannt;

    switch(gstate)
    {
        case idle:
	    if(erkenneNullLage(pitch, roll))
		nullTimer++;
	    else
		nullTimer = 0;

	    if(nullTimer > NULLLAGE)
	    {
		nullTimer = 0;
		startPitch = pitch;
		startRoll = roll;
		deadTimer = 0;
		moveTimer = 0;
		alteGeste = -1; // STRANGE
		gstate = maybe;
	    }
	    break;

        case maybe:
	    deadTimer++;
	    erkannt = erkenneGeste(pitch, roll);

	    if(erkannt != -1)
	    {
		if(alteGeste == -1)
		    alteGeste = erkannt;
		else if(erkannt == alteGeste)
		{
		    moveTimer++;

		    if(moveTimer >= GESTESEIT)
		    {
			switch(erkannt)
			{
			case 1: printf("\nOBEN\n"); break;
			case 2: printf("\nUNTEN\n"); break;
			case 5: printf("\nROTLINKS\n"); break;
			case 6: printf("\nROTRECHTS\n"); break;
			default: printf("\nSTRANGE\n");
			}
			moveTimer = 0;
			deadTimer = 0;
			gstate = deadtime;
			alteGeste = -1;
		    }
		}
		else if(erkannt != alteGeste)
		{
		    alteGeste = erkannt;
		    moveTimer = 0;
		}
	    }
	    else
	    {
		alteGeste = -1;
		moveTimer = 0;
	    }

	    if(deadTimer > 100)
		gstate = idle;
	    break;

        case deadtime:
	    deadTimer++;

	    if(deadTimer > DEADTIME)
	    {
		deadTimer = 0;
		gstate = idle;
		printf("\nOK\n");
	    }
	    break;
    }

    return gesture;
}

static void initButterworth()
{
    int i,j;

    for(i = 0; i < 3; i++)
    {
	c[i][0][0] = -2 * (OMEGASQ - 1) / W1;
	c[i][1][0] = -(1 - W0 + OMEGASQ) / W1;
	c[i][2][0] = 2;
	c[i][3][0] = 1;
  	c[i][4][0] = OMEGASQ / W1;
	c[i][0][1] = (1 - OMEGA) / (1 + OMEGA);
	c[i][1][1] = 0;
	c[i][2][1] = 1;
	c[i][3][1] = 0;
	c[i][4][1] = OMEGA / (1 + OMEGA);

	for(j = 0; j < 2; j++)
	{
	    d[i][j][0] = 0;
	    d[i][j][1] = 0;
	}
    }
}

void check_keyboard( int analog ) {
  int i;
  FILE* f;

  for( i=0; i<keyb_keycount; i++ ) {
    if( (keyb_analog[i]-keyb_tolerance <= analog)
	&& (keyb_analog[i]+keyb_tolerance >= analog ) ) {
      
      if( i != keyb_lastpressed ) {
	keyb_lastpressed = i;
	keyb_testcounter=0;
      } else {
	keyb_testcounter++;
	if( (keyb_testcounter == keyb_okcounter) && (i!=0) ) {
	  if( beverbose ) 
	    printf( "KEYBOARD: '%c'\n", keyb_charcodes[i] );
	  f = fopen( keyb_filename, "w" );
	  if( f == NULL ) {
	    perror( "Keyboard file error" );
	  } else {
	    fwrite( &(keyb_charcodes[i]), 1, 1, f );
	    fclose( f );
	  }
	}
      }
    }
  }
  
}

int main(int argc, char **argv)
{
  int newbutton;
  int oldbutton;
  int i;
  int timestep;
  int ufd;
  int oldx=-1;
  int oldy=-1;
  int x,y,z;
  int argoff=0;
  destfd=-1;
  int rotation = 0;
  int invx = 0;
  int invy = 0;

static  int xraw,yraw,zraw;

  double xsmooth=0.0;
  double ysmooth=0.0;
  double zsmooth=0.0;

  int gesture;


  // Parse Arguments
  for( argoff=1; argoff<argc; argoff++ ) {
    if( strcmp( argv[argoff], "-abs" ) == 0 ) {
      use_abs = 1;
    } else if( strcmp( argv[argoff], "-v" ) == 0 ) {
      beverbose = 1;
    } else if( strcmp( argv[argoff], "-minx" ) == 0 ) {
      argoff++;
      minx = atoi( argv[argoff] );
    } else if( strcmp( argv[argoff], "-batteryport" ) == 0 ) {
      int x;
      struct sockaddr_in addr;
      argoff++;
      addr.sin_family = AF_INET;
      addr.sin_port = htons(atoi(argv[argoff]));
      addr.sin_addr.s_addr = htonl(INADDR_LOOPBACK);
      listen_sock = socket(AF_INET, SOCK_STREAM, 0);
      x=1;
      setsockopt(listen_sock,SOL_SOCKET,SO_REUSEADDR,&x,sizeof (x));
      if (bind(listen_sock, (struct sockaddr *) &addr, sizeof(addr))) {
        fprintf(stderr,"cannot bind battery port\n");  
	close(listen_sock);
	listen_sock=-1;
      } else {
        listen(listen_sock,2);
      }
      
    } else if( strcmp( argv[argoff], "-maxx" ) == 0 ) {
      argoff++;
      maxx = atoi( argv[argoff] );
    } else if( strcmp( argv[argoff], "-miny" ) == 0 ) {
      argoff++;
      miny = atoi( argv[argoff] );
    } else if( strcmp( argv[argoff], "-maxy" ) == 0 ) {
      argoff++;
      maxy = atoi( argv[argoff] );
    } else if( strcmp( argv[argoff], "-s" ) == 0 ) {
      argoff++;
      exps = atof( argv[argoff] );
    } else if( strcmp( argv[argoff], "-syslog" ) == 0) {
      openlog("xscipio", LOG_PID, LOG_LOCAL1);
      use_syslog=1;
    } else if( strcmp( argv[argoff], "-r" ) == 0 ) {
      rotation = 1;
    } else if( strcmp( argv[argoff], "-invx" ) == 0 ) {
      invx = 1;
    } else if( strcmp( argv[argoff], "-invy" ) == 0 ) {
      invy = 1;
    } else if( strcmp( argv[argoff], "-gesture" ) == 0 ) {
	argoff++;  
	if( strcmp ( argv[argoff], "kenn" ) == 0 )
	    gesturemode = 1;
	else if( strcmp( argv[argoff], "midi" ) == 0)
	    gesturemode = 2;
	else
	    gesturemode = 0;
    } else if( strcmp( argv[argoff], "-record" ) == 0 ) {
      recorder = 1;
    } else if( strcmp( argv[argoff], "-left" ) == 0 ) {
      lefthand = 1;
    } else if( strcmp( argv[argoff], "-voice" ) == 0 ) {
      voicectl = 1;
    } else if( strcmp( argv[argoff], "-z" ) == 0 ) {
      enable_z = 1;
    } else if( strcmp( argv[argoff], "-keyb" ) == 0 ) {
      keyb_filename = strdup( argv[argoff+1] );
      argoff ++;
    } else {
      break;
    } 
  } 

  if (argc-argoff != 4 ) {
    printf(
	    "Usage: %s [-abs] [-v] [-r] [-invx] [-invy] [-syslog] [-minx i] [-maxx i] [-miny i] [-maxy i] [-batteryport port] [-s f] [-gesturemode kenn|midi] [-z] [-left] [-voice] [-keyb keyboard_filename] [ bluetooth-mac | :bluetooth-name ] channel rfidfilename x11|uinp\nor %s comm-device baudrate rfidfilename x11|uinp [ abs ]\nor %s host port rfidfilename x11|uinp [ abs ]\n",
	    argv[0],argv[0],argv[0]);
    printf("x11 means to feed mouse input through x11 and\nuinp means to use /dev/uinput and register as a pointer device to the kernel\n");
    printf("-gesturemode recognizes gestures and emits keypress events instead of mouse clicks and mouse position.\n");
    return 1;
  }

  signal(SIGPIPE,SIG_IGN); 
  rfidfilename=argv[argoff+2];
  use_x11=0;
  if ((!strcasecmp("x11",argv[argoff+3]))&&((dpy=XOpenDisplay(NULL)))) {
    XSynchronize(dpy,1);
    width=WidthOfScreen(DefaultScreenOfDisplay(dpy));
    height=HeightOfScreen(DefaultScreenOfDisplay(dpy));
    use_x11=1;
    if( beverbose )
      printf( "Screen height=%d, width=%d\n", height, width );
  } else {
    ufd=init_udev(use_abs);
    if (ufd<0) {
      fprintf(stderr,"Cannot open /dev/uinput\n");
    }
  }


  if (recorder) {
    printf("recoder mode enabled\n");
  } else if (gesture) {
    printf("gesture mode enabled\n");
    if(lefthand) {
      printf("using lefthand gestures\n" );
    } else {
      printf("using righthand gestures\n" );
    }
    if(voicectl) {
      printf("voice input control enabled\n" );
    }

  }

  if( rotation ) {
    int hminx;
    int hmaxx;
    hminx = minx;
    hmaxx = maxx;
    minx = miny;
    maxx = maxy;
    miny = hminx;
    maxy = hmaxx;
  }

  if(gesturemode == 2)
      initButterworth();

  // Und los
  i=0;
  oldbutton=0;

  if (recorder)
    {
      timestep=0;
      while(1) {
	FILE *f;
	if (enable_z) {
	  get_values_z(argv[argoff+0],atoi(argv[argoff+1]),&xraw,&yraw,&zraw,&newbutton);
	} else {
	  get_values(argv[argoff+0],atoi(argv[argoff+1]),&yraw,&xraw,&newbutton);
	}
	  if( beverbose )
	    printf("Achse x: (%4d) Achse y: (%4d) Achse z: (%4d) Buttons: %c %c %c\n",
		   xraw, yraw, zraw,
		   (newbutton&1)?'1':' ',
		   (newbutton&2)?'2':' ',
		   (newbutton&4)?'3':' ');
      
	f=fopen(rfidfilename,"a"); 
	if (f) {
	  fprintf(f,"%d %d %d %d %d\n",timestep++,xraw,yraw,zraw,newbutton);
	  fclose(f);
	}
      } // while 1
    } // if recorder


  while(1) {
    if (!enable_z) {
      get_values(argv[argoff+0],atoi(argv[argoff+1]),&yraw,&xraw,&newbutton);
    if( rotation ) {
      int help = yraw;
      yraw = xraw;
      xraw = help;
    }

    if( invx ) 
      xraw =- xraw;

    if( invy ) 
      yraw =- yraw;

    /* jrei: exponentielle glaettung der koordinaten */
    xsmooth = ((double)xraw)*exps + xsmooth*(1.0-exps);
    ysmooth = ((double)yraw)*exps + ysmooth*(1.0-exps);

    } else {
      get_values_z(argv[argoff+0],atoi(argv[argoff+1]),&xraw,&yraw,&zraw,&newbutton);

/*       if( rotation ) { */
/*       int help = yraw; */
/*       yraw = xraw; */
/*       xraw = help; */
/*     if( invx )  */
/*       xraw =- xraw; */

/*     if( invy )  */
/*       yraw =- yraw; */


/*        xsmooth = ((double)xraw)*exps + xsmooth*(1.0-exps); */
/*        ysmooth = ((double)yraw)*exps + ysmooth*(1.0-exps); */
    xsmooth = ((double)xraw)*exps + xsmooth*(1.0-exps);
    ysmooth = ((double)yraw)*exps + ysmooth*(1.0-exps);
    zsmooth = ((double)zraw)*exps + zsmooth*(1.0-exps);

    }

    // Keyboard Mode
    if( keyb_filename != NULL ) {
      // The keyboard is attached to the x-axis.
      check_keyboard( xraw );
    }

    if (gesturemode) 
    {
	if (!enable_z)
	{
	    fprintf(stderr,"Gesture mode requires z-axis. Enable with -z if device has one.\n");
	    exit(-1);
	}

	switch(gesturemode)
	{
	    // gesture_kenn | gesture_midi
	    case 1: gesture = gesture_kenn(xraw, yraw, zraw, gesture); break;
	    case 2: gesture = gesture_midi(xraw, yraw, zraw, gesture); break;
	}
    }
    else
      {

	i++;
	if (i==20) {

	  /* Skalierung des Eingabebereiches (minx, maxx, miny, maxy)
	     auf die Bildschirmaufloesung (width, height) */
	  x = (int)( ((double)(xsmooth-minx)) * ((double)width) / ((double)(maxx-minx)) );
	  y = (int)( ((double)(ysmooth-miny)) * ((double)height) / ((double)(maxy-miny)) );
	  z = (int)( ((double)(zsmooth-miny)) * ((double)height) / ((double)(maxy-miny)) );
      
	  if( beverbose )
	    printf("Achse 1: %4d (%4d) Achse 2: %4d (%4d) Achse 3: %4d (%4d) Buttons: %c %c %c\n",
		   x, xraw, y, yraw,z,zraw,
		   (newbutton&1)?'1':' ',
		   (newbutton&2)?'2':' ',
		   (newbutton&4)?'3':' ');

      
	  if (use_x11) {
	    XTestFakeMotionEvent(dpy,0,x,y,0);

	  } else if (oldx>=0) {
#ifndef DISABLE_UDEV_IFACE
	    struct input_event ev;
	    if (use_abs) {
	      ev.type=EV_ABS;
	      ev.code=ABS_X;
	      ev.value=x;
	      write(ufd,&ev,sizeof(ev));
	      ev.type=EV_ABS;
	      ev.code=ABS_Y;
	      ev.value=y;
	      write(ufd,&ev,sizeof(ev));
	    } else {
	      ev.type=EV_REL;
	      ev.code=REL_X;
	      ev.value=x-oldx;
	      write(ufd,&ev,sizeof(ev));
	      ev.code=REL_Y;
	      ev.value=y-oldy;
	      write(ufd,&ev,sizeof(ev));
	    }
#endif
	  }
	  oldx=x;
	  oldy=y;

	  if (oldbutton!=newbutton) {
	    handle_button_values(dpy,ufd,oldbutton,newbutton);
	  }
	  if (!use_x11) {
#ifndef DISABLE_UDEV_IFACE
	    struct input_event ev;
	    ev.type=EV_SYN;
	    ev.code=SYN_REPORT;
	    ev.value=0;
	    write(ufd,&ev,sizeof(ev));
#endif
	  }
	  oldbutton=newbutton;
	} 
	i=0;
      }// if (gesture) else
    
  }// while(1)
  
  return 0;
} // main()
