package tzi.UserInput.Serial;

import javax.comm.*;
import java.io.*;
import java.util.*;

/**
 *    The SerialCommunication class provides access to a downlevel hardwarelayer
 *    based on serial communication. This class implements the elementary
 *    methods to communicate with devices connected directly to a COM Port.
 */
public class SerialCommunication implements  Runnable, SerialPortEventListener {

    // Instanzvariablen definieren
    /**
     *      A PipedOutputStream to receive permanent serial data.
     *      This Stream has to be activated with setStreamActivity(boolean)!
     */
    public PipedOutputStream mPOS;
    public final static char CR = 0x0d;
    public final static char LF = 0x0a;


    private DataOutputStream mDOS;
    private boolean mStreamActive;private boolean mPullActive;
    private StringBuffer mCharBuffer;

    private InputStream  mInputStream;
    private OutputStream mOutputStream;
    private SerialPort   mSerialPort;
    private SerialPortConfiguration mPortConfig;


    private Thread       mPortThread;
    private boolean      mWasWakedUp = false;  // Warum geht das, obwohl doch keine Instanz existiert ?


    /**
     *      Creates an object with standard values.
     *      This contructor doesn't open or initialize the serial port.
     */
    public SerialCommunication() {
	// Standardwerte benutzen
	mPortConfig = new SerialPortConfiguration();
	mPullActive = true;
	mStreamActive = false;
    }

    /**
     *      Creates an object with specialized port attributes.
     *      This contructor doesn't open or initialize the serial port.
     */
    public SerialCommunication(String port, int timeout, int bps, int databits, int stopbits, int parity, boolean setStream, boolean setPull) {
	mPortConfig = new SerialPortConfiguration(port, timeout, bps, databits, stopbits, parity);
	mCharBuffer = new StringBuffer(0);
	mPullActive = setPull;
	mStreamActive = setStream;
    }

    /**
     *      After the SerialCommunication object was constructed,
     *      the attributes of the COM port can individually
     *      configured with this method.
     */
    public void configGUI() {
	SerialPortConfigurationGUI configGUI = new SerialPortConfigurationGUI();
	mPortConfig = configGUI.startGUI(mPortConfig);
    }

    /**
     *      This method searchs the choosen serial port and tries to open it.
     *      After the port is found and successfully opened,
     *      the in- and output streams will be created and an own
     *      communication-thread will become alive.
     */
    public void openSerialPort() throws SerialPortErrorException {
	mCharBuffer = new StringBuffer(0);
	mPOS = new PipedOutputStream();
	mDOS = new DataOutputStream(mPOS);
	CommPortIdentifier mPortId = null;
	Enumeration portList = CommPortIdentifier.getPortIdentifiers();
	boolean finished = false;
	while (portList.hasMoreElements() && !finished) {
	    mPortId = (CommPortIdentifier) portList.nextElement();
	    if (mPortId.getPortType() == CommPortIdentifier.PORT_SERIAL) {
		if(mPortId.getName().equals( mPortConfig.getPort() )){
		    finished=true;
		    System.out.println("Found " + mPortConfig.getPort() );
		}
	    }
	}
	try {
	    mSerialPort = (SerialPort) mPortId.open("SerialDataAcquisition", 2000);
	} catch (PortInUseException e) { throw new SerialPortErrorException(); }
	try {
	    mInputStream  = mSerialPort.getInputStream();
	    mOutputStream = mSerialPort.getOutputStream();

	} catch (IOException e) { throw new SerialPortErrorException(); }
	try {
	    mSerialPort.addEventListener(this);
	} catch (TooManyListenersException e) {}

	mSerialPort.notifyOnDataAvailable(true);


	try {
	    mSerialPort.setSerialPortParams(mPortConfig.getBps(), mPortConfig.getDataBits(), mPortConfig.getStopBits(), mPortConfig.getParity() );
	} catch (UnsupportedCommOperationException e) { throw new SerialPortErrorException(); }

	mSerialPort.setDTR(true);


	mPortThread = new Thread(this);
	mPortThread.setName("Serial Port Thread");
	mPortThread.start();

	mSerialPort.setDTR(false);
    }

    /**
     *        Clear the character buffer for pull operations.
     *        This method doesn't clear the output stream!
     */
    public synchronized void resetBuffer() {
	mCharBuffer.setLength(0);
    }

    /**
     *        Starts the IO Thread to receive permanently serial data from the port.
     */
    public void run() {
	while(true){
	    try{
		Thread.sleep(100000);
	    } catch(InterruptedException e) {}
	}
    }

    /**
     *      If a serial data packet becomes available,
     *      this data will have to received.
     */
    public void serialEvent(SerialPortEvent event) {  // warum muß diese methode public sein ???
        switch(event.getEventType()) {
        case SerialPortEvent.BI:
        case SerialPortEvent.OE:
        case SerialPortEvent.FE:
        case SerialPortEvent.PE:
        case SerialPortEvent.CD:
        case SerialPortEvent.CTS:
        case SerialPortEvent.DSR:
        case SerialPortEvent.RI:
        case SerialPortEvent.OUTPUT_BUFFER_EMPTY:
            break;
        case SerialPortEvent.DATA_AVAILABLE:
	    try{
		readSerialData();
	    } catch(SerialPortErrorException e) { e.printStackTrace();}  // Hier geht eine Exception verloren!
            break;
        }
    }

    /**
     *        This method writes a String directly to the configured COM port.
     */
    public void send(String data){
	try{
	    mOutputStream.write(data.getBytes());
	} catch(IOException e) {}
    }


    /**
     *        Serial data from the port will append to the character
     *        buffer for pull operations in dependence of the boolean parameter.
     */
    public synchronized void setPullActivity( boolean b ) { mPullActive = b; }

    /**
     *        Serial data from the port will write into the PipedOutputStream
     *        in dependence of the boolean parameter.
     */ 
    public synchronized void setStreamActivity( boolean b ) { mStreamActive = b; }


    /**
     *        This method realized a possibility to receive a piece of data
     *        from the serial port.
     *        The length of the data fragment has to be configured as parameter.
     */
    public synchronized String readValueCount(int len) throws SerialPortErrorException {
	while (mCharBuffer.length()<len) { waitForData(); }
	String retString = mCharBuffer.substring(0,len);
	if (mCharBuffer.length() > len) { mCharBuffer.delete(0,len); }
	else                            { mCharBuffer.setLength(0); }
	return retString;
    }

    /**
     *        This method realized a possibility to receive a piece of data
     *        from the serial port.
     *        The method needs a parameter to configure a separator character.
     *        The returned substring ends with the last character before the
     *        separator character. The separator character is killed and won't
     *        be returned next time the method will be called.
     */
    public synchronized String readValueUpTo(char sepChar) throws SerialPortErrorException {
	String retString = ""; int i=0;
	while ( (mCharBuffer.length() == i) || (mCharBuffer.charAt(i)!=sepChar) ) {
	    if (mCharBuffer.length() == i) {
		waitForData();
	    }
	    if (mCharBuffer.charAt(i)!=sepChar) {
		retString = retString + (char) mCharBuffer.charAt(i);
		i++;
	    }
	}

	if ( mCharBuffer.length() > i+1 ) {
	    mCharBuffer.delete(0,i+1);
	} else {
	    mCharBuffer.setLength(0);
	}
	return retString;
    }

    public synchronized String readValueUpTo(char sepChar, boolean clear) throws SerialPortErrorException {
	String retVal = readValueUpTo(sepChar);

	try{
	    Thread.sleep(10);
	} catch(InterruptedException ex){
	    ex.printStackTrace();
	}

	if(clear){
	    mCharBuffer.setLength(0);
	    try{
		while(mInputStream.available()>0){
		    mInputStream.read();
		}
	    } catch(IOException ex){
		ex.printStackTrace();
	    }
	}
	return retVal;

    }

    private synchronized void waitForData() throws SerialPortErrorException {
        try{
	    mWasWakedUp = false;
	    //System.out.println("vorm schlafen");
	    wait(mPortConfig.getTimeOut());
	    //System.out.println("nach schlafen");
	    if( !mWasWakedUp ) {
		throw new SerialPortErrorException();
	    }
        } catch(InterruptedException e){
	    System.out.println(e);
        }
    }

    private synchronized void wakeUpCallers(){
	mWasWakedUp = true;
	// System.out.println("vorm wecken");
	notifyAll();
    }

    private synchronized void readSerialData() throws SerialPortErrorException {
	try {
  	    int newData = 0;
	    while ( mInputStream.available()>0 ) {
		newData = mInputStream.read();
		if (mPullActive) {
		    mCharBuffer.append((char)newData);
		}
		if (mStreamActive) {
		    mDOS.writeChar(newData);
		    try {
			mDOS.flush();
			mPOS.flush();
		    } catch(IOException e) {}
		}
	    }

	    if (mPullActive) wakeUpCallers();

	} catch (IOException ex) {
	    System.err.println(ex);
	    throw new SerialPortErrorException();
	}
    }

}
