/**
 * XMLService implements the DataService interface to support data from an XML
 * file.
 *
 * XMLService.java
 *
 * Created on February 15, 2005, 2:08 PM
 */

package tzi.winspect.DataService;

import java.util.Vector;
import java.io.FileNotFoundException;
import org.w3c.dom.*;
import javax.xml.parsers.*;

import tzi.winspect.Debug.Debug;
import tzi.winspect.Main;

import tzi.winspect.Inspector.Inspector;
import tzi.winspect.InvestigateComponent.InvestigateComponentPart;
import tzi.winspect.InvestigateComponent.InvestigateComponent;
import tzi.winspect.InvestigateComponent.SecureViolation;
import tzi.winspect.Investigation.Finding;

/**
 * XML Data Service will provide data from an XML file.
 * @author  jrei
 * @version $Revision: 1.5 $
 */
public class XMLService implements DataService {
    
    public static final String COMPONENT = "component";
    
    public static final String CRANE = "crane";
    
    public static final String CYCLE_TIME = "cycletime";
    
    public static final String DESCRIPTION = "description";
    
    public static final String DETAIL = "detail";
    
    public static final String FIRSTNAME = "firstname";
    
    public static final String ID = "id";
    
    public static final String INSPECTOR = "inspector";
    
    public static final String INSPECTION_STATE = "inspectionstate";
    
    public static final String INVESTIGATE_PART = "investigatepart";
    
    public static final String LOCATION = "location";
    
    public static final String MAINTENANCE_STATES = "maintenance_states";
    
    public static final String MODULE = "module";
    
    public static final String MODULE_ID = "moduleid";
    
    public static final String NAME = "name";
    
    public static final String PICTURE_URL = "pictureurl";
    
    public static final String POSSIBLE_RESULT = "possibleresult";
    
    public static final String STATE = "state";
    
    public static final String SURNAME = "surname";
    
    public static final String TABLE = "table";
    
    public static final String TRANSPONDER_ID = "transponderid";
    
    public static final String VALUE = "value";
    
    private static final String PICTURE_DIR = Main.mRB.getString("Picture_Directory");

    /**
     * The Winspect XML Config as a DOM Document.
     */
    private Document winspectConfig;
    
    /**
     * Last set of inspectors which was loaded from config.
     */
    private Vector inspectors;
    
    /** Creates a new instance of XMLService */
    public XMLService() {
    }
    
    /**
     * Creates a new instance of XMLService
     * @param configUri The URI of the xml config file.
     */
    public XMLService(String configUri) {
        Main.mDebug.print(Debug.DEBUG, "ConfigURI: " + configUri);
        try {
            this.loadConfigFile(configUri);
        }
        catch(DataServiceException dsEx){
            Main.mDebug.print(Debug.DEBUG, dsEx.getMessage());
        }
    }
    
    /**
     * Loads an XML config file of winspect into the DataService.
     * The javax.xml and org.w3c packages are used to parse the XML file.
     * @param configUri The URI of the config file which should be loaded.
     * @throws DataServiceException If an error occures while reading and parsing the config file a
     * DataServiceException will be thrown.
     */
    public void loadConfigFile(String configUri) throws DataServiceException{
        DocumentBuilderFactory documentBuilderFactory =
        DocumentBuilderFactory.newInstance();
        documentBuilderFactory.setValidating(false);
        documentBuilderFactory.setIgnoringComments(true);
        documentBuilderFactory.setNamespaceAware(false);
        try{
            Main.mDebug.print(Debug.DEBUG, "Start parsing ...");
            Main.mDebug.print(Debug.DEBUG, "Found file: " +configUri + " = "
            + new java.io.File(configUri).exists());
            DocumentBuilder documentBuilder = documentBuilderFactory.newDocumentBuilder();
            this.winspectConfig = documentBuilder.parse(configUri);
            if (this.winspectConfig == null){
                Main.mDebug.print(Debug.ERROR, "Config file not found.\n");
            }
            Main.mDebug.print(Debug.DEBUG, "finished parsing");
        }
        catch (ParserConfigurationException pcEx){
            Main.mDebug.print(Debug.DEBUG, pcEx.getMessage());
            throw new DataServiceException(XMLService.class.getPackage().getName(),
            XMLService.class.getName(),
            "loadConfigFile(String configUri) throws DataServiceException ",
            "An Error occured while parsing the document. " +
            "\nPlease make sure that the file exists and is well formed.",
            pcEx.getStackTrace());
        }
        catch (org.xml.sax.SAXException saxEx){
            Main.mDebug.print(Debug.DEBUG, saxEx.getMessage());
            throw new DataServiceException(XMLService.class.getPackage().getName(),
            XMLService.class.getName(),
            "loadConfigFile(String configUri) throws DataServiceException ",
            "An Error occured while parsing the document. " +
            "\nPlease make sure that the file exists and is well formed.",
            saxEx.getStackTrace());
        }
        catch (java.io.IOException ioEx){
            Main.mDebug.print(Debug.DEBUG, ioEx.getMessage());
            throw new DataServiceException(XMLService.class.getPackage().getName(),
            XMLService.class.getName(),
            "loadConfigFile(String configUri) throws DataServiceException ",
            "An Error occured while parsing the document. " +
            "\nPlease make sure that the file exists and is well formed.",
            ioEx.getStackTrace());
        }
    }
    
    /**
     * Gets one of the table elements within the config document.
     * @param tableName The value of the <I>name</I> attribute from the search <I>table</I> node.
     * @throws ObjectNotFound If the table is not found an ObjectNoFound exception will be thrown.
     * @return The found table as an DOM Element.
     */
    public Element getTableByName(String tableName) throws ObjectNotFound{
        NodeList tableList = this.winspectConfig.getElementsByTagName(TABLE);
        /* search table */
        for (int i = 0; i < tableList.getLength();i++){
            if (((Element)tableList.item(i)).getAttribute(NAME).equals(tableName)){
                return (Element)tableList.item(i);
            }
        }
        throw new ObjectNotFound(XMLService.class.getPackage().getName(),
        XMLService.class.getName(),
        "getTableByName(String tableName) throws ObjectNotFound ",
        "An Error occured while parsing the document. " +
        "\nPlease make sure that the searched table exists.");
    }
    
    /**
     * Gets the inspector by the dataset id.
     * @param inID The inspector id.
     * @throws ObjectNotFound If no inspector with that id is found the ObjectNotFound Exception is thrown.
     * @return The inspector object without features.
     */
    public Inspector getInspectorByID(int inID) throws ObjectNotFound {
        Element inspectorTable = this.getTableByName(INSPECTOR);
        NodeList inspectorList = inspectorTable.getElementsByTagName(INSPECTOR);
        /* search table */
        for (int i = 0; i < inspectorList.getLength();i++){
            if (((Element)inspectorList.item(i)).getAttribute(ID).equals(Integer.toString(inID))){
                Inspector inspector = new Inspector();
                inspector.setFirstName(((Element)inspectorList.item(i)).getAttribute(FIRSTNAME));
                inspector.setLastName(((Element)inspectorList.item(i)).getAttribute(SURNAME));
                return inspector;
            }
        }
        throw new ObjectNotFound(XMLService.class.getPackage().getName(),
        XMLService.class.getName(),
        "getTableByName(String tableName) throws ObjectNotFound ",
        "An Error occured while parsing the document. " +
        "\nPlease make sure that the searched table exists.");
    }
    
    /**
     * Gets a Vector of all Inspectors specefied within the XML Config file.
     * @return A Vector with all inspectors.
     */
    public Vector getAllInspectors() {
        Vector inspectorVector = new Vector();
        try {
            Element inspectorTable = this.getTableByName(INSPECTOR);
            NodeList inspectorList = inspectorTable.getElementsByTagName(INSPECTOR);
            /* search table */
            for (int i = 0; i < inspectorList.getLength();i++){
                int id = Integer.parseInt(((Element)inspectorList.item(i)).getAttribute(ID));
                String firstName = ((Element)inspectorList.item(i)).getAttribute(FIRSTNAME);
                String familyName = ((Element)inspectorList.item(i)).getAttribute(SURNAME);
                /* TODO: features are allways on. */
                Inspector inspector = new Inspector(id, firstName, familyName, 1,1,1,1,1);
                inspectorVector.add(inspector);
            }
        }
        catch (ObjectNotFound onfEx){
            Main.mDebug.print(Debug.DEBUG, onfEx.getMessage());
            /* empty vector is returned */
        }
        Main.mDebug.print(Debug.DEBUG, "Getting Inspectors ... done");
        return inspectorVector;
    }
    
    /**
     * Gets the Dataset id of a component from its transponder id.
     * @param transponderID The TransponderID of the dataset which is searched.
     * @throws ObjectNotFound If no dataset is found an ObjectNotFound Exception is thrown.
     * @return The dataset id which belongs to the transponder id.
     */
    public int getTypeOfInfoCode(String transponderID) throws ObjectNotFound {
        Element componentTable = this.getTableByName(COMPONENT);
        NodeList componentList = componentTable.getElementsByTagName(COMPONENT);
        /* search table */
        for (int i = 0; i < componentList.getLength();i++){
            if (((Element)componentList.item(i)).getAttribute(TRANSPONDER_ID).equals(transponderID)){
                return Integer.parseInt(((Element)componentList.item(i)).getAttribute(ID));
            }
        }
        throw new ObjectNotFound(XMLService.class.getPackage().getName(),
        XMLService.class.getName(),
        "getTypeOfInfoCode(String transponderID) throws ObjectNotFound",
        "TransponderID was not found or ID is in wrong format." +
        "\nPlease make sure that the tables are correct.");
    }
    
    /**
     * Gets all InvestigationParts of a specefied Component.
     * @param inID The id of the component for which the investigationparts are searched.
     * @return A Vector with all InvestigationParts for the specefied Component.
     *  An empty Vector is returned if no objecz was found.
     */
    public Vector getAllInvestigatePartsByInvestigateComponentID(int inID) {
        try {
            Element componentTable = this.getTableByName(COMPONENT);
            NodeList componentList = componentTable.getElementsByTagName(COMPONENT);
            for (int i = 0; i < componentList.getLength(); i++){
                if (((Element)componentList.item(i)).getAttribute(ID).equals(Integer.toString(inID))){
                    Element component = ((Element)componentList.item(i));
                    return this.getInvestigatePartOfComponent(component);
                }
            }
        }
        catch (ObjectNotFound onfEx){
            return new Vector();
        }
        return new Vector();
    }
    
    /**
     * Gets all InvestigationParts of a component DOM Element.
     * @param component The DOM Element which contaions the component.
     * @return A Vector of all InvestigationParts.
     */
    private Vector getInvestigatePartOfComponent(Element component){
        Vector investigatePartVector = new Vector();
        NodeList investigatePartList = component.getElementsByTagName(INVESTIGATE_PART);
        for (int k = 0; k < investigatePartList.getLength(); k++){
            InvestigateComponentPart investigateComponentPart = new InvestigateComponentPart();
            try{
                investigateComponentPart.setDBID(
                Integer.parseInt(((Element)investigatePartList.item(k)).getAttribute(ID)));
                investigateComponentPart.setDetail(
                ((Element)investigatePartList.item(k)).getAttribute(DETAIL));
                investigateComponentPart.setInspectionState(
                Integer.parseInt(((Element)investigatePartList.item(k)).getAttribute(INSPECTION_STATE)));
                investigateComponentPart.setInspectionCycleTime(
                Integer.parseInt(((Element)investigatePartList.item(k)).getAttribute(CYCLE_TIME)));
                /* get possible results */
                Vector possibleResults =
                this.getPossibleResultsOfInvestigatePart((Element)investigatePartList.item(k));
                investigateComponentPart.setPossibleResultsID(possibleResults);
                investigatePartVector.add(investigateComponentPart);
            }
            catch (SecureViolation svEx){
                return investigatePartVector;
            }
        }
        return investigatePartVector;
    }
    
    /**
     * Gets all possible results for a InvestigationPart
     * @param investigatePart The InvestigationPart the Results are searched for.
     * @return A Vector of all possible results.
     */
    private Vector getPossibleResultsOfInvestigatePart(Element investigatePart){
        Vector possibleResultVector = new Vector();
        NodeList possibleResultList = investigatePart.getElementsByTagName(POSSIBLE_RESULT);
        for (int i = 0; i < possibleResultList.getLength(); i++){
            possibleResultVector.add(new Integer(((Element)possibleResultList.item(i)).getAttribute(VALUE)));
        }
        return possibleResultVector;
    }
    
    
    /**
     * Gets the InvestigateComponent of a specified treansponder id.
     * @param transponderID The transponder id of the searched dataset.
     * @throws ObjectNotFound If no Component with the specified transponder id is found
     * an ObjectNoFoundException is thrown.
     * @return The InvestigateComponent with the specified transponder id.
     */
    public InvestigateComponent getInvestigateComponentByTransponderID(String transponderID) throws ObjectNotFound {
        Element componentTable = this.getTableByName(COMPONENT);
        NodeList componentList = componentTable.getElementsByTagName(COMPONENT);
        /* search table */
        for (int i = 0; i < componentList.getLength();i++){
            if (((Element)componentList.item(i)).getAttribute(TRANSPONDER_ID).equals(transponderID)){
                InvestigateComponent investigateComponent = new InvestigateComponent();
                try {
                    investigateComponent.setDBID(Integer.parseInt(((Element)componentList.item(i)).getAttribute(ID)));
                    investigateComponent.setComponentID(Integer.parseInt(((Element)componentList.item(i)).getAttribute(ID)));
                    investigateComponent.setModuleID(Integer.parseInt(((Element)componentList.item(i)).getAttribute(MODULE_ID)));
                    investigateComponent.setPictureLocation(PICTURE_DIR + ((Element)componentList.item(i)).getAttribute(PICTURE_URL));
		    investigateComponent.setCrane(((Element)componentList.item(i)).getAttribute(CRANE));
		    investigateComponent.setComponentName(((Element)componentList.item(i)).getAttribute(DESCRIPTION));
		    investigateComponent.setLocation(((Element)componentList.item(i)).getAttribute(LOCATION));
		    Main.mDebug.print(Debug.DEBUG, PICTURE_DIR + ((Element)componentList.item(i)).getAttribute(PICTURE_URL));
                    investigateComponent.setTransponderID(((Element)componentList.item(i)).getAttribute(TRANSPONDER_ID));
                }
                catch (SecureViolation svEx){
                    throw new ObjectNotFound(XMLService.class.getPackage().getName(),
                    XMLService.class.getName(),
                    "getInvestigateComponentByTransponderID(String transponderID) " +
                    "throws ObjectNotFound",
                    "A problem occured while getting the data of the investigateComponent.");
                }
                return investigateComponent;
            }
        }
        throw new ObjectNotFound(XMLService.class.getPackage().getName(),
        XMLService.class.getName(),
        "getInvestigateComponentByTransponderID(String transponderID) throws ObjectNotFound",
        "TransponderID was not found or ID is in wrong format." +
        "\nPlease make sure that the tables are correct.");
    }
    
    /**
     * Displays the InvestigateComponent.
     * Is required by the DataService interface, since some DataSources might be very slow.
     * The XMLService only wraps this function.
     * @param transponderID The transponder id of the component which is searched.
     * @param dataObserver A data observer the components will be passed to.
     */
    public void getInvestigateComponentByTransponderIDThread(String transponderID, DataObserver dataObserver) {
        try {
            dataObserver.returnInvestigateComponent(
            this.getInvestigateComponentByTransponderID(transponderID));
        }
        catch (ObjectNotFound onfEx){
            Main.mDebug.print(Debug.DEBUG, onfEx.getMessage());
        }
    }
    
    /**
     * Gets the module name by the specified id.
     * @param inID The id of the module that should be returned.
     * @throws ObjectNotFound If no module is found an ObjectNotFound exception is thrown.
     * @return The module name.
     */
    public String getModuleNameByID(int inID) throws ObjectNotFound {
        Element moduleTable = this.getTableByName(MODULE);
        NodeList moduleList = moduleTable.getElementsByTagName(MODULE);
        for (int i = 0; i < moduleList.getLength(); i++){
            if (((Element)moduleList.item(i)).getAttribute(ID).equals(Integer.toString(inID))){
                return ((Element)moduleList.item(i)).getAttribute(VALUE);
            }
        }
        throw new ObjectNotFound(XMLService.class.getPackage().getName(),
        XMLService.class.getName(),
        "getModuleNameByID(int inID) throws ObjectNotFound",
        "Module id was not found or id is in wrong format." +
        "\nPlease make sure that the tables are correct.");
    }
    
    /**
     * Gets the component description to a specified id.
     * @param inID The id of the searched component.
     * @throws ObjectNotFound If no component with the specefied ID is found a
     * ObjectNotFound exception is thrown.
     * @return The description of the found component.
     */
    public String getComponentNameByID(int inID) throws ObjectNotFound {
        Element componentTable = this.getTableByName(COMPONENT);
        NodeList componentList = componentTable.getElementsByTagName(COMPONENT);
        for (int i = 0; i < componentList.getLength(); i++){
	    
            if (((Element)componentList.item(i)).getAttribute(ID).equals(Integer.toString(inID))){
		return ((Element)componentList.item(i)).getAttribute(DESCRIPTION);
            }
        }
        throw new ObjectNotFound(XMLService.class.getPackage().getName(),
        XMLService.class.getName(),
        "getComponentNameByID(int inID) throws ObjectNotFound",
        "Component id was not found or id is in wrong format." +
        "\nPlease make sure that the tables are correct.");
    }
    
    /** Not used.
     * TODO: implement
     */
    public void updateInspectionState(InvestigateComponentPart inPart, int State) {
        Main.mDebug.print(Debug.DEBUG, "TODO: Implement - updateInspectionState");
        // TODO: implement
    }
    
    /**
     *
     */
    public void addFinding(InvestigateComponentPart inPart, Finding inFinding) {
        Main.mDebug.print(Debug.DEBUG, "TODO: Implement - addFinding");
        // TODO: implement
    }
    
    public String getUniqueURLString(int Type) {
        Main.mDebug.print(Debug.DEBUG, "TODO: Implement - getUniqueURLString");
        // TODO: implement
        return new String();
    }
    
    public void addMediaFinding(int Type, String URL, InvestigateComponent inComp, InvestigateComponentPart inPart, float Value1, float Value2, float Value3) {
        Main.mDebug.print(Debug.DEBUG, "TODO: Implement - addMediaFinding");
        // TODO: implement
    }
    
    public void addInspector(String FirstName, String LastName, int Feature1, int Feature2, int Feature3, int Feature4, int Feature5) {
        Main.mDebug.print(Debug.DEBUG, "TODO: Implement - addInspector");
        // TODO: implement
    }
    
    public void addComponentPart(int InCoID, int InPaID, String Detail, String PossibleResults, int InspectionState, int InspectionCycleTime, int Feature1, int Feature2, int Feature3, int Feature4, int Feature5) {
        Main.mDebug.print(Debug.DEBUG, "TODO: Implement - addComponentPart");
        // TODO: implement
    }
    
    public String getInspectionStateByID(int inID) throws ObjectNotFound {
        Main.mDebug.print(Debug.DEBUG, "TODO: Implement - getInspectionStateByID");
        // TODO: implement
        return new String();
    }
    
    
    /**
     * Gets the maintenance description state for a specific ID.
     * @param inID The state id.
     * @throws ObjectNotFound If no state is found an ObjectNotFound exception is thrown.
     * @return The finding object with the maintenance description state.
     */
    public Finding getMaintainDescriptionByID(int inID) throws ObjectNotFound {
        Element maintenanceDescriptionTable = this.getTableByName(MAINTENANCE_STATES);
        NodeList stateList = maintenanceDescriptionTable.getElementsByTagName(STATE);
        Main.mDebug.print(Debug.DEBUG, "getInspectionStateByID - find:" + inID);
        for (int i = 0; i < stateList.getLength(); i++){
            if (((Element)stateList.item(i)).getAttribute(ID).equals(Integer.toString(inID))){
                Finding finding = new Finding();
                finding.setDBID(inID);
                finding.setDescription(((Element)stateList.item(i)).getAttribute(VALUE));
                return finding;
            }
        }
//        // jrei TODO: Workarroung
        Main.mDebug.print(Debug.DEBUG, "getInspectionStateByID - No Element Found.");
        return new Finding();
//        throw new ObjectNotFound(XMLService.class.getPackage().getName(),
//            XMLService.class.getName(),
//            "getMaintainDescriptionByID(int inID) throws ObjectNotFound",
//            "Maintenance state id was not found or id is in wrong format." +
//            "\nPlease make sure that the tables are correct.");
    }
    
}

/*
 * $Log: XMLService.java,v $
 * Revision 1.5  2005/03/04 18:23:03  jrei
 * tested working cebit version
 *
 * Revision 1.4  2005/03/03 17:31:02  jrei
 * tested version
 *
 * Revision 1.3  2005/03/03 13:56:43  jrei
 * xml config running
 *
 * Revision 1.2  2005/02/27 20:18:00  jrei
 * restructuring for xml service\nfirst tests
 *
 * Revision 1.1  2005/02/17 15:39:23  jrei
 * not finished version of the data service. More to come, before integration in the working system.
 *
 */
